import each from 'lodash/each';
import { unitTest } from 'tests/utils/module';
import { localStore, memoryStore } from 'tests/fakes/local-store.fake';

unitTest('util | storage', function(hooks) {
    const stores = {
        localStore,
        memoryStore,
    };

    hooks.afterEach(function() {
        Object.keys(stores).forEach(storeName => stores[storeName].clear());
    });

    each(stores, (store, storeName) => {
        QUnit.module(storeName, function() {
            QUnit.module('#get', function() {
                QUnit.test('returns default value when key is not in store', function(assert) {
                    const key = `key_${QUnit.config.current.testId}`;
                    const valueOne = { value: 1 };
                    const valueTwo = { value: 2 };

                    assert.equal(localStore.get(key, valueOne), valueOne);
                    assert.notDeepEqual(localStore.get(key), valueOne, 'should not set a value');
                    assert.equal(localStore.get(key, valueTwo), valueTwo);
                    assert.notDeepEqual(localStore.get(key), valueTwo, 'should not set a value');
                });
            });

            QUnit.module('#set', function() {
                QUnit.test('correctly serializes and deserializes JS objects', function(assert) {
                    const testCases = [
                        'this is a string',
                        ['this', 'is', 'an', 'array'],
                        // eslint-disable-next-line object-property-newline
                        { this: 'is', an: 'object' },
                        [{ this: 'is' }, { an: 'array' }, { of: 'objects' }],
                    ];

                    testCases.forEach((testCase, i) => {
                        const key = `key_${i}`;
                        localStore.set(key, testCase);

                        assert.deepEqual(localStore.get(key), testCase, 'equivalent value');
                    });
                });
            });

            QUnit.module('#has', function() {
                QUnit.test('returns `true` if the storage contains the key', function(assert) {
                    const key = `key_${QUnit.config.current.testId}`;

                    localStore.set(key, 'anything');

                    assert.equal(localStore.has(key), true);
                });

                QUnit.test('returns `false` if the storage does not contain the key', function(assert) {
                    const key = `key_nope_${QUnit.config.current.testId}`;

                    assert.equal(localStore.has(key), false);
                });
            });

            QUnit.module('#remove', function() {
                QUnit.test('can remove keys', function(assert) {
                    const key = `key_${QUnit.config.current.testId}`;
                    const value = { some: 'value' };

                    localStore.set(key, value);

                    assert.deepEqual(localStore.get(key), value);

                    localStore.remove(key);

                    assert.notDeepEqual(localStore.get(key), value);
                });
            });

            QUnit.module('#clear', function() {
                QUnit.test('removes all keys from the store', function(assert) {
                    const keys = [
                        `key_one_${QUnit.config.current.testId}`,
                        `key_two_${QUnit.config.current.testId}`,
                    ];
                    const NOT_FOUND = {};

                    keys.forEach((key, i) => localStore.set(key, i));

                    keys.forEach((key, i) => {
                        assert.deepEqual(localStore.get(key), i);
                    });

                    localStore.clear();

                    keys.forEach(key => {
                        assert.equal(localStore.get(key, NOT_FOUND), NOT_FOUND);
                    });
                });
            });
        });
    });
});
