import sinon from 'sinon';
import { subscribe } from 'util/subscribe';
import { createStore } from 'redux';
import merge from 'lodash/merge';

QUnit.module('util | subscribe', function(hooks) {
    hooks.beforeEach(function() {
        const defaultState = {
            one: 1,
            two: 2,
            three: false,
        };

        this.store = createStore(function(state = defaultState, action) {
            return action.state || state;
        });
        this.update = newState => {
            this.store.dispatch({
                type: 'irrelevant',
                state: newState,
            });
        };
    });

    QUnit.test('does not call fn when props do not change', function(assert) {
        const spy = sinon.spy();
        const newState = {
            one: 'uno',
            two: 'dos',
            three: this.store.getState().three,
        };

        subscribe(this.store, ['three'], spy);
        this.update(newState);
        assert.equal(spy.callCount, 0);
    });

    QUnit.test('calls fn when one of the props changes', function(assert) {
        const spy = sinon.spy();
        const oldState = this.store.getState();
        const newState = {
            one: 1,
            two: 2,
            three: true,
        };

        subscribe(this.store, ['three'], spy);
        this.update(newState);
        assert.equal(spy.callCount, 1);
        assert.deepEqual(spy.getCall(0).args[0], { three: true });
        assert.equal(spy.getCall(0).args[1], oldState);
    });

    QUnit.test('does not call fn when one of the props changes during nested dispatch', function(assert) {
        const nestedDispatch = () => {
            const newState = this.store.getState();
            newState.four = 'cuatro';

            this.store.dispatch({
                type: 'irrelevant',
                state: newState,
            });
        };
        const nestedDispatchSpy = sinon.spy(nestedDispatch);
        const newState = {
            one: 1,
            two: 2,
            three: true,
        };

        subscribe(this.store, ['three'], nestedDispatchSpy);
        this.update(newState);
        assert.equal(nestedDispatchSpy.callCount, 1, 'nestedDispatch should only be called once');
    });

    QUnit.test('returns unsubscribe function', function(assert) {
        const spy = sinon.spy();
        const oldState = this.store.getState();
        const newState = {
            one: 1,
            two: 2,
            three: true,
        };

        const unsubscribe = subscribe(this.store, ['three'], spy);
        this.update(newState);
        assert.equal(spy.callCount, 1);
        unsubscribe();
        this.update(oldState);
        assert.equal(spy.callCount, 1);
    });

    QUnit.module('when subscribing to sub-properties', function() {
        QUnit.test('does not call fn when props do not change', function(assert) {
            const spy = sinon.spy();
            const beginState = {
                one: 'uno',
                two: 'dos',
                three: this.store.getState().three,
                four: {
                    one: 'uno',
                    two: 'dos',
                    three: 'tres',
                },
            };
            const newState = merge({}, beginState, {
                two: 'deux',
            });
            this.update(beginState);
            subscribe(this.store, ['four.one'], spy);
            this.update(newState);
            assert.equal(spy.callCount, 0);
        });

        QUnit.test('does not call fn when other subprops change', function(assert) {
            const spy = sinon.spy();
            const beginState = {
                one: 'uno',
                two: 'dos',
                three: this.store.getState().three,
                four: {
                    one: 'uno',
                    two: 'dos',
                    three: 'tres',
                },
            };
            const newState = merge({}, beginState, {
                four: {
                    two: 'deux',
                },
            });
            this.update(beginState);
            subscribe(this.store, ['four.one'], spy);
            this.update(newState);
            assert.equal(spy.callCount, 0);
        });

        QUnit.test('calls fn when one of the subscribed subprops change', function(assert) {
            const spy = sinon.spy();
            const beginState = {
                one: 'uno',
                two: 'dos',
                three: this.store.getState().three,
                four: {
                    one: 'uno',
                    two: 'dos',
                    three: 'tres',
                },
            };
            const newState = merge({}, beginState, {
                four: {
                    one: 'un',
                },
            });
            newState.four.one = 'un';
            this.update(beginState);
            subscribe(this.store, ['four.one'], spy);
            this.update(newState);
            assert.equal(spy.callCount, 1);
        });

        QUnit.test('does not call fn when unrelated props changes during nested dispatch', function(assert) {
            const nestedDispatch = () => {
                const newState = this.store.getState();
                newState.four.two = 'deux';

                this.store.dispatch({
                    type: 'irrelevant',
                    state: newState,
                });
            };
            const nestedDispatchSpy = sinon.spy(nestedDispatch);
            const beginState = {
                one: 'uno',
                two: 'dos',
                three: this.store.getState().three,
                four: {
                    one: 'uno',
                    two: 'dos',
                    three: 'tres',
                },
            };
            const newState = merge({}, beginState, {
                four: {
                    one: 'un',
                },
            });
            this.update(beginState);
            subscribe(this.store, ['four.one'], nestedDispatchSpy);
            this.update(newState);
            assert.equal(nestedDispatchSpy.callCount, 1, 'nestedDispatch should only be called once');
        });
    });
});
