import { init as initStore } from 'state';
import { unitTest } from 'tests/utils/module';
import { waitFor } from 'tests/utils/waitFor';
import sinon from 'sinon';
import { VodMidrollManager, ARCHIVE } from 'vod-midroll-manager';
import { ACTION_SET_STREAMMETADATA } from 'actions/stream-metadata';
import { ACTION_PLAYING, ACTION_PAUSE } from 'actions/playback';
import { ACTION_SET_AD_METADATA, AdContentTypes } from 'actions/ads';
import { LiveTwitchContentStream } from 'stream/twitch-live';

unitTest('vod-midroll-manager', function(hooks) {
    hooks.beforeEach(function() {
        this.store = initStore();
        sinon.spy(this.store, 'dispatch');
        this.vodMidrollManager = new VodMidrollManager(this.store);
        this.oauthToken = this.api.setLoggedIn(true);
        this.stream = new LiveTwitchContentStream(
            `channel_${QUnit.config.current.testId.toLowerCase()}`,
            Promise.reject(),
            {}
        );
        this.channelInfo = this.api.expectChannelInfo(this.stream.channel);
        this.channelAdProperties = this.api.expectChannelAdProperties(this.channelInfo._id);
    });

    QUnit.test('should start timer when type is archive and is playing', function(assert) {
        this.store.dispatch({
            type: ACTION_SET_AD_METADATA,
            currentMetadata: {
                contentType: AdContentTypes.NONE,
            },
        });
        this.store.dispatch({
            type: ACTION_PLAYING,
        });

        this.store.dispatch({
            type: ACTION_SET_STREAMMETADATA,
            streamMetadata: {
                type: ARCHIVE,
                channel: {
                    name: this.stream.channel,
                },
            },
        });

        return waitFor(() => this.vodMidrollManager.timer !== null).then(() => {
            assert.equal(this.store.getState().playback.playing, true, 'is playing');
            assert.equal(this.store.getState().streamMetadata.type, ARCHIVE, 'is archive');
            assert.notEqual(this.vodMidrollManager._timer, null, 'timer is not null');
        });
    });

    QUnit.test('should destroy timer when type is no longer archive', function(assert) {
        this.store.dispatch({
            type: ACTION_PLAYING,
        });
        this.store.dispatch({
            type: ACTION_SET_STREAMMETADATA,
            streamMetadata: {
                type: ARCHIVE,
                channel: {
                    name: this.stream.channel,
                },
            },
        });

        return waitFor(() => this.vodMidrollManager.timer !== null).then(() => {
            this.vodMidrollManager._timer.pause = sinon.spy();
            this.store.dispatch({
                type: ACTION_SET_STREAMMETADATA,
                streamMetadata: {
                    type: '',
                },
            });

            assert.equal(this.store.getState().playback.playing, true, 'is playing');
            assert.notEqual(this.store.getState().streamMetadata.type, ARCHIVE, 'is not archive');
            assert.equal(this.vodMidrollManager._timer, null, 'timer is null');
        });
    });

    QUnit.test('should pause timer when playback is paused', function(assert) {
        this.store.dispatch({
            type: ACTION_PLAYING,
        });
        this.store.dispatch({
            type: ACTION_SET_STREAMMETADATA,
            streamMetadata: {
                type: ARCHIVE,
                channel: {
                    name: this.stream.channel,
                },
            },
        });

        return waitFor(() => this.vodMidrollManager.timer !== null).then(() => {
            this.vodMidrollManager._timer.pause = sinon.spy();
            this.store.dispatch({
                type: ACTION_PAUSE,
            });

            assert.equal(this.store.getState().playback.playing, false, 'is playing');
            assert.equal(this.store.getState().streamMetadata.type, ARCHIVE, 'is archive');
            assert.notEqual(this.vodMidrollManager._timer, null, 'timer is not null');
            assert.equal(this.vodMidrollManager._timer.pause.callCount, 1, '_timer.pause called');
        });
    });

    QUnit.test('should pause timer when ad is run', function(assert) {
        this.store.dispatch({
            type: ACTION_PLAYING,
        });
        this.store.dispatch({
            type: ACTION_SET_STREAMMETADATA,
            streamMetadata: {
                type: ARCHIVE,
                channel: {
                    name: this.stream.channel,
                },
            },
        });

        return waitFor(() => this.vodMidrollManager.timer !== null).then(() => {
            this.vodMidrollManager._timer.pause = sinon.spy();
            this.store.dispatch({
                type: ACTION_SET_AD_METADATA,
                currentMetadata: {
                    contentType: '',
                },
            });

            assert.equal(this.store.getState().playback.playing, true, 'is playing');
            assert.equal(this.store.getState().streamMetadata.type, ARCHIVE, 'is archive');
            assert.notEqual(this.store.getState().ads.currentMetadata.contentType, AdContentTypes.NONE,
                'is not ad type none');
            assert.notEqual(this.vodMidrollManager._timer, null, 'timer is not null');
            assert.equal(this.vodMidrollManager._timer.pause.callCount, 1, '_timer.pause called');
        });
    });

    QUnit.test('should start timer when ad is over', function(assert) {
        this.store.dispatch({
            type: ACTION_PLAYING,
        });
        this.store.dispatch({
            type: ACTION_SET_STREAMMETADATA,
            streamMetadata: {
                type: ARCHIVE,
                channel: {
                    name: this.stream.channel,
                },
            },
        });

        return waitFor(() => this.vodMidrollManager.timer !== null).then(() => {
            this.store.dispatch({
                type: ACTION_SET_AD_METADATA,
                currentMetadata: {
                    contentType: '',
                },
            });
            this.vodMidrollManager._timer.start = sinon.spy();
            this.store.dispatch({
                type: ACTION_SET_AD_METADATA,
                currentMetadata: {
                    contentType: AdContentTypes.NONE,
                },
            });

            assert.equal(this.store.getState().playback.playing, true, 'is playing');
            assert.equal(this.store.getState().streamMetadata.type, ARCHIVE, 'is archive');
            assert.equal(this.store.getState().ads.currentMetadata.contentType, AdContentTypes.NONE, 'is ad type none');
            assert.notEqual(this.vodMidrollManager._timer, null, 'timer is not null');
            assert.equal(this.vodMidrollManager._timer.start.callCount, 1, '_timer.start called');
        });
    });

    QUnit.test('should start timer when playback is playing from paused', function(assert) {
        this.store.dispatch({
            type: ACTION_PLAYING,
        });
        this.store.dispatch({
            type: ACTION_SET_AD_METADATA,
            currentMetadata: {
                contentType: AdContentTypes.NONE,
            },
        });
        this.store.dispatch({
            type: ACTION_SET_STREAMMETADATA,
            streamMetadata: {
                type: ARCHIVE,
                channel: {
                    name: this.stream.channel,
                },
            },
        });

        return waitFor(() => this.vodMidrollManager.timer !== null).then(() => {
            this.store.dispatch({
                type: ACTION_PAUSE,
            });
            this.vodMidrollManager._timer.start = sinon.spy();
            this.store.dispatch({
                type: ACTION_PLAYING,
            });

            assert.equal(this.store.getState().playback.playing, true, 'is playing');
            assert.equal(this.store.getState().streamMetadata.type, ARCHIVE, 'is archive');
            assert.equal(this.store.getState().ads.currentMetadata.contentType, AdContentTypes.NONE, 'is ad type none');
            assert.notEqual(this.vodMidrollManager._timer, null, 'timer is not null');
            assert.equal(this.vodMidrollManager._timer.start.callCount, 1, '_timer.start called');
        });
    });
});
