import fetchMock from 'fetch-mock';
import { apiHost as API_HOST, clipsAPIHost as CLIPS_HOST } from 'settings';
import assign from 'lodash/assign';
import merge from 'lodash/merge';
import omit from 'lodash/omit';
import pick from 'lodash/pick';
import times from 'lodash/times';
import { parseUri } from 'util/parseuri';
import { TEST_COLLECTION_METADATA, TEST_COLLECTION_ITEMS } from 'tests/fixtures/collection';
import { DEFAULT_LOGGED_IN_USER_V3_RESPONSE } from 'tests/fixtures/user-api';
import { DEFAULT_GET_FOLLOW_SUCCESS, DEFAULT_GET_FOLLOW_FAIL,
         DEFAULT_SET_FOLLOW_SUCCESS } from 'tests/fixtures/follow-api';
import { TEST_EXTENSION_ITEMS, createExtension } from 'tests/fixtures/extensions';
import { ExtensionCoordinator } from 'tests/fakes/extension-coordinator.fake';

/**
 * Configures the fetch mock to respond to viewer token requests (i.e. logged in
 * status).
 *
 * @param {Boolean} loggedIn
 * @return {String} response sent by the server
 */
export function setLoggedIn(loggedIn, tokenResponse) {
    let responseStatus;
    let responseJson;

    if (loggedIn) {
        responseStatus = 200;
        responseJson = tokenResponse || {
            token: `oauth_${QUnit.config.current.testId}`,
        };
    } else {
        responseStatus = 401;
        responseJson = {
            error: 'Not logged in',
            status: 401,
            url: '/api/viewer/token.json',
        };
    }

    fetchMock.mock(`${API_HOST}/api/viewer/token.json`, {
        status: responseStatus,
        headers: {},
        body: responseJson,
    });

    return responseJson;
}

export function expectUserInfo(overrides = {}) {
    const response = assign({}, overrides);

    fetchMock.mock(`${API_HOST}/api/viewer/info.json`, {
        status: 200,
        headers: {},
        body: response,
    });

    return response;
}

/**
 * Configures the fetch mock to respond to channel access token requests.
 *
 * @param {String} channelName
 * @param {?String} oauthToken
 * @param {Object} opts Options for the expected channel access token request
 *        `canAccess`   : if the user has permission to access this channel [ default: true ]
 *        `expires`     : expire timestamp for the returned token [ default: now ]
 *        `accessTokenParams` : extra parameters sent along to the endpoint
 *        `oauthToken`  : OAuth token for the request
 *
 *        a value for the `expires` field of the accessToken
 * @return {Object} response sent
 */
export function expectChannelAccessToken(channelName, opts) {
    const responseJson = {
        token: JSON.stringify({
            id: `token_${QUnit.config.current.testId}`,
            // eslint-disable-next-line camelcase
            chansub: { restricted_bitrates: [] },
            expires: opts.expires || Math.floor(Date.now() / 1000),
        }),
        sig: `sig_${QUnit.config.current.testId}`,
    };

    fetchMock.mock(new RegExp(`^${API_HOST}/api/channels/${channelName}/access_token`), function(rawUrl) {
        const url = parseUri(rawUrl);
        const nauthOk = Object.keys(opts.accessTokenParams).reduce((ok, prop) => {
            return ok && String(opts.accessTokenParams[prop]) === String(url.queryKey[prop]);
        }, true);

        if (!nauthOk) {
            // eslint-disable-next-line max-len
            throw new Error(`Expected Access Token parameters ${JSON.stringify(opts.accessTokenParams)}, got ${JSON.stringify(url.queryKey)}`);
        } else if (!opts.canAccess) {
            return {
                status: 401,
                headers: {},
                body: {},
            };
        } else if (!opts.hasOwnProperty('oauthToken') || url.queryKey.oauth_token === opts.oauthToken) {
            return {
                status: 200,
                headers: {},
                body: responseJson,
            };
        } else {
            return {
                status: 500,
                headers: {},
                body: 'Invalid channel access token request',
            };
        }
    });

    return responseJson;
}

/**
 * Configures the fetch mock to respond to VOD access token requests.
 *
 * @param {String} vodId
 * @param {Object} opts Options for the expected vod access token request
 *        `canAccess`   : if the user has permission to access this vod [ default: true ]
 *        `expires`     : expire timestamp for the returned token [ default: now ]
 *        `accessTokenParams` : extra parameters sent along to the endpoint
 *        `oauthToken`  : OAuth token for the request
 * @return {Object} response sent
 */
export function expectVodAccessToken(vodId, opts) {
    const path = new RegExp(`^${API_HOST}/api/vods/${vodId.substring(1)}/access_token`);
    const responseJson = {
        token: JSON.stringify({
            id: `token_${QUnit.config.current.testId}`,
            // eslint-disable-next-line camelcase
            chansub: { restricted_bitrates: [] },
            expires: opts.expires || Math.floor(Date.now() / 1000),
        }),
        sig: `sig_${QUnit.config.current.testId}`,
    };

    fetchMock.mock(path, function(rawUrl) {
        const url = parseUri(rawUrl);
        const nauthOk = Object.keys(opts.accessTokenParams).reduce((ok, prop) => {
            return ok && String(opts.accessTokenParams[prop]) === String(url.queryKey[prop]);
        }, true);

        if (!nauthOk) {
            // eslint-disable-next-line max-len
            throw new Error(`Expected Access Token parameters ${JSON.stringify(opts.accessTokenParams)}, got ${JSON.stringify(url.queryKey)}`);
        } else if (!opts.canAccess) {
            return {
                status: 401,
                headers: {},
                body: {},
            };
        } else if (!opts.hasOwnProperty('oauthToken') || url.queryKey.oauth_token === opts.oauthToken) {
            return {
                status: 200,
                headers: {},
                body: responseJson,
            };
        } else {
            return {
                status: 500,
                headers: {},
                body: 'Invalid channel access token request',
            };
        }
    });

    return responseJson;
}

/**
 * Configures the fetch mock to expect a Kraken User Info request.
 *
 * @return {Object} response sent
 */
export function expectKrakenUserInfo(overrides = {}) {
    const responseJSON = assign({}, DEFAULT_LOGGED_IN_USER_V3_RESPONSE, overrides);

    fetchMock.mock(`${API_HOST}/kraken/user`, {
        status: 200,
        headers: {},
        body: responseJSON,
    });
    return responseJSON;
}

/**
 * Configures the fetch mock to expect a Stream API request.
 *
 * @param {String} channelName
 * @param {Boolean} online
 * @param {Object} channelInfo
 * @return {Object} response sent
 */
export function expectStreamInfo(channelName, online, channelInfo = {}, responseInfo = {}) {
    const responseStatus = 200;
    const responseJson = {};

    if (online) {
        responseJson.stream = assign({
            /* eslint-disable camelcase */
            viewers: Math.floor(Math.random() * 1000),
            channel: channelInfo,
            preview: {
                template: '',
            },
            stream_type: 'live',
            /* eslint-enable */
        }, responseInfo);
    } else {
        responseJson.stream = null;
    }

    fetchMock.mock(`${API_HOST}/kraken/streams/${channelName}`, {
        status: responseStatus,
        headers: {},
        body: responseJson,
    });

    return responseJson;
}

/**
 * Configures the fetch mock to expect a VOD API request.
 *
 * @param {String} videoId
 * @param {Object} responseValues
 * @return {Object} response sent
 */
export function expectVideoInfo(videoId, responseValues = {}) {
    const API_ONLY_KEYS = [
        'muted_segments',
        'increment_view_count_url',
        'restrictions',
        'seek_previews_url',
    ];
    const response = assign({
        /* eslint-disable camelcase */
        broadcast_id: parseInt(QUnit.config.current.testId, 36),
        broadcast_type: 'archive',
        channel: {
            display_name: `Channel ${QUnit.config.current.testId}`,
            name: `Channel ${QUnit.config.current.testId}`,
            _id: 27446517,
        },
        length: 500 + Math.floor(500 * Math.random()),
        partner: true,
        recorded_at: new Date(2016, Math.floor(Math.random() * 6), Math.floor(Math.random() * 30)).getTime(),
        title: `KappaSpam welcome! ${QUnit.config.current.testId}`,
        /* eslint-enable */
    }, responseValues);

    fetchMock.mock(`${API_HOST}/kraken/videos/${videoId}`, {
        status: 200,
        headers: {},
        body: omit(response, API_ONLY_KEYS),
    });

    fetchMock.mock(`${API_HOST}/api/videos/${videoId}`, {
        status: 200,
        headers: {},
        body: pick(response, API_ONLY_KEYS),
    });

    return response;
}

export function expectChannelAPIInfo(channelName, overrides = {}) {
    const response = mergeOverrides({
        /* eslint-disable camelcase */
        display_name: channelName,
        game: 'Music',
        status: `Super Awesome ${channelName} stream`,
        fight_ad_block: false,
        _id: 27446517,
        name: channelName.toLowerCase(),
        partner: true,
        twitch_liverail_id: 727933,
        liverail_id: 722700,
        comscore_id: null,
        comscore_c6: null,
        // eslint-disable-next-line max-len
        video_banner: `https://static-cdn.jtvnw.net/jtv_user_pictures/${channelName.toLowerCase()}-channel_offline_image-1920x1080.jpeg`,
        steam_id: null,
        ppv: false,
        broadcaster_software: 'unknown_rtmp',
        prerolls: true,
        postrolls: true,
        product_path: `/products/${channelName.toLowerCase()}`,
        /* eslint-enable camelcase */
    }, overrides);

    fetchMock.mock(`${API_HOST}/api/channels/${channelName}`, {
        status: 200,
        headers: {},
        body: response,
    });

    return response;
}

export function expectChannelInfo(channelName, overrides = {}) {
    const response = assign({
        /* eslint-disable camelcase */
        mature: false,
        status: `The Channel ${channelName} Bein' Awesome`,
        broadcaster_language: 'en',
        display_name: channelName.toUpperCase(),
        game: 'Music',
        language: 'en',
        _id: 27446517,
        name: channelName,
        created_at: '2012-01-15T03: 18: 08Z',
        updated_at: '2016-06-19T05: 33: 00Z',
        delay: null,
        logo: `image://static-cdn.jtvnw.net/${channelName}-profile_image-300x300.png`,
        banner: null,
        video_banner: `https://static-cdn.jtvnw.net/${channelName}-channel_offline_image-1920x1080.jpeg`,
        background: null,
        profile_banner: `https://static-cdn.jtvnw.net/${channelName}-profile_banner-480.jpeg`,
        profile_banner_background_color: '#303030',
        partner: true,
        url: `https://www.twitch.tv/${channelName}`,
        views: 11304335,
        followers: 593652,
        /* eslint-enable camelcase */
    }, overrides);

    fetchMock.mock(`${API_HOST}/kraken/channels/${channelName}`, {
        status: 200,
        headers: {},
        body: response,
    });

    return response;
}

export function expectChannelInfoByChannelId(channelId, channelName,  overrides = {}) {
    const response = assign({
        /* eslint-disable camelcase */
        _id: channelId,
        broadcaster_language: 'en',
        created_at: '2013-06-03T19:12:02Z',
        display_name: `${channelName}`,
        followers: '40',
        game: 'Final Fantasy XV',
        language: 'en',
        logo: 'https://static-cdn.jtvnw.net/jtv_user_pictures/dallas-profile_image-1a2c906ee2c35f12-300x300.png',
        mature: true,
        name: `${channelName}`,
        partner: false,
        profile_banner: null,
        profile_banner_background_color: null,
        status: 'The Finalest of Fantasies',
        updated_at: '2016-12-06T22:02:05Z',
        url: `https://www.twitch.tv/${channelName}`,
        video_banner: null,
        views: 232,
        /* eslint-enable camelcase */
    }, overrides);

    fetchMock.mock(`${API_HOST}/kraken/channels/${channelId}`, {
        status: 200,
        headers: {},
        body: response,
    }, { method: 'GET' });

    return response;
}

export function expectCommunitiesInfo(channelId, overrides = {}) {
    const response = assign({
        communities: [{
            _id: '111-222-333-444-555-666',
            name: 'speed-running',
            summary: 'beat games fast',
            description: 'this community is for speed running through games',
            rules: 'no slowpokes',
        }],
    }, overrides);

    fetchMock.mock(`${API_HOST}/kraken/channels/${channelId}/communities`, {
        status: 200,
        headers: {},
        body: response,
    });

    return response;
}

export function expectChannelAdProperties(channelId, overrides = {}) {
    const response = assign({
        /* eslint-disable camelcase */
        channel_id: channelId,
        valid_responses: {
            postrolls_disabled: false,
            prerolls_disabled: false,
            vod_ads_enabled: true,
            vod_archive_midrolls: 'inserted',
            vod_archive_midrolls_break_length: 60,
            vod_archive_midrolls_frequency: 15,
        },
        /* eslint-enable camelcase */
    }, overrides);

    fetchMock.mock(`${API_HOST}/kraken/channels/${channelId}/ads/properties`, {
        status: 200,
        headers: {},
        body: response,
    });

    return response;
}

/**
 * Configures the fetch mock to expect a Channel Viewer Info API request
 *
 * @param {String} channel
 * @param {Object=} overrides
 * @return {Object} the response from the server
 */
export function expectChannelViewerInfo(channel, overrides = {}) {
    const response = assign({
        /* eslint-disable camelcase */
        chansub: null,
        has_ad_free_subscription: null,
        steam_id: null,
        is_admin: false,
        is_editor: false,
        /* eslint-enable camelcase */
    }, overrides);

    fetchMock.mock(`${API_HOST}/api/channels/${channel}/viewer`, {
        status: 200,
        headers: {},
        body: response,
    });

    return response;
}

export function expectFeaturedCollection(channelId, hasCollections) {
    const collections = hasCollections ? [TEST_COLLECTION_METADATA] : [];
    const response = {
        collections,
        _cursor: null,
    };

    fetchMock.mock(`${API_HOST}/v5/channels/${channelId}/collections?limit=1&exclude_empty=true`, {
        status: 200,
        headers: {},
        body: response,
    });
}

/**
 * Configures the fetch mock to expect a follow info request
 *
 * @param {Object} Has channelId, following, notificationsEnabled
 * @return {Object} response sent
 */
export function expectFollowInfo({ channelId, following, notificationsEnabled }) {
    const urlregex = /\/kraken\/users\/\d+\/follows\/channels\/\d+/;
    const success = assign(DEFAULT_GET_FOLLOW_SUCCESS, {
        notifications: notificationsEnabled,
    });

    success.channel = assign(DEFAULT_GET_FOLLOW_SUCCESS.channel, {
        _id: channelId,
    });

    const fail = assign(DEFAULT_GET_FOLLOW_FAIL, {
        message: 'Follow not found',
    });

    const response = following ? success : fail;

    fetchMock.mock(urlregex, () => {
        const status = following ? 200 : 404;
        return {
            status,
            headers: {},
            body: response,
        };
    });

    return response;
}

/**
 * Configures the fetch mock to expect a Follow Update via PUT/DELETE request
 *
 * @param {Boolean} enableFollow
 */
export function expectSetFollow(enableFollow, { channelId, notificationsEnabled }) {
    const urlregex = /\/kraken\/users\/\d+\/follows\/channels\/\d+/;
    const method = enableFollow ? 'PUT' : 'DELETE';
    const success = assign({}, DEFAULT_SET_FOLLOW_SUCCESS, {
        notifications: notificationsEnabled,
    });

    success.channel = assign(DEFAULT_GET_FOLLOW_SUCCESS.channel, {
        _id: channelId,
    });

    fetchMock.mock(urlregex, function() {
        if (enableFollow) {
            return {
                status: 200,
                headers: {},
                body: success,
            };
        }
        return {
            status: 204,
            headers: {},
        };
    }, {
        method,
    });
}

/**
 * Configures fetch mock to expect GET api.twitch.tv/kraken/users/:userId/subscriptions/:channelId
 * @param {Object} subscriptionData
 */
export function expectGetSubscriptionInfo({ userId, channelId, isSubscribed }) {
    fetchMock.mock(`${API_HOST}/kraken/users/${userId}/subscriptions/${channelId}`, () => {
        if (isSubscribed) {
            return {
                status: 200,
                headers: {},
                body: {},
            };
        }
        return {
            status: 404,
            headers: {},
        };
    });
}

/**
 * Configures the fetch mock to expect a Notifications update via PUT request
 *
 * @param {Boolean} enableNotifications
 */
export function expectSetNotifications(enableNotifications, { channelId }) {
    const urlregex = /\/kraken\/users\/\d+\/follows\/channels\/\d+/;
    const success = assign(DEFAULT_SET_FOLLOW_SUCCESS, {
        notifications: enableNotifications,
    });

    success.channel = assign(DEFAULT_GET_FOLLOW_SUCCESS.channel, {
        _id: channelId,
    });

    fetchMock.mock(urlregex, {
        status: 200,
        headers: {},
        body: success,
    }, {
        method: 'PUT',
    });
}

/**
 * Configures the fetch mock to expect a V3 channel video request.
 *
 * @param {String} channelName
 * @param {Array<Object>} recommendedVODs
 */
export function expectV3VODs(channelName, recommendedVODs = []) {
    const urlregex = /\/kraken\/channels\/\w+\/videos\?broadcast_type=archive&limit=\d+/;

    fetchMock.mock(urlregex, rawUrl => {
        const numVids = parseInt(parseUri(rawUrl).queryKey.limit, 10);
        const responseJson = {};
        responseJson._total = recommendedVODs.total;
        responseJson._links = recommendedVODs._links;
        responseJson.videos = recommendedVODs.videos.slice(0, numVids);
        return {
            status: 200,
            headers: {},
            body: responseJson,
        };
    });
}

/**
 * Configures the fetch mock to expect a V5 channel video request.
 *
 * @param {String} channelName
 * @param {Array<Object>} recommendedVODs
 */
// eslint-disable-next-line object-property-newline
export function expectV5VODs(channelID, recommendedVODs = { _total: 0, videos: [] }) {
    const urlRegex = new RegExp(`/kraken/channels/${channelID}/videos\\?limit=\\d+`);

    fetchMock.mock(urlRegex, rawUrl => {
        const numVids = parseInt(parseUri(rawUrl).queryKey.limit, 10) || 10;
        const responseJson = {
            _total: recommendedVODs._total,
            videos: recommendedVODs.videos.slice(0, numVids),
        };

        return {
            status: 200,
            headers: {},
            body: responseJson,
        };
    });
}

/**
 * Configures the fetch mock to expect a resume times request.
 *
 * @param {Number} userID
 * @param {Array<Object>} resumeTimes
 */
export function expectResumeTimes(userID, resumeTimes = []) {
    const resumeTimeURL = `${API_HOST}/api/resumewatching/user?id=${userID}`;
    fetchMock.mock(resumeTimeURL, {
        status: 200,
        headers: {},
        body: {
            videos: resumeTimes,
        },
    });
}

/**
 * Configures the fetch mock to expect a CSRF_TOKEN request
 *
 */

export function expectCSRFToken(response) {
    fetchMock.mock(`${API_HOST}/api/me`, {
        status: 200,
        headers: {},
        body: response,
    });
    return response;
}

/**
 * Merges overrides into a base response, erroring if unexpected keys are present.
 *
 * @param {Object} baseResponse
 * @param {Object} overrides
 * @return {Object}
 */
function mergeOverrides(baseResponse, overrides) {
    const badKeyvals = omit(overrides, Object.keys(baseResponse));

    if (Object.keys(badKeyvals).length > 0) {
        throw new Error(`Unexpected keys in response overrides:\n${JSON.stringify(badKeyvals, null, 2)}`);
    }

    return merge(baseResponse, overrides);
}

/**
 * Configures the fetch mock to expect a collection metadata request.
 *
 * @param {String} the collection id
 * @param {Object} if overrides provided, api responds with base response merged with overrides
 */
export function expectCollectionMetadata(collectionId, overrides = {}) {
    /* eslint-disable camelcase */
    const response = mergeOverrides(TEST_COLLECTION_METADATA, overrides);
    /* eslint-enable camelcase */

    fetchMock.mock(`${API_HOST}/v5/collections/${collectionId}`, {
        status: 200,
        headers: {},
        body: response,
    });
}

/**
 * Configures the fetch mock to expect a collection items request.
 *
 * @param {String} the collection id
 * @param {Array<Object>} if items provided, api responds with provided items (otherwise, one item)
 */
export function expectCollectionItems(collectionId, collectionItems) {
    const items = collectionItems === undefined ? TEST_COLLECTION_ITEMS : collectionItems;
    const response = {
        _id: collectionId,
        items,
    };

    fetchMock.mock(`${API_HOST}/v5/collections/${collectionId}/items`, {
        status: 200,
        headers: {},
        body: response,
    });
}

/**
 * Configures the fetch mock to expect a request to fetch active video overlay extensions.
 *
 * @param {String} channel the channel name to load extensions for
 * @param {Array<Object>} activeExtensions if extensions are provided, api
 *        responds with provided items
 */
export function expectOverlayExtensionsForChannel(channel, activeExtensions) {
    let response;
    /* eslint-disable camelcase */
    if (typeof activeExtensions === 'number') {
        response = times(activeExtensions, createExtension).reduce((response, extension) => ({
            installed_extensions: response.installed_extensions.concat(extension.installed_extensions),
            tokens: response.tokens.concat(extension.tokens).sort(() => Math.random() - 0.5),
        }), {
            installed_extensions: [],
            tokens: [],
        });
    } else {
        const extensions = (activeExtensions || TEST_EXTENSION_ITEMS);
        response = {
            installed_extensions: extensions.map(extension => {
                const id = extension.id;
                const config = {
                    anchor: extension.anchor,
                    slot: `extension-overlay-${id}`,
                };
                const installation_status = {
                    extension_id: id,
                    activation_state: 'active',
                    can_activate: true,
                    activation_config: config,
                };

                return {
                    extension,
                    installation_status,
                };
            }),
            tokens: extensions.map(({ id }) => ({
                extension_id: id,
                token: `${id}_token`,
            })),
        };
    }
    /* eslint-enable camelcase */

    fetchMock.mock(`${API_HOST}/v5/channels/${channel}/extensions`, {
        status: 200,
        headers: {},
        body: response,
    });

    ExtensionCoordinator.ExtensionService.getInstalledExtensions.withArgs(channel).returns(response);
    return response;
}

export function expectLTVStream(response) {
    fetchMock.mock(/kraken\/content.*/, {
        status: 200,
        headers: {},
        body: response,
    });
}

export function expectClipStatus(slug, response) {
    fetchMock.mock(`${CLIPS_HOST}${slug}/status`, {
        status: 200,
        headers: {},
        body: response,
    });
}

export function expectClipInfo(response) {
    fetchMock.mock(/kraken\/clips.*/, {
        status: 200,
        headers: {},
        body: response,
    });
}
export function expectClipView(slug, response) {
    fetchMock.mock(`${CLIPS_HOST}${slug}/view`, {
        status: 200,
        headers: {},
        body: response,
    });
}
export function expectPostExtensionReport() {
    fetchMock.mock('https://www.twitch.tv/user/report', {
        status: 200,
        headers: {},
        body: {},
    }, {
        method: 'POST',
    });
}

export function expectPutResumeWatching(id) {
    fetchMock.mock(`https://api.twitch.tv/api/resumewatching/user-video?id=${id}`, {
        status: 200,
        headers: {},
        body: {},
    }, {
        method: 'PUT',
    });
}

export function mock(...args) {
    return fetchMock.mock(...args);
}

export function clearFakeResponses() {
    fetchMock.restore();
}

export function calls(matcherName) {
    return fetchMock.calls(matcherName);
}

export function called(matcherName) {
    return fetchMock.called(matcherName);
}

export function lastCall(matcherName) {
    return fetchMock.lastCall(matcherName);
}
