jenkins-jobs
============

One of our goals with simplifying builds and deploys at Twitch is to provide reusable components for "common" projects, while allowing innovative teams to define their own process when it doesn't follow our common practices.

The [Jenkins Job DSL plugin](https://github.com/jenkinsci/job-dsl-plugin) allows us to do this by giving each team the ability to control their build and deploy process, opting to inherit existing components for most projects. For an exhaustive DSL reference, check out their [wiki](https://github.com/jenkinsci/job-dsl-plugin/wiki).

Job definitions are stored in a ```jenkins.groovy``` file that lives in the root of your git repository. These will get picked up when the ```job-dsl-seed``` job runs. You may need to kick this job off manually if it isn't running automatically.

When a job script changes on the master branch of this repository our Jenkins server will automatically apply the changes, creating and editing jobs as necessary.

## Build Jobs

Give your job a name, and inherit from a template.  ```TEMPLATE-autobuild``` should be used if possible. ```TEMPLATE-brigade-build``` can be used for build jobs triggered by brigade.

```
job {
	name 'web-jax'
	using 'TEMPLATE-autobuild'
}
```

The following sections give examples of methods which you can use inside your ```job``` block.

### Specifying your repository

This tells jenkins where to get your code.  For more advanced options, refer to their [wiki](https://github.com/jenkinsci/job-dsl-plugin/wiki/Job-reference#git).

Declaring a branch will cause your job to only trigger if changes are detected on that branch. Since brigade assigns particular branch pointers for each server group, you will want to configure your jobs to watch those branches. The branch name can be found as the "internal name" on your server group settings page.

```
scm {
	git {
		remote {
			github 'web/jax', 'ssh', 'git.internal.justin.tv'
			credentials 'git-aws-read-key'
		}
		branches "origin/production"
	}
}
```

### Running manta

Commonly you will only need to run the manta command to perform your build. Assuming you are using the folder ".manta/" as your configured manta output directory, your build steps would be the following.

```
steps {
	shell 'rm -rf .manta/'
	shell 'manta -proxy'
}
```

The ```-proxy``` flag is required on servers behind the HTTP proxy, which includes Jenkins.

### Uploading build output to S3

We've defined a helper DSL method for uploading post-build artifacts. This allows you to easily download them in your deploy job, given a GIT_COMMIT.

```
steps {
	saveDeployArtifact 'web/jax', '.manta'
}
```

### Updating Github commit status

By defining your job as a child of the ```TEMPLATE-base``` or ```TEMPLATE-brigade-build``` config via the [using method](https://github.com/jenkinsci/job-dsl-plugin/wiki/Job-reference#using), your job will automatically update commit statuses as builds start and finish. You can view the source of those templates for details on how to implement this manually in custom jobs.

### (Advanced) Publishing Docker images

In your build steps, you can add calls to ```docker build``` and ```docker push``` to create and save docker images.

```
steps {
	shell 'docker build -t docker-registry.internal.justin.tv/jax .'
	shell 'docker tag docker-registry.internal.justin.tv/jax:\$GIT_COMMIT'
	shell 'docker push docker-registry.internal.justin.tv/jax'
}
```

## Deploy Jobs

Ideally you should only need one job to handle deployments to any environment. Having a deploy job defined per environment is legal, but may cause you maintenance headaches in the future.  (Remember, you and your team own these jobs!)

All deploy jobs should have at least two parameters: GIT_COMMIT and HOSTS. If you inherit from one of our base templates, these will be defined for you. The parameters are passed in from [skadi](https://git.internal.justin.tv/release/skadi/) and translate into environment variables available to all steps. The ```HOSTS``` parameter will be populated from a Consul search and contain the list of servers to deploy to as a comma-separated list.

```
job {
	name 'web-jax-deploy'
	using 'TEMPLATE-deploy'
}
```

For aws deploys use `TEMPLATE-deploy-aws`

### Run courier

Running courier will use the ```deploy.json``` file in your repository to configure your deploy details. Courier knows to read the commonly-set environment variables to load deploy context.

```
steps {
	shell 'courier deploy --repo web/jax'
}
```

### Using capistrano

If you are using capistrano to deploy instead of courier, you will need to prepare the files first.  By using ```saveDeployArtifact``` in your build job, you can use the sibling method ```downloadDeployArtifact``` to fetch those files into the current directory.

Ruby's Net::SSH library is unfortunately incompatible with the ssh-agent created by Jenkins. Inheriting from the ```TEMPLATE-deploy-capistrano``` template instead of ```TEMPLATE-deploy``` will setup a custom ssh-agent process which is compatible. Then for your build steps, you can call your capistrano tasks.

```
steps {
	downloadDeployArtifact 'web/jax'
	shell 'cap deploy:update'
	shell 'cap deploy:restart'
}
```
