var path = require('path');
var fs = require('fs');
var webpack = require('webpack');

var HtmlWebpackPlugin = require('html-webpack-plugin');
var ExtractTextPlugin = require('extract-text-webpack-plugin');
var CopyWebpackPlugin = require('copy-webpack-plugin');

var SRC_PATH = path.resolve(__dirname, 'src');
var DEST_PATH = path.resolve(__dirname, 'build');
var DEV_MODE = (process.env.NODE_ENV !== 'production');
var IS_ROOT = !!(process.getuid && process.getuid() === 0);

module.exports = function(grunt) {
    // `default` task: clean the output directory and build all assets
    grunt.registerTask('default', ['clean', 'build']);
    // `build` task: build all assets
    grunt.registerTask('build', ['webpack:player']);
    // `serve` task: serve assets from the build directory
    grunt.registerTask('serve', ['browserSync']);
    // `dev` task: build assets, then serve them
    grunt.registerTask('dev', (IS_ROOT ? ['serve'] : ['clean', 'build', 'serve', 'watch']));

    grunt.loadNpmTasks('grunt-webpack');
    grunt.loadNpmTasks('grunt-browser-sync');
    grunt.loadNpmTasks('grunt-contrib-clean');
    grunt.loadNpmTasks('grunt-contrib-watch');

    grunt.initConfig({
        localization: grunt.file.readJSON(path.resolve(__dirname, 'smartling.config.json')),
        webpack: {
            options: {
                resolve: {
                    alias: {
                        "event-emitter": "wolfy87-eventemitter",
                    },
                },
                module: {
                    loaders: [
                        {
                            test:   /\.js$/,
                            loader: 'babel-loader',
                            include: [
                                path.resolve(SRC_PATH, 'js'),
                                path.resolve(__dirname, 'node_modules/pubsub-js-client'),
                            ],
                            query: {
                                presets: ['es2015'],
                            },
                        },
                        {
                            test: /\.sass$/,
                            loader: ExtractTextPlugin.extract('style-loader', 'css!sass?indentedSyntax=true'),
                        },
                    ],
                },
            },
            player: {
                devtool: (DEV_MODE ? 'source-map' : null),
                entry: {
                    player:     path.resolve(SRC_PATH, 'js', 'player'),
                    "embed-v0": path.resolve(SRC_PATH, 'js', 'embed', 'embedv0'),
                    "embed/v1": path.resolve(SRC_PATH, 'js', 'embed', 'embedv1'),
                },
                output: {
                    path:          DEST_PATH,
                    publicPath:    '',
                    filename:      'js/[name].[chunkhash].js',
                    libraryTarget: 'umd',
                    library:       ['Twitch', 'video'],
                    sourceMapFilename: '[file].map',
                },
                plugins: [
                    new webpack.ProvidePlugin({
                        '$': 'jquery',
                        '_': 'underscore',
                    }),
                    new ExtractTextPlugin("css/[name].[contenthash].css"),
                    new webpack.ResolverPlugin(
                        new webpack.ResolverPlugin.DirectoryDescriptionFilePlugin("bower.json", ["main"])
                    ),
                    (DEV_MODE ? function() {} : new webpack.optimize.UglifyJsPlugin()),
                    new HtmlWebpackPlugin({
                        template: path.resolve(SRC_PATH, 'html', 'index.html'),
                        chunks:   ['player', 'vendor'],
                        favicon:  path.resolve(__dirname, 'img', 'favicon.png'),
                        minify: {
                            collapseWhitespace:        !DEV_MODE,
                            minifyJS:                  !DEV_MODE,
                            removeRedundantAttributes: true,
                            removeComments:            !DEV_MODE,
                            useShortDoctype:           true,
                        },
                    }),
                    (DEV_MODE ? new HtmlWebpackPlugin({
                        template: path.resolve(SRC_PATH, 'html', 'embed.html'),
                        chunks:   ['embed/v1'],
                        filename: 'embed.html',
                    }) : function() {}),
                    new SymlinkPlugin(),
                    new CopyWebpackPlugin([
                        { from: 'localization', to: 'localization' },
                        { from: 'localization/en-US', to: 'localization/en' }
                    ], { ignore: ['*.pot', '*.po'] })
                ],
            },
        },
        clean: {
            build: [ DEST_PATH ],
        },
        watch: {
            scripts: {
                files: [
                    path.join(SRC_PATH, 'js', '**', '*.js'),
                    path.join(SRC_PATH, '**', '*.html'),
                    path.join(SRC_PATH, 'sass', '*.sass'),
                    path.join(__dirname, 'vendor', '*.swf'),
                ],
                tasks: ['build'],
                options: {
                    spawn: false,
                },
            },
        },
        browserSync: {
            dev: {
                bsFiles: {
                    src: [
                        path.join(DEST_PATH, '**/*'),
                    ]
                },
                options: {
                    watchTask: true,
                    host:      'localhost.twitch.tv',
                    port:      (IS_ROOT ? 80 : 8000),
                    open:      'external',
                    startPath: "?channel=monstercat&debug=true",
                    server:    {
                        baseDir: DEST_PATH,
                    },
                }
            }
        },
    });

    grunt.loadTasks('tasks');
};

function SymlinkPlugin() {}

SymlinkPlugin.prototype.apply = function(compilation) {
    compilation.plugin("after-emit", function(compilation, callback) {
        var assets = compilation.getStats().toJson({
            hash: true,
            assets: true,
        }).assets;

        Promise.all(Object.keys(assets).map(function(k) {
            var asset = assets[k];
            return new Promise(function(resolve) {
                var assetName = path.basename(asset.name).replace(/\.[0-9a-f]+\./, '.');
                var assetPath = path.resolve(DEST_PATH, path.dirname(asset.name), assetName);
                var target = path.basename(asset.name);
                if ((asset.chunkNames.length === 0 && !/\.swf$/.test(target)) || (target === assetName)) {
                    // if it doesn't belong to any chunks, or didn't contain a
                    // hash in the original asset name, don't symlink
                    resolve();
                } else {
                    fs.unlink(assetPath, function(err) {
                        if (err && err.code !== 'ENOENT') {
                            compilation.errors.push(err);
                            resolve();
                        } else {
                            fs.symlink(target, assetPath, function(err) {
                                if (err) {
                                    compilation.errors.push(err);
                                }
                                resolve();
                            });
                        }
                    });
                }
            });
        })).then(function() {
            callback();
        });
    });
};
