#!/usr/bin/env bash
set -e

usage() {
	echo "Usage: $0 -d <directory> -e <environment>" 1>&2
	exit 1
}

while getopts ":b:d:e:" o; do
	case "${o}" in
		b)
			BRANCH=${OPTARG}
			;;
		d)
			SOURCE=${OPTARG}
			;;
		e)
			ENVIRONMENT=${OPTARG}
			;;
		*)
			usage
			;;
	esac
done
shift $((OPTIND-1))

if [ -z "${SOURCE}" ] || [ -z "${ENVIRONMENT}" ]; then
	usage
fi

BUCKET="twitch-video-player"
DESTINATION="$ENVIRONMENT"

# Only add the branch postfix on test.
if [ "$ENVIRONMENT" == "test" ]; then
	DESTINATION="$DESTINATION/$BRANCH"
fi

# Upload the non symlink files first.
aws s3 sync "$SOURCE/" "s3://$BUCKET/$DESTINATION/" --no-follow-symlinks --cache-control "public, max-age=1296000"

# Find and loop over all symlinks in the directory.
SYMLINKS="$(cd $SOURCE; find * -type l)"
for SYMLINK in $SYMLINKS; do
	# Get the target of the symlink.
	TARGET="$(dirname "$SYMLINK")/$(readlink "$SOURCE/$SYMLINK")"

	# Upload each symlink individually with different cache control headers.
	aws s3 cp "$SOURCE/$TARGET" "s3://$BUCKET/$DESTINATION/$SYMLINK" --follow-symlinks --cache-control "public, max-age=60"
done

# Configure the index/error file.
aws s3 website s3://$BUCKET/ --index-document "index.html"

# Print the url to make it easier.
case "$ENVIRONMENT" in
	production)
		echo "http://player.twitch.tv/"
		;;
	beta)
		echo "http://betaplayer.twitch.tv/"
		;;
	test)
		echo "http://testplayer.twitch.tv/$BRANCH/"
		;;
esac
