# Overview

`minixperiment-client-js` is a Twitch internal library for interfacing with the
Minixperiment configuration service in Javascript environments.

The client library handles interfacing with the various experiment configuration
services via `providers`, which are minimally configured interfaces for
retrieving experiment configurations from various sources. It also handles
reporting experiment treatments into various services that track split testing
results.

It is **VERY IMPORTANT** that experiment treatments are only requested at the
time of use; it will send a tracking event back to the data aggregation service
that a user has been assigned into a specific treatment, and requesting the
treatment before use *does not* guarantee that the user experiences the
treatment, which can negatively affect the validity of the data set.

# Installation

`npm install --save git+ssh://git@git-aws.internal.justin.tv:common/minixperiment-client-js.git`

Typically, you should use one of the tagged releases. There are three variants
to choose from: AMD, UMD, and Common JS. Each release is versioned in accordance
with Semver; major releases are backwards-incompatible, minor releases are
backwards-compatible but may change functionality, and patch releases should not
affect the interface or return values except as required to correct buggy behavior.

*Common JS* releases are tagged as *vx.y.z*
*AMD* releases are tagged as `vx.y.z-amd`
*UMD* releases are tagged as `vx.y.z-umd`

AMD and UMD releases contain a few precompiled files meant for inclusion via
module loaders or `<script>` tags directly; these files are located under
`dist/`. Include the primary `minixperiment.client.js` bundle and one of the
providers, as necessary. See [Providers](#Providers).

# API

## new MinixperimentClient(config:Object)

Creates a new Minixperiment client. This is *required* to use Minixperiment.

Parameters:
```js
{
    /* The default values for the experiments used by the application. Should
       anything go wrong with retrieving the experiment configuration, these
       defaults will guarantee that your application does not break. */
    "defaults": {
        "2ef17794-14af-4448-bf23-691c96a72995": "override me",
        "3f589f3f-d5a3-4191-9f70-5d1532925da9": "default value"
    },

    /* The unique device ID associated with the consumer. For web-client, this
       is the device ID assigned to the web browser. Other consumers of this
       library will need to generate a unique identifier for a client/user as
       needed, and persist it across sessions as necessary. */
    "deviceID": "1234567890abcdef",

    /* The override values for any experiments with separate business logic
       controlling whether or not a user should be forced into a particular
       treatment group. Values may be strings, or Promises that resolve to
       strings, or (if the determination is made to *not* override the assignment
       for the user), a rejected Promise. If the Promise is rejected, then the
       assignment proceeds as if there were no override; otherwise, the given
       value is returned in all subsequent calls to `get`. */
    "overrides": {
        "2ef17794-14af-4448-bf23-691c96a72995": "override value",
        "3f589f3f-d5a3-4191-9f70-5d1532925da9": Promise.reject("ignore this")
    },

    /* The location of the consumer. For web, this is "web". */
    "platform": "web",

    /* The experiment configuration source. Minixperiment is bundled
       with two providers: Local and Service. See [Providers](#Providers)
       for more information. */
    "provider": new Provider(...),

    /* If this client is being used in a context where users can be
       logged in, this should be set to the login name of the user. */
    "login":    "YourUser", // or null, if the user is not logged in

    /* The injected Promises library, if being used in a context where
       the native Promise object is not available. */
    "Promise":  Ember.RSVP.Promise
}
```

## minixperiment.get(experimentID:String):Promise<String>

Retrieves a group assignment from the configured experiment set. The return value
is a Promise containing a string; this is configured value, either from a locally
sourced configuration or from the service.

The group assignment should ALWAYS be normalized against a known set of valid
values; if an unexpected value is returned from the service, the program should
not fail. For example, if your program expects to be assigned into either
`"control"` or `"experiment"`, then if the service is configured to return
another group `"test"`, the program should fall back to a valid "base" group;
in this example, it would be `"control"`.

The return value _SHOULD NOT_ be cached; you will get consistent assignments per
experiment at each call, but caching the value will prevent analytics from being
collected.

If an error is encountered at any point, the value contained in the returned
promise will be the default value configured when instantiating the client.

# Example

See the [`example`](https://git-aws.internal.justin.tv/common/minixperiment-client-js/tree/master/example)
directory for a working example of this library.

# Appendix

## Client

```js
var MinixperimentClient = require('minixperiment-client');

// or ES6

import MinixperimentClient from 'minixperiment-client';

const EXPERIMENT_UUID = "3f589f3f-d5a3-4191-9f70-5d1532925da9";

// Configure the service; this MUST be done before attempting to retrieve any
// experiment treatment assignments!
// Additionally, this should only happen once; reconfiguring the library after
// it has been used may cause unexpected behavior.
let client = new MinixperimentClient({
    // Consumers MUST provide defaults for all experiments they will use over
    // the course of the application.
    defaults: {
        [EXPERIMENT_UUID]: "experiment default value"
    },

    // Consumers MUST provide the device ID used to assign users into
    // treatment groups
    deviceID: "abc123",

    // Consumers MUST provide the platform for this particular integration;
    // e.g. the web client should pass in `web` for this parameter
    platform: "web",

    // Consumers SHOULD provide the username associated with this event, if
    // the user is logged in. The property may be omitted otherwise, or can be
    // set to `null`
    login: "KappaSpammer98",

    // See [Providers](#Providers)
    provider: new Provider(/* ... */),

    // You'll also need to provide a valid Promises/A+-compliant library
    Promise:  Promise
});

// Get the experimental treatment assignment
var assignment = minixperiment.get(EXPERIMENT_UUID);

// `assignment` is a Promise containing the treatment value to use for the
// experiment; if there is a problem retrieving the assignment, the default
// value provided during client instantiation will be returned (or `null`, if
// a default value is not specified)

assignment.then(function(treatment) {
    switch (treatment) {
    case 'control':
        doTheNormalThing();
        break;
    case 'treatment A':
        doSomethingExperimental();
        break;
    }
});
```

## Experiments Configuration

A valid experiments configuration is a hash of experiment uuid to experiment
configuration. An experiment configuration should have a field, `groups`, which
is an array of `value`s and `weight`s. Example:

```js
var experimentsConfig = {
    "de305d54-75b4-431b-adb2-eb6b9e546014": {
        name:   "A human-readable name of the experiment",
        groups: [
            { value: "true",  weight: 1 },
            { value: "false", weight: 9 }
        ]
    }
};
```

Values are always strings; as above, `"true"` and `"false"` will returned by
the service, not the Javascript `true` and `false` boolean values.

Weights must be non-negative integers, and represent the relative size of the
treatment groups to which user will be assigned. In the above example, 10% of
user will receive a value of `true` for the `enable_new_feature` experiment,
while 90% of users will receive a value of `false`.

## Providers

Providers are Minixperiment's way of accessing the experiment configuration.
You will need to provide Minixperiment with a provider in the initial call to
instantiate the client. The provider you use to configure Minixperiment will be
used to retrieve experiment assignments needed at various points in your app.

### LocalProvider

The `LocalProvider` is a thin wrapper around an experiment configuration that
you already have, and just need to expose through Minixperiment. This is currently
used on web-client; the experiment configuration is set in an ArbKey and made
available through `SiteOptions`, which is passed in to Minixperiment via a
`LocalProvider`. It is also useful for testing various configurations during
development, and defaulting back to the `ServiceProvider` when development is
done.

```js
// note: this example uses the globals created by the UMD build
var client = new Minixperiment.Client({
    // ...
    provider: new Minixperiment.providers.local.LocalProvider(SiteOptions.experiments)
});
```

### ServiceProvider

The `ServiceProvider` fetches the experiment configuration from the specified
Minixperiment server (local for development, or a staging/production server),
so that users can change experiments without requiring a code deployment. This
does introduce a small delay before `get` calls resolve while the request is in
progress. The `ServiceProvider` exposes a `SERVICE_URL` constant that points to
the production Minixperiment service.

```js
// note: this example uses the globals created by the UMD build
var client = new Minixperiment.Client({
    // ...
    provider: new Minixperiment.providers.service.ServiceProvider(Minixperiment.providers.service.SERVICE_URL)
});
```
