var experiments = require('./experiments');
var spade       = require('./tracking/spade');

module.exports = MinixperimentClient;

/**
 * Create a new Minixperiment client object, from which assignments can be
 * determined.
 * @class
 *
 * @param {Object} config
 *        Configuration object for the library. Valid properties:
 *        `defaults` : an object mapping expected experiment UUIDs to their
 *                     default values, to be returned should an error occur
 *                     somewhere within the library. It is considered an error to
 *                     request an experimental treatment from an experiment
 *                     without a default value.
 *        `deviceID` : the unique ID associated with the device using this library
 *        `login`    : the username of the current user
 *        `overrides`: An object hash of experiment ID to forced assignment,
 *                      which causes the client to ignore any configuration and
 *                      simply return the overridden value. The value may be a
 *                      Promise, which is used if it resolves and is ignored if it
 *                      is rejected.
 *        `platform` : The consumer of this library (e.g. `web`, `xboxone`)
 *        `provider` : a minixperiment-defined Experiments provider
 *        `Promise`  : a Promises/A+-compliant implementation
 */
function MinixperimentClient(config) {
	var configError = validateConfig(config);
	if (configError !== null) {
		throw configError;
	}

	this._config      = getExperimentsConfiguration(config);
	this._Promise     = config.Promise;
	this._deviceID    = config.deviceID;
	this._platform    = config.platform;
	this._username    = config.login || null;
	this._defaults    = determineDefaults(config.Promise, config.defaults, config.overrides || {});
	this._assignments = determineAssignments(
		config.Promise,
		this._config,
		this._defaults,
		config.overrides || {},
		this._deviceID
	);
}

/**
 * Get the treatment for a particular named experiment
 *
 * @param {String} experimentUUID
 *        The UUID of the experiment from which the client will draw a treatment
 * @param {Object} options
 *        `mustTrack` [default: false]: if true, then tracking must complete
 *            before the assignment is returned to the caller
 * @return {Promise}
 *         Resolves to the value of the treatment for the given experiment, or
 *         the provided default value in the event of an error
 */
MinixperimentClient.prototype.get = function(experimentUUID, opts) {
	var options = applyDefaults(opts || {}, {
		mustTrack: false
	});
	var assignment = (this._assignments[experimentUUID] ||
					  this._Promise.reject(new Error("No experiment with ID `" + experimentUUID + "`")));

	var trackedEvent = this._Promise.all([this._config, assignment]).then(function(data) {
		var expConfig = data[0];
		var treatment = data[1];

		// only track assignments that are valid
		var trackingProperties = {
			// epoch time of the event, in seconds
			"client_time":      (new Date()).getTime() / 1000,
			// user's unique device ID
			"device_id":        this._deviceID,
			// experiment identifier
			"experiment_id":    expConfig[experimentUUID].name,
			// which group the user was assigned to
			"experiment_group": treatment,
			// the platform from which this experiment was experienced
			"platform":         this._platform,
		};
		if (this._username !== null) {
			// the user's username, if logged in
			trackingProperties.login = this._username;
		}

		var spadePromise = new this._Promise(function(resolve, _) {
			spade.sendEvent('experiment_branch', trackingProperties, resolve);
		}).then(null, function() { return null; });

		return spadePromise;
	}.bind(this));

	return this._Promise.all([assignment, options.mustTrack ? trackedEvent : null]).then(
		function(data) {
			return data[0];
		},
		function(err) {
			console.warn(err);
			return this._defaults[experimentUUID] || null;
		}.bind(this)
	);
};

/**
 * Validate the Minixperiment client configuration, returning an error if there
 * are any issues, or `null` for "OK".
 *
 * @param {Object} config
 * @return {Error?}
 */
function validateConfig(config) {
	if (!config.defaults || Object.getPrototypeOf(config.defaults) !== Object.prototype) {
		return new Error("Invalid defaults; expected object, got " + JSON.stringify(config.defaults));
	} else if (typeof config.deviceID !== 'string' || config.deviceID.length === 0) {
		return new Error("Invalid device ID; expected non-empty string, got `" + config.deviceID + "`");
	} else if (typeof config.platform !== 'string' || config.platform.length === 0) {
		return new Error("Invalid platform; expected non-empty string, got `" + config.platform + "`");
	} else if (
		typeof config.provider !== 'object' ||
		typeof config.provider.getExperimentConfiguration !== 'function'
	) {
		return new Error("Invalid provider");
	} else if (typeof config.Promise !== 'function') {
		return new Error("Invalid Promise implementation");
	}

	return null;
}

function getExperimentsConfiguration(config) {
	return new config.Promise(function(resolve, reject) {
		config.provider.getExperimentConfiguration(resolve, reject);
	}).then(function(experimentConfig) {
		var error = experiments.validate(experimentConfig);
		if (error) {
			throw error;
		}

		return experimentConfig;
	});
}

/**
 * Combine overrides with the defaults to generate the actual set of defaults
 * used when an issue is encountered.
 *
 * @param {*} _Promise The Promises/A+ implementation object
 * @param {Object<String, String>} defaults
 * @param {Object<String, (String|Promise<String>)>} overrides
 * @return {Object<String, Promise<String>>}
 */
function determineDefaults(_Promise, defaults, overrides) {
	var actualDefaults = {};

	for (var uuid in defaults) {
		actualDefaults[uuid] = (function(expID) {
			return _Promise.resolve(overrides[uuid]).then(
				function(override) {
					// possibly an undefined override; ensure a valid value is given
					return (typeof override === 'string' ? override : defaults[expID]);
				},
				function() {
					return defaults[expID];
				}
			);
		})(uuid);
	}

	return actualDefaults;
}

/**
 * Translate the experiment configuration and client configuration into a set of
 * experiment treatment assignments. This is resolved during client instantiation,
 * which prevents any reconfiguration after the client is running.
 *
 * @param {*} _Promise The Promises/A+ implementation object
 * @param {Promise<Object<UUID, ExperimentConfig>>} experimentConfig
 * @param {Object<UUID, String>} defaults
 * @param {Object<UUID, (String|Promise<String>)>} overrides
 * @param {String} deviceID
 * @return {Object<UUID, Promise<String>>}
 */
function determineAssignments(_Promise, experimentConfig, defaults, overrides, deviceID) {
	var assignments = {};

	for (var uuid in defaults) {
		if (!defaults.hasOwnProperty(uuid)) {
			continue;
		}
		assignments[uuid] = (function(expID) {
			return experimentConfig.then(
				function(cfg) {
					if (!cfg.hasOwnProperty(expID)) {
						throw new Error("Experiment `" + expID + "` is deprecated");
					}

					return experiments.selectTreatment(expID, cfg[expID], deviceID);
				},
				function(err) {
					return defaults[expID];
				}
			).then(function(assignment) {
				return _Promise.resolve(overrides[expID]).then(function(override) {
					return (typeof override === 'string' ? override : assignment);
				}, function() {
					return assignment;
				});
			});
		})(uuid);
	}

	return assignments;
}

/**
 * Creates a new object with the properties of the given source object, filling
 * in any missing/non-existent properties with values from the defaults object.
 *
 * @param {Object} src
 * @param {Object} defaults
 * @return {Object}
 */
function applyDefaults(src, defaults) {
	var prop;
	var rv = {};

	for (prop in src) {
		if (src.hasOwnProperty(prop)) {
			rv[prop] = src[prop];
		}
	}

	for (prop in defaults) {
		if (defaults.hasOwnProperty(prop) && !src.hasOwnProperty(prop)) {
			rv[prop] = defaults[prop];
		}
	}

	return rv;
}
