var experiments = require('../lib/experiments');

describe("experiments", function() {
	var specID = 1;
	var deviceID;
	var experimentID;

	beforeEach(function() {
		specID++;
		deviceID = 'experiments_deviceid_' + specID;
		experimentID = 'experiments_expid_' + specID;
	});

	describe("#validate", function() {
		var failingCases = [
			{
				config: {
					"no_groups_property": {"not": "empty"}
				},
				expectedError: "missing a `groups` property"
			},
			{
				config: {
					"no_groups_members": { groups: [] }
				},
				expectedError: "`groups` has no members"
			},
			{
				config: {
					"no_value_property": {
						groups: [
							{ not_value: 1, weight: 10 }
						]
					}
				},
				expectedError: "missing a `value` property"
			},
			{
				config: {
					"no_weight_property": {
						groups: [
							{ value: 1, not_weight: 10 }
						]
					}
				},
				expectedError: "missing a `weight` property"
			},
			{
				config: {
					"negative_weight": {
						groups: [
							{ value: 1, weight: -1 }
						]
					}
				},
				expectedError: "has a negative weight"
			},
			{
				config: {
					"non-integer_weight": {
						groups: [
							{ value: 1, weight: 0.3 }
						]
					}
				},
				expectedError: "has a non-integer weight"
			}
		];

		failingCases.forEach(function(testCase) {
			var experimentName = Object.keys(testCase.config)[0];

			it("should return an error when " + experimentName, function() {
				var error = experiments.validate(testCase.config);

				expect(error.message).toContain(testCase.expectedError);
			});
		});

		var passingCases = [
			{
				// empty
			},
			{
				one_treatment: {
					groups: [
						{ value: 1, weight: 3 }
					]
				}
			},
			{
				disabled_treatment: {
					groups: [
						{ value: 1, weight: 0 },
						{ value: 2, weight: 1 }
					]
				}
			},
			{
				string_value: {
					groups: [
						{ value: "one", weight: 1 },
						{ value: "two", weight: 1 }
					]
				}
			},
			{
				boolean_value: {
					groups: [
						{ value: true,  weight: 1 },
						{ value: false, weight: 1 }
					]
				}
			}
		];

		passingCases.forEach(function(testCase) {
			var experimentName = Object.keys(testCase)[0];

			it("should successfully validate experiments with " + experimentName, function() {
				var error = experiments.validate(testCase);

				expect(error).toBe(null);
			});
		});
	});

	describe("#selectTreatment", function() {
		it("should select a treatment from the set of configured treatments", function() {
			var experiment = {
				groups: [
					{ value: "1", weight: 1 },
					{ value: "2", weight: 1 },
					{ value: "3", weight: 1 }
				]
			};

			var result = experiments.selectTreatment(experimentID, experiment, deviceID);

			expect(result).toBeOneOf(["1", "2", "3"]);
		});

		it("should give a reasonably uniform distribution of selections", function() {
			var trials           = 10000;
			var tolerance        = 0.05;
			var experimentConfig = {
				groups: [
					{ value: "0", weight: 1 },
					{ value: "1", weight: 3 },
					{ value: "2", weight: 5 },
					{ value: "3", weight: 7 }
				]
			};
			var results = [0, 0, 0, 0];
			for (var i = 0; i < trials; i++) {
				var assignment = experiments.selectTreatment(
					experimentID,
					experimentConfig,
					"device_" + i
				);
				// use the assignment as the index to increment
				results[parseInt(assignment, 10)]++;
			}

			var totalWeight = experimentConfig.groups.reduce(function(sum, treatment) {
				return sum + treatment.weight;
			}, 0);
			for (var i = 0; i < experimentConfig.groups.length; i++) {
				var actual   = (results[i] / trials);
				var expected = (experimentConfig.groups[i].weight / totalWeight);
				var residual = (actual - expected);
				expect(Math.abs(residual)).toBeLessThan(tolerance);
			}
		});
	});
});
