import logging from "./log";

const TWITCH_PROTOCOL = "pubsub";

var logger = logging._getLogger("IframeHost");

class IframeHost {
  constructor (driver) {
    this._driver = driver;
    this._sources = [];
    this._driver.on("connected", this.handleConnected, this);
    this._driver.on("disconnected", this.handleDisconnected, this);
    window.addEventListener("message", this.receiveMessage.bind(this), false);
  }

  receiveMessage (event) {
    if (event.data.twitch_protocol != TWITCH_PROTOCOL) {
      return;
    }
    logger.debug("Received message: " + JSON.stringify(event.data));
    switch (event.data.type) {
    case "LISTEN":
      this.handleListen(event.source, event.data.nonce, event.data.data);
      break;
    case "UNLISTEN":
      this.handleUnlisten(event.source, event.data.nonce, event.data.data);
      break;
    case "connect":
      this._sources.push(event.source);
      this._driver.connect();
      break;
    case "verify":
      event.source.postMessage({
        twitch_protocol: TWITCH_PROTOCOL,
        type: "verify"
      }, "*");
      break;
    }
  }

  handleListen (source, nonce, data) {
    var success = (function () {
      source.postMessage({twitch_protocol: TWITCH_PROTOCOL, type: "success", nonce: nonce}, "*");
    });
    var failure = (function (err) {
      source.postMessage({twitch_protocol: TWITCH_PROTOCOL, type: "failure", nonce: nonce, error: err}, "*");
    });
    var message = (function (msg) {
      source.postMessage({twitch_protocol: TWITCH_PROTOCOL, type: "message", topic: data.topics[0], message: msg}, "*");
    });
    this._driver.Listen({
      topic: data.topics[0],
      auth: data.auth_token,
      success: success,
      failure: failure,
      message: message
    });
  }

  handleUnlisten (source, nonce, data) {
    var success = (function () {
      source.postMessage({twitch_protocol: TWITCH_PROTOCOL, type: "success", nonce: nonce}, "*");
    });
    var failure = (function (err) {
      source.postMessage({twitch_protocol: TWITCH_PROTOCOL, type: "failure", nonce: nonce, error: err}, "*");
    });
    var message = (function (msg) {
      source.postMessage({twitch_protocol: TWITCH_PROTOCOL, type: "message", topic: data.topics[0], message: msg}, "*");
    });
    this._driver.Unlisten({
      topic: data.topics[0],
      auth: data.auth_token,
      success: success,
      failure: failure,
      message: message
    });
  }

  handleConnected () {
    for (var i = 0; i < this._sources.length; i++) {
      this._sources[i].postMessage({
        twitch_protocol: TWITCH_PROTOCOL,
        type: "connected"
      }, "*");
    }
  }

  handleDisconnected () {
    for (var i = 0; i < this._sources.length; i++) {
      this._sources[i].postMessage({
        twitch_protocol: TWITCH_PROTOCOL,
        type: "disconnected"
      }, "*");
    }
  }
}

export default IframeHost;
