/* global Hls */
// Interfaces with libhls to feed a video element with MSE.
export function AdapterMse(video) {
    var self = this;

    let hls;
    let hlsLoaded;
    let mediaSource;
    let tracks = {};

    var init = function() {
        hlsLoaded = AdapterMse.loadScript().then(initHls);
    };

    var initHls = function() {
        hls = new Hls();
        hls.onWrite(onWrite);
        hls.onComplete(onComplete);

        mediaSource = new MediaSource();

        video.addEventListener('progress', onProgress);
        video.src = window.URL.createObjectURL(mediaSource);

        return hls;
    };

    var softReset = function() {
        // Remove any existing tracks.
        for (var id in tracks) {
            var track = tracks[id];
            mediaSource.removeSourceBuffer(track.buffer);
        }

        tracks = {};
    };

    var hardReset = function() {
        // Need to make a brand new MediaSource
        mediaSource = new MediaSource();
        video.src = window.URL.createObjectURL(mediaSource);

        tracks = {};
    };

    // Called when libhls has some data for us.
    var onWrite = function(mime, buffer) {
        var track = tracks[mime];
        if (!track) {
            track = {};

            track.queue = [];
            track.buffer = mediaSource.addSourceBuffer(mime);
            track.buffer.mode = 'segments';
            track.buffer.addEventListener('updateend', function() {
                onUpdate(track);
            });

            tracks[mime] = track;
        }

        // Try writing the data to the buffer.
        var written = tryWrite(track, buffer);
        if (!written) {
            // Make a copy of the data and add it to a queue.
            var copy = new Uint8Array(buffer);
            track.queue.push(copy);
        }
    };

    var onComplete = function() {
        // TODO Implement multiple source buffers for supporting browsers.
        // TODO Use the error code to determine what to do.
        hardReset();
    };

    var onUpdate = function(track) {
        if (!track.queue.length) {
            // Use the free cycles to trim the track buffer size.
            trimBuffer(track);
            return;
        }

        var buffer = track.queue.shift();
        if (!tryWrite(track, buffer)) {
            // Put the buffer back if we failed to write.
            track.queue.unshift(buffer);
        }
    };

    var tryWrite = function(track, buffer) {
        if (mediaSource.readyState === 'ended') {
            // Stream is finished, ignore everything.
            // TODO Make sure libhls is shut down.
            return true;
        }

        if (mediaSource.readyStart === 'closed') {
            // mediaSource has not finish initializing yet.
            return false;
        }

        if (track.queue.length) {
            // There is a queue, get in line.
            return false;
        }

        if (track.buffer.updating) {
            // Currently updating, not allowed to do anything.
            return false;
        }

        track.buffer.appendBuffer(buffer);
        return true;
    };

    // Remove any buffered video older than 10 seconds.
    var trimBuffer = function(track) {
        if (!track.buffer && track.buffer.updating) {
            return;
        }

        var cutoff = video.currentTime - 10;
        if (video.buffered.length > 0 && video.buffered.start(0) < cutoff) {
            track.buffer.remove(0, cutoff);
        }
    };

    var onProgress = function() {
        // This is a hack because VODs do not start at timestamp 0.
        // TODO Fix the underlying VODs if possible.
        if (video.buffered.length === 0) {
            return;
        }

        var time = video.currentTime;
        var buffered = video.buffered;

        for (var i = 0; i < buffered.length; i++) {
            var start = buffered.start(i);
            if (start > time) {
                console.warn(`seeking ahead ${(start - time)} seconds`);
                video.currentTime = start;

                break;
            }

            var end = buffered.end(i);
            if (end >= time) {
                break;
            }
        }
    };

    // Ensure there is only one scheduled perform.
    var performTimeout = null;
    var perform = function() {
        var timeout = hls.perform();

        // If libhls is working correctly then it should never return -1,
        // because that means perform never needs to be called again.
        // TODO Can we remove this, does it actually work?
        if (timeout < 0) {
            timeout = 1000;
        }

        if (timeout >= 0) {
            // Clear any previously scheduled perform.
            clearTimeout(performTimeout);

            // Schedule the next perform.
            performTimeout = setTimeout(perform, timeout);
        }
    };

    self.destroy = function() {
        // Cancel the perform loop.
        clearTimeout(performTimeout);
        performTimeout = null;

        // Stop appending the media source.
        softReset();

        mediaSource.endOfStream();
        mediaSource = null;

        tracks = {};

        // Free the memory.
        hls.free();
        hls = null;
    };

    self.load = function(src) {
        hlsLoaded.then(function() {
            hls.load(src);
            perform();
        });
    };

    self.getQuality = function() {
        if (hls) {
            return hls.getQuality();
        }
    };

    self.setQuality = function(value) {
        hlsLoaded.then(function() {
            hls.setQuality(value);
        });
    };

    self.getQualities = function() {
        if (hls) {
            return hls.getQualities();
        }
    };

    init();
}

// hls.js is so large that we load it asynchronously.
AdapterMse.loadScript = _.memoize(function() {
    return new Promise(function(resolve) {
        require.ensure([], function() {
            resolve(require('script!../../../vendor/hls'));
        });
    });
});

AdapterMse.canPlay = function() {
    // Chrome is bugged and MUST have double quotes.
    var mime = 'video/mp4; codecs="avc1.42E01E,mp4a.40.2"';
    return window.MediaSource && MediaSource.isTypeSupported(mime);
};
