import { MinutesWatched } from '../util/minutes-watched';
import { CountessTracker } from './countess';
import * as Settings from '../settings';
import {
    userInfo,
    channelAPIInfo,
    channelViewerInfo,
    channelInfo,
    videoInfo,
} from '../api';
import * as Stats from '../util/stats';
import { ValveClient } from './valve';
import { NETWORK_PROFILE_COLLECTION } from '../experiments';

const VIDEO_INIT = 'video_init';
const VIDEO_PLAY = 'video-play';
const VIDEO_ERROR = 'video_error';
const MINUTE_WATCHED = 'minute-watched';
const BUFFER_EMPTY = 'buffer-empty';
const BUFFER_REFILL = 'buffer-refill';
const VOD_SEEK = 'vod_seek';

const MINIMUM_SEEK_TIME = 500;

export function Analytics(player, tracker, experiments, options) {
    const minutesWatchedTimer = new MinutesWatched();

    let hasPlayed = false;
    let bufferEmptyStartTime = null;
    let bufferEmptyCount;
    let lastNetworkProfile = -Infinity;
    let trackNetworkProfile;
    let countessTracker;
    let valveClient;
    let lastSeekTime = null;
    let timeStampBeforeSeek = 0;
    let isSeekInProgress = false;

    var init = function() {
        // Create the Countess tracking client.
        countessTracker = new CountessTracker({
            host: Settings.countessHost,
        });

        valveClient = new ValveClient();
        trackNetworkProfile = experiments.get(NETWORK_PROFILE_COLLECTION);

        initProperties();
        initEvents();
    };

    var initProperties = function() {
        // Gather user info from the API.
        tracker.setProperties(
            userInfo().then(function(info) {
                return {
                    login: info.login,
                    turbo: Boolean(info.turbo),
                };
            })
        );

        // Unable to use getChannel as it hasn't allocated a value in time...
        if (options.channel) {
            var channelInfo = channelAPIInfo(options.channel);
            valveClient.channelInfo(channelInfo);

            // Gather channel viewer info if live.
            var viewerInfo = channelViewerInfo(options.channel);
            valveClient.viewerInfo(viewerInfo);
            tracker.setProperties(
                viewerInfo.then(function(info) {
                    return {
                        subscriber: Boolean(info.chansub),
                    };
                })
            );
        }
    };

    var initEvents = function() {
        minutesWatchedTimer.on('minute', onMinuteWatched);

        player.addEventListener('init', onInit);
        player.addEventListener('loadstart', onLoadStart);
        player.addEventListener('pause', onPause);
        player.addEventListener('waiting', onWaiting);
        player.addEventListener('playing', onPlaying);
        player.addEventListener('playing', onQualityChange);
        player.addEventListener('ended', onEnded);
        player.addEventListener('qualitychange', onQualityChange);
        player.addEventListener('castingchange', onCastingChange);
        player.addEventListener('error', onVideoError);
        player.addEventListener('seeking', onSeeking);
        player.addEventListener('timeupdate', onTimeUpdate);

        // TODO Clean up this interface.
        player.addEventListener('isspectre', onSpectre);
    };

    /**
     * Compute the aggregate stats for a given network profile, in the format
     * sent as part of minute-watched.
     *
     * @param {Array<NetworkProfile>} networkProfile
     * @return {Object}
     */
    var getNetworkProfileStats = function(networkProfile) {
        if (networkProfile.length === 0) {
            return {};
        }

        // Compute aggregate statistics for the segment stats
        var ttfbStats = Stats.getStats(_.pluck(networkProfile, 'firstByteLatency'));
        var dlStats = Stats.getStats(_.pluck(networkProfile, 'downloadDuration'));
        var aggBitrate = networkProfile.reduce(function(props, segment) {
            return {
                bytes: props.bytes + segment.bytes,
                duration: props.duration + segment.segmentDuration,
            };
        }, {
            bytes: 0,
            duration: 0,
        });
        var avgBitrate = ((aggBitrate.bytes * 8) / 1024) / (aggBitrate.duration / 1000);

        /* eslint-disable camelcase */
        return {
            time_to_first_byte: ttfbStats.mean,
            stdev_time_to_first_byte: ttfbStats.stdev,
            download_duration: dlStats.mean,
            stdev_download_duration: dlStats.stdev,
            bitrate: avgBitrate,
        };
        /* eslint-enable camelcase */
    };

    var trackEvent = function(name, properties) {
        // TODO Implement these properties for mse/hls backends.
        var videoInfo = player.getVideoInfo();

        // There are 30+ properties stuffed into videoInfo, so filter them.
        // Currently not used: buffer_empty_count, deblocking, format,
        // hls_buffer_duration, hls_load_attempted, implementation,
        // paused, play_load_time, playback_bytes_per_second,
        // playing, smoothing, stream_protocol, stream_time,
        // time, video_load_attempt

        videoInfo = _.pick(videoInfo,
            'bandwidth', // TODO Replace with bitrate.
            'cluster', // Cluster serving the video content.
            'current_bitrate', // Average bitrate in kb/s
            'current_fps', // Average fps.
            'dropped_frames', // Number of unrendered/skipped frames.
            'hls_latency_broadcaster', // Delay between streamer -> viewer.
            'hls_latency_encoder', // Delay between encoder -> viewer.
            'hls_target_duration', // Maximum segment duration.
            'manifest_cluster', // Cluster serving the manifest.
            'manifest_node', // Node serving the manifest.
            'node', // Node serving the video content.
            'origin', // Origin of the VOD (typically "swift" or "s3")
            'stream_format', // The internal name for the current quality.
            'user_ip', // User's IP address.
            'vid_display_height', // Size of the video container.
            'vid_display_width',
            'vid_height', // Size of the actual video content. (ex 720p)
            'vid_width',
            'video_buffer_size' // Seconds of video ready to display.
        );

        properties = _.extend({}, videoInfo, properties);

        // The volume of video playback (moment of trackevent)
        properties.volume = player.getVolume();
        // The muted state of video playback (moment of trackevent)
        properties.muted = player.getMuted();

        tracker.trackEvent(name, properties);
    };

    // Fired every "minute" while the user is actively trying to watch
    // video, starting when the first frame is visible (video-play) and
    // only stopping on pause or ended.
    var onMinuteWatched = function() {
        // Filter the network profile data to only new segment stats
        var networkProfile = player.getNetworkProfile().filter(function(stats) {
            return stats.startTime > lastNetworkProfile;
        });
        lastNetworkProfile = networkProfile.reduce(function(max, stats) {
            return Math.max(max, stats.startTime);
        }, lastNetworkProfile);

        var networkProfileStats = getNetworkProfileStats(networkProfile);

        /* eslint-disable camelcase */
        //if VOD add current timestamp
        if (player.video) {
            _.extend(networkProfileStats, {
                vod_timestamp: player.getCurrentTime(),
            });
        }

        trackEvent(MINUTE_WATCHED, _.extend(networkProfileStats, {
            seconds_offset: minutesWatchedTimer.initialDelay / 1000,
            minutes_logged: minutesWatchedTimer.totalMinutes,
        }));
        /* eslint-enable camelcase */

        // Notify Valve of Minute-Watched event
        valveClient.notify();

        trackNetworkProfile.then(function(collect) {
            if (collect === 'yes' && networkProfile.length > 0) {
                tracker.trackEvents(networkProfile.map(function(stats) {
                    /* eslint-disable camelcase */
                    return {
                        event: 'network_profile',
                        properties: {
                            bytes: stats.bytes,
                            duration: stats.segmentDuration,
                            client_time: (stats.startTime + stats.firstByteLatency + stats.downloadDuration),
                            time_to_first_byte: stats.firstByteLatency,
                            download_duration: stats.downloadDuration,
                        },
                    };
                    /* eslint-enable camelcase */
                }));
            }
        });
    };

    // Fired when the backend has fully initialized and is ready to play.
    var onInit = function() {
        tracker.setProperty('app_version', player.getVersion());
        tracker.setProperty('backend', player.getBackend());

        tracker.trackEvent(VIDEO_INIT);
    };

    // Fired when the user starts trying to load a channel or video.
    // The channel might actually be offline, we don't know yet.
    var onLoadStart = function() {
        // Initial the tracking properties for this new channel/video.
        if (player.getChannel()) {
            onLoadChannel();
        } else if (player.getVideo()) {
            onLoadVideo();
        }

        // Fire video-play on the next `playing` event.
        hasPlayed = false;

        // Reset the buffer empty count.
        bufferEmptyCount = 0;
    };

    var onLoadChannel = function() {
        // Fetch the properties we want to track for this channel.
        var channel = player.getChannel();
        var request = channelInfo(channel);

        // Count the channel view with countess.
        request.then(function(channelInfo) {
            countessTracker.trackChannelView(channelInfo);
        });

        var properties = request.then(function(channelInfo) {
            /* eslint-disable camelcase */
            return {
                channel: channelInfo.name,
                channel_id: channelInfo._id,
                game: channelInfo.game,
                live: true,
                content_mode: 'live',
                partner: channelInfo.partner,
            };
            /* eslint-enable camelcase */
        });

        tracker.setProperties(properties);
    };

    var onLoadVideo = function() {
        // Fetch the properties we want to track for this video.
        var video = player.getVideo();
        var videoInfoRequest = videoInfo(video);

        // Acquire channel info as well
        tracker.setProperties(
            videoInfoRequest.then(function(videoInfo) {
                return channelInfo(videoInfo.channel.name);
            }).then(function(channelInfo) {
                return _.pick(channelInfo, 'partner');
            })
        );

        // Count the video view with countess.
        videoInfoRequest.then(function(videoInfo) {
            countessTracker.trackVODView(videoInfo);
        });

        // Asynchronously set the analytics properties.
        // This promise avoid sending tracking events until resolved.
        var properties = videoInfoRequest.then(function(videoInfo) {
            /* eslint-disable camelcase */
            return {
                channel: videoInfo.channel.name,
                vod: videoInfo._id, // TODO DELETE when removing x_untrusted
                vod_id: videoInfo._id,
                vod_type: videoInfo.broadcast_type,
                game: videoInfo.game,
                live: false,
                content_mode: 'vod',
            };
            /* eslint-enable camelcase */
        });

        tracker.setProperties(properties);
    };

    // Fired when there is not enough video to continue playback.
    var onWaiting = function() {
        if (!player.getSeeking()) {
            bufferEmptyStartTime = (new Date()).getTime();
            bufferEmptyCount++;

            trackEvent(BUFFER_EMPTY, {
                buffer_empty_count: bufferEmptyCount, // eslint-disable-line camelcase
            });
        }
    };

    // Fired when there is enough data and playback has continued.
    var onPlaying = function() {
        // Fire buffer-refill if we previously fired buffer-empty.
        if (bufferEmptyStartTime !== null) {
            var now = (new Date()).getTime();
            trackEvent(BUFFER_REFILL, {
                buffering_time: (now - bufferEmptyStartTime) / 1000, // eslint-disable-line camelcase
            });

            bufferEmptyStartTime = null;
        }

        if (!hasPlayed) {
            trackEvent(VIDEO_PLAY);
            hasPlayed = true;

            // Notify Valve of video-play event
            valveClient.notify();
        }

        minutesWatchedTimer.start();

        if (isSeekInProgress) {
            isSeekInProgress = false;

            /* eslint-disable camelcase */
            trackEvent(VOD_SEEK, {
                timestamp_departed: timeStampBeforeSeek,
                timestamp_target: player.getCurrentTime(),
                time_spent_seeking: (Date.now() - lastSeekTime) / 1000,
            });
            /* eslint-enable camelcase */
        }
        timeStampBeforeSeek = player.getCurrentTime();
    };

    var onSeeking = function() {
        if (!hasPlayed || isSeekInProgress) {
            return;
        }
        isSeekInProgress = true;
        lastSeekTime = Date.now();
    };

    var onTimeUpdate = function() {
        if (isSeekInProgress) {
            return;
        }

        // Update the timestamp so we can refer to it when the user seeks;
        // otherwise, there's no way to know what the timestamp was
        // before the user initiated a seek
        const currentTime = player.getCurrentTime();
        if (currentTime - timeStampBeforeSeek < MINIMUM_SEEK_TIME && timeStampBeforeSeek < currentTime) {
            timeStampBeforeSeek = currentTime;
        }
    };

    // Fired when the user hits the pause button.
    var onPause = function() {
        // TODO Use `stop` instead.
        minutesWatchedTimer.reset();
    };

    // Fired when the stream/video is over.
    var onEnded = function() {
        // TODO Use `stop` instead.
        minutesWatchedTimer.reset();

        // Don't fire buffer-refill if the stream goes offline
        bufferEmptyStartTime = null;
    };

    // Fired to signal if the current stream is live/playlist.
    var onSpectre = function(isSpectre) {
        if (player.getChannel()) {
            tracker.setProperties({
                content_mode: isSpectre ? 'playlist' : 'live', // eslint-disable-line camelcase
            });
        }
    };

    // Fired when the quality changes for whatever reason.
    var onQualityChange = function() {
        var quality = player.getQuality();

        // TODO Stop using chunked internally so we can remove this check.
        if (quality === 'chunked') {
            quality = 'source';
        }

        tracker.setProperty('quality', quality);
    };

    // Fired on Chromecast casting update
    var onCastingChange = function() {
        var casting = player.getCasting();

        if (casting === 'connected' || casting === 'connecting') {
            tracker.setProperties({
                player: 'chromecast',
                chromecast_sender: Settings.trackingPlatform, // eslint-disable-line camelcase
            });
        } else {
            tracker.setProperties({
                player: Settings.trackingPlatform,
                chromecast_sender: null, // eslint-disable-line camelcase
            });
        }
    };

    // Fired when a playback error has occured.
    var onVideoError = function(data) {
        tracker.trackEvent(VIDEO_ERROR, {
            error_type: data.error_type, // eslint-disable-line camelcase
        });
    };

    init();
}
