import { MixpanelClient } from './mixpanel';
import { SpadeClient } from './spade';
import * as Settings from '../settings';
import { parseUri } from '../util/parseuri';
import * as DeviceId from '../util/device-id';
import * as UniqueId from '../util/unique-id';
import * as flash from '../util/flash';

export function AnalyticsTracker(options) {
    const self = this;
    const promises = [];

    let mixpanelClient;
    let spadeClient;
    let globalProperties;

    var init = function() {
        mixpanelClient = new MixpanelClient({
            host: Settings.mixpanelHost,
            token: Settings.mixpanelToken,
        });

        spadeClient = new SpadeClient({
            host: Settings.spadeHost,
        });

        var fpv = flash.getFlashPlayerVersion();
        var flashVersion = `${fpv.major},${fpv.minor},${fpv.release}`;

        var embedLocation = getLocation();
        var embedReferrer = getReferrer();
        var embedReferrerURI = parseUri(embedReferrer);

        // false returns device_id, true returns session_device_id
        var deviceId = DeviceId.get(false);
        var sessionDeviceId = DeviceId.get(true);

        /* eslint-disable camelcase */
        // These tracking properties are added to every event.
        globalProperties = {
            platform: Settings.trackingPlatform,
            ui_version: Settings.version,
            flash_version: flashVersion,

            url: embedLocation.href,
            host: embedLocation.host,
            domain: topLevelHost(embedLocation.host),
            referrer_url: embedReferrer,
            referrer_host: embedReferrerURI.host,
            referrer_domain: topLevelHost(embedReferrerURI.host),
            browser: navigator.appVersion || '',

            device_id: deviceId,
            distinct_id: deviceId,
            session_device_id: sessionDeviceId,
            play_session_id: UniqueId.generate(),
        };
        /* eslint-enable camelcase */

        // Set the passed in tracking properties. (ex. player: frontpage)
        self.setProperties(options.tracking);

        // Ensure the benchmark ID is set by options or manual generation
        self.setProperty('benchmark_session_id', (options.benchmarkId) ? options.benchmarkId : UniqueId.generate());

        // Track whether or not this browser supports MSE playback
        self.setProperty('mse_support', (typeof MediaSource !== 'undefined'));
    };

    // Returns the last parts of the host name.
    // TODO Move this to a util file.
    // ex. player.twitch.tv -> twitch.tv
    var topLevelHost = function(host) {
        var parts = host.split('.');
        return parts.slice(-2).join('.');
    };

    var getLocation = function() {
        var loc;

        // For backwards compatibility, we need the parent's url.
        // This only works for twitch because of cross-origin constaints.
        try {
            document.domain = Settings.domain;
            loc = window.parent.location;

            // Try to access one of the properties. This will throw an
            // exception on Firefox/Chrome or return undefined on Safari.
            if (!loc.host) {
                // Failover to the current window.
                loc = window.location;
            }
        } catch (e) {
            // Failover to the current window.
            loc = window.location;
        }

        // Return the location.
        return loc;
    };

    var getReferrer = function() {
        // For backwards compatibility, we need the parent's url.
        // This only works for twitch because of cross-origin constaints.
        try {
            document.domain = Settings.domain;
            return window.parent.document.referrer;
        } catch (e) {
            return window.document.referrer;
        }
    };

    /**
     * Send multiple tracking events to the data tracking services.
     *
     * @param {Array<{event:String,properties:Object}>} events
     */
    self.trackEvents = function(events) {
        // Grab the time before waiting for our properties promises.
        var time = (new Date()).getTime() / 1000;

        // Block tracking until all of the current promises have returned.
        Promise.all(promises).then(function(asyncData) {
            // Combine all results from the promises into a single object.
            var asyncProperties = _.extend.apply(_, asyncData);

            // Loop over all of the events and add the global properties.
            events = events.map(function(e) {
                var properties = _.extend({}, e.properties, globalProperties, asyncProperties);
                properties.time = time;

                return {
                    event: e.event,
                    properties: properties,
                };
            });

            if (options.debug) {
                _.each(events, function(e) {
                    console.log('track event:', e.event, e.properties);
                });
            }

            mixpanelClient.trackEvents(events);
            spadeClient.trackEvents(events);
        });
    };

    /**
     * Send a single tracking event to the data tracking services.
     *
     * @param {String} name
     * @param {Object} properties
     */
    self.trackEvent = function(name, properties) {
        self.trackEvents([{
            event: name,
            properties: properties,
        }]);
    };

    self.setProperty = function(name, value) {
        var properties = {};
        properties[name] = value;

        self.setProperties(properties);
    };

    // Sets tracking properties part of every event.
    // If the argument is a promise, we block all tracking events from
    // being sent until the promise is resolved. This is useful when
    // tracking properties require an asychronous call which would otherwise
    // block playback.
    self.setProperties = function(arg) {
        // Convert the arg into a promise.
        // If it's an object, then we create an already resolved promise.
        // If it's a promise, then we pass through the promise.
        var promise = Promise.resolve(arg);

        // Create a wrapper promise that always resolves.
        // We treat rejections as warnings instead of fatal.
        promise = promise.then(null, function(err) {
            console.warn('failed to resolve properties promise', err);
            return {};
        });

        promises.push(promise);
    };

    init();
}
