import * as Settings from './settings';

// Detect if the browser supports cross-domain xhr requests.
// If it doesn't (IE9), we fall back to using jsonp instead.
const xhrSupported = !!('withCredentials' in new XMLHttpRequest());
const xhrDataType = xhrSupported ? 'json' : 'jsonp';

// Only allow alphanumeric characters in the path for security.
const cleanPath = function(path) {
    return path.replace(/[^A-Za-z0-9_]/g, '');
};

// TODO These results should expire after a certain amount of time.
const channelInfo = _.memoize(function(channel) {
    var request = $.ajax({
        url: `${Settings.apiHost}/kraken/channels/${cleanPath(channel)}`,
        dataType: xhrDataType,
        timeout: Settings.apiTimeout,
    });

    // Convert the jQuery Deferred to a native promise.
    return Promise.resolve(request);
});

const videoInfo = _.memoize(function(video) {
    var request = $.ajax({
        url: `${Settings.apiHost}/kraken/videos/${cleanPath(video)}`,
        dataType: xhrDataType,
        timeout: Settings.apiTimeout,
    });

    // We need the API request just for muted_segments
    // TODO Add it to kraken instead and stop making this request.
    var videoRequest = $.ajax({
        url: `${Settings.apiHost}/api/videos/${cleanPath(video)}`,
        dataType: 'jsonp', // TODO Add CORS support.
        timeout: Settings.apiTimeout,
    }).then(function(data) {
        return _.pick(data, 'muted_segments', 'increment_view_count_url', 'restrictions');
    });

    // Wait until both requests finish.
    var both = Promise.all([request, videoRequest]);

    // Return the combination of both requests.
    return both.then(function(requests) {
        return _.extend({}, requests[0], requests[1]);
    });
});

const productInfo = _.memoize(function(channel) {
    var request = $.ajax({
        url: `${Settings.apiHost}/api/channels/${cleanPath(channel)}/product`,
        dataType: xhrDataType,
        timeout: Settings.apiTimeout,
    });

    return Promise.resolve(request);
});

/**
 * Allocate stream info from Kraken.
 *
 * @params {String} channel
 */
const streamInfo = _.memoize(function(channel) {
    var request = $.ajax({
        url: `${Settings.apiHost}/kraken/streams/${cleanPath(channel)}`,
        dataType: xhrDataType,
        timeout: Settings.apiTimeout,
    });

    // Convert the jQuery Deferred to a native promise.
    return Promise.resolve(request);
});

const offlinePlaylistInfo = function(channelId) {
    var request = $.ajax({
        url: `${Settings.spectreHost}/v1/channels/${channelId}`,
        dataType: 'jsonp', // TODO Add CORS support to avoid JSONP.
        timeout: Settings.apiTimeout,
    });

    return Promise.resolve(request);
};

const channelViewerInfo = _.memoize(function(channel) {
    var request = $.ajax({
        url: `${Settings.apiHost}/api/channels/${cleanPath(channel)}/viewer`,
        xhrFields: {
            withCredentials: true,
        },
        dataType: xhrDataType,
        timeout: Settings.apiTimeout,
    });

    return Promise.resolve(request);
});

// TODO this function is currently unusable due to CORS/JSONP issues
// see https://twitchtv.atlassian.net/browse/RAILS-650
const userInfo = _.memoize(function() {
    return oauthToken().then(function() {
        var infoRequest = $.ajax({
            url: `${Settings.apiHost}/api/viewer/info.json`,
            xhrFields: {
                withCredentials: true,
            },
            dataType: xhrDataType,
            timeout: Settings.apiTimeout,
        });

        return Promise.resolve(infoRequest);
    });
});

// Returns the channel info from API.
const channelAPIInfo = _.memoize(function(channel) {
    return oauthToken().then(function() {
        var infoRequest = $.ajax({
            url: `${Settings.apiHost}/api/channels/${channel}`,
            dataType: xhrDataType,
            timeout: Settings.apiTimeout,
        });

        return Promise.resolve(infoRequest);
    });
});

// Return an oauth token for the current viewer.
const oauthToken = _.memoize(function() {
    var request = $.ajax({
        url: `${Settings.apiHost}/api/viewer/token.json`,
        xhrFields: {
            withCredentials: true,
        },
        dataType: xhrDataType,
        timeout: Settings.apiTimeout,
    });

    // Convert the jQuery Deferred to a native promise.
    return Promise.resolve(request);
});

// Returns access tokens required to get the stream url.
// NOTE: These tokens have a short expiration date.
const accessToken = function(type, media) {
    var requestAccess = function(token) {
        var url;

        if (type === 'channel') {
            url = `${Settings.apiHost}/api/channels/${cleanPath(media)}/access_token`;
        } else if (type === 'video') {
            url = `${Settings.apiHost}/api/vods/${cleanPath(media)}/access_token`;
        }

        var request = $.ajax({
            url: url,
            data: {
                oauth_token: token, // eslint-disable-line camelcase
            },
            xhrFields: {
                withCredentials: true,
            },
            dataType: xhrDataType,
            timeout: Settings.apiTimeout,
        });

        // Convert the jQuery Deferred to a native promise.
        return Promise.resolve(request);
    };

    // Try to request an oauth token, which will fail unless logged in.
    // In that case, we can still request an access token.
    // Some features like paywalls won't work unless logged in.
    return oauthToken().then(function(auth) {
        return requestAccess(auth.token);
    }, function() {
        return requestAccess();
    });
};

// Returns a promise for a m3u8 url for the given channel.
// This is asynchronous because we must request an access token.
const streamUrl = function(type, media, params) {
    var url;
    if (type === 'channel') {
        url = `${Settings.usherHost}/api/channel/hls/${cleanPath(media)}.m3u8`;
    } else if (type === 'video') {
        url = `${Settings.usherHost}/vod/${cleanPath(media)}.m3u8`;
    }

    var accessRequest = accessToken(type, media);
    return accessRequest.then(function(access) {
        // Make a copy of the parameters.
        params = _.extend({}, params);

        if (type === 'channel') {
            params.token = access.token;
            params.sig = access.sig;
        } else if (type === 'video') {
            params.nauth = access.token;
            params.nauthsig = access.sig;
        }

        // Add the parameters to the url.
        url += `?${$.param(params)}`;

        return url;
    });
};

// Returns a Boolean value representing if VOD content
// is restricted (ChanSubs) for this viewer.
const isVODRestricted = function(viewerInfo, videoInfo) {
    // If this viewer is a Channel Subscriber there is
    // no VOD playback restriction.
    if (viewerInfo.chansub !== null || viewerInfo.is_admin) {
        return false;
    }

    return _.reduce(videoInfo.restrictions, function(required, restriction) {
        return (required || (restriction === 'chansub'));
    }, false);
};

// Returns the URL to the channel page on Twitch.
const channelUrl = function(channel, params) {
    var url = `${Settings.twitchHost}/${cleanPath(channel)}`;
    if (params) {
        url += `?${$.param(params)}`;
    }

    return url;
};

// Returns the URL to the video page on Twitch.
const videoUrl = function(channel, video, params) {
    // Start with the channel URL, no params.
    var url = channelUrl(channel);

    // Split the video into parts.
    var videoType = video[0];
    var videoId = video.substring(1);

    // Add the remainder of the url.
    url += `/${cleanPath(videoType)}/${cleanPath(videoId)}`;

    if (params) {
        url += `?${$.param(params)}`;
    }

    return url;
};

export {
    channelInfo,
    videoInfo,
    productInfo,
    streamInfo,
    offlinePlaylistInfo,
    channelViewerInfo,
    userInfo,
    channelAPIInfo,
    oauthToken,
    accessToken,
    streamUrl,
    isVODRestricted,
    channelUrl,
    videoUrl,
};
