import EventEmitter from 'event-emitter';
import { Deferred } from '../util/deferred';
import { chrome } from 'exports?chrome=window.chrome!../../../vendor/cast_sender';
import * as Settings from '../settings';
import { streamUrl, channelInfo, offlinePlaylistInfo } from '../api';

// This entire backend is static because of how Chromecast works.
export var BackendChromecast = {};

(function() {
    const self = BackendChromecast;

    const events = new EventEmitter();
    const cache = {};

    let session;
    let media;
    let channel;
    let available;
    let error;
    let connecting;
    let networkState;
    let readyState;
    let initialized = false;

    function initExtension() {
        // This deferred object stores the success/failure state.
        var deferred = new Deferred();

        if (!window.chrome) {
            // Only supports Chrome.
            deferred.resolve(false);
        } else if (chrome.cast && chrome.cast.isAvailable) {
            // Chromecast was already initialized successfully, fire the callback.
            deferred.resolve(true);
        } else {
            // Register the callback called after cast_sender.js is loaded.
            window.__onGCastApiAvailable = function(loaded) {
                deferred.resolve(loaded);
            };

            // A fallback if this file is included after cast_sender.js
            // We have to wait for the extension to load, which takes time.
            // We check if the extension is loaded every second for 5 seconds.
            var attempts = 5;
            var retryLoop = setInterval(function() {
                var loaded = !!(chrome.cast && chrome.cast.isAvailable && false);
                if (loaded || attempts <= 0) {
                    deferred.resolve(loaded);
                    clearInterval(retryLoop);
                }

                attempts--;
            }, 1000);
        }

        return deferred.promise;
    }

    function initApi(loaded) {
        if (!loaded) {
            return;
        }

        var sessionRequest = new chrome.cast.SessionRequest(Settings.chromecastId);
        var apiConfig = new chrome.cast.ApiConfig(sessionRequest, onSession, onReceiver, chrome.cast.AutoJoinPolicy.ORIGIN_SCOPED);

        chrome.cast.initialize(apiConfig);
    }

    function initDefaults() {
        available = false;
        updateState();
    }

    // Called when the extension creates a session.
    // This will happen when the user clicks on the cast browser icon
    // or when the user navigates to the page while already casting
    function onSession(s) {
        session = s;
        session.addUpdateListener(onSessionUpdate);

        // Automatically start casting if we have a channel.
        if (channel) {
            loadChannel(channel);
        }
    }

    // Called when a session changes state, usually when killed.
    function onSessionUpdate() {
        if (session.status === chrome.cast.SessionStatus.CONNECTED) {
            var receiver = session.receiver;
            if (cache.volume !== receiver.volume.level) {
                cache.volume = receiver.volume.level;
                events.emit('volumechange');
            }

            if (cache.muted !== receiver.volume.muted) {
                cache.muted = receiver.volume.muted;
                events.emit('volumechange');
            }

            return;
        }

        if (session.status === chrome.cast.SessionStatus.STOPPED) {
            onMediaChange(null);

            session.removeUpdateListener(onSessionUpdate);
            session = null;

            connecting = false;
            updateState();

            return;
        }

        if (session.status === chrome.cast.SessionStatus.DISCONNECTED) {
            // TODO Anything special we need to do?
        }
    }

    // Called when a castable device is connected.
    // We don't know any specifics about the device though.
    function onReceiver(e) {
        available = (e === chrome.cast.ReceiverAvailability.AVAILABLE);
        updateState();
    }

    // Called when the media stream changes.
    // TODO: Figure out what happens when our tab is no longer cast.
    function onMediaChange(newmedia) {
        if (media) {
            media.removeUpdateListener(onMediaUpdate);
        }

        media = newmedia;
        connecting = false;
        updateState();

        if (media) {
            media.addUpdateListener(onMediaUpdate);
        }
    }

    // Called when the media changes for whatever reason.
    function onMediaUpdate() {
        if (cache.currentTime !== media.currentTime) {
            cache.currentTime = media.currentTime;
            events.emit('timeupdate');
        }

        if (cache.volume !== media.volume.level) {
            cache.volume = media.volume.level;
            events.emit('volumechange');
        }

        if (cache.muted !== media.volume.muted) {
            cache.muted = media.volume.muted;
            events.emit('volumechange');
        }

        if (cache.playbackRate !== media.playbackRate) {
            cache.playbackRate = media.playbackRate;
            events.emit('ratechange');
        }

        var state = media.playerState;
        if (cache.playerState !== media.playerState) {
            if (state === chrome.cast.media.PlayerState.PLAYING) {
                events.emit('playing');
            } else if (state === chrome.cast.media.PlayerState.PAUSED) {
                events.emit('pause');
            } else if (state === chrome.cast.media.PlayerState.BUFFERING) {
                // TODO?
            }

            cache.playerState = media.playerState;
        }
    }

    // Starts casting the given player.
    function loadChannel(id) {
        connecting = true;
        updateState();

        // Request the channel url and metadata.
        var urlRequest = streamUrl('channel', id, {
            allow_spectre: true, // eslint-disable-line camelcase
        });
        var infoRequest = channelInfo(id);

        Promise.all([urlRequest, infoRequest]).then(function(data) {
            var url = data[0];
            var info = data[1];

            offlinePlaylistInfo(info._id).then(function(spectreData) {
                var spectreInfo = spectreData || {};
                var mediaInfo = new chrome.cast.media.MediaInfo(url, 'application/x-mpegurl');
                mediaInfo.streamType = chrome.cast.media.StreamType.LIVE;

                var metadata = new chrome.cast.media.GenericMediaMetadata();
                metadata.subtitle = info.game;
                metadata.title = info.display_name;
                metadata.images = [new chrome.cast.Image(info.logo)];

                mediaInfo.metadata = metadata;

                // TODO TODO: Analytics?
                mediaInfo.customData = {
                    channel: id,
                    is_spectre: Boolean(spectreInfo.active), // eslint-disable-line camelcase
                        //analytics: info.analytics,
                };
                var request = new chrome.cast.media.LoadRequest(mediaInfo);
                request.autoplay = true;

                session.loadMedia(request, onMediaChange);
            });
        }, function(e) {
            // Clean up the session on error.
            session.stop();
            setError(e);
        });
    }

    function setError(newError) {
        if (newError) {
            error = newError;
            events.emit('error');
        }
    }

    // We use networkState and readyState to signal if we are connected.
    // It's a little hacky but compliant with the HTML5 video spec.
    function updateState() {
        var oldNetworkState = networkState;
        var oldReadyState = readyState;

        if (media) {
            // "connected"
            networkState = self.NETWORK_LOADING;
            readyState = self.HAVE_METADATA;

            // Once the Chromecast is connected, dispatch the event to switch
            // over to a new backend.
            events.emit('canplay');
        } else if (connecting) {
            // "connecting"
            networkState = self.NETWORK_LOADING;
            readyState = self.HAVE_NOTHING;
        } else if (available && channel) {
            // "available"
            networkState = self.NETWORK_IDLE;
            readyState = self.HAVE_NOTHING;
        } else {
            // "unavailable"
            networkState = self.NETWORK_EMPTY;
            readyState = self.HAVE_NOTHING;
        }

        // Fire any events needed because of the state change.
        if (networkState !== oldNetworkState) {
            if (networkState === self.NETWORK_LOADING) {
                events.emit('loadstart');
                events.emit('progress');
            } else if (networkState === self.NETWORK_IDLE) {
                events.emit('suspend');
            } else if (networkState === self.NETWORK_EMPTY) {
                events.emit('emptied');
            }
        }

        if (readyState !== oldReadyState) {
            if (readyState === self.HAVE_METADATA) {
                events.emit('loadedmetadata');
            } else if (readyState === self.HAVE_NOTHING) {
                events.emit('ended');
            }
        }
    }

    function onLaunchError(e) {
        setError(e);
    }

    self.init = function() {
        // Remove listeners in case old player is gone.
        events.removeAllListeners();

        if (initialized) return;
        initialized = true;
        initDefaults();

        var ready = initExtension();
        ready.then(initApi);
    };

    self.destroy = function() {
        // empty
    };

    self.addEventListener = function(name, callback) {
        events.on(name, callback);
    };

    self.removeEventListener = function(name, callback) {
        events.off(name, callback);
    };

    self.getNetworkProfile = function() {
        // Chromecast doesn't appear to expose any of the internal data required
        // to generate a network profile.
        return [];
    };

    self.getError = function() {
        return error;
    };

    self.getSrc = function() {
        // unimplemented
    };

    self.setSrc = function() {
        // unimplemented
    };

    self.getCurrentSrc = function() {
        // unimplemented
    };

    self.NETWORK_EMPTY = 0;
    self.NETWORK_IDLE = 1;
    self.NETWORK_LOADING = 2;
    self.NETWORK_NO_SOURCE = 3;

    self.getNetworkState = function() {
        return networkState;
    };

    self.getPreload = function() {
        // unimplemented
    };

    self.setPreload = function() {
        // unimplemented
    };

    self.load = function() {
        // Start a new session and try to autoplay.
        chrome.cast.requestSession(onSession, onLaunchError);
    };

    self.getBuffered = function() {
        // TODO Implement properly
        return {
            length: 0,
        };
    };

    self.HAVE_NOTHING = 0;
    self.HAVE_METADATA = 1;
    self.HAVE_CURRENT_DATA = 2;
    self.HAVE_FUTURE_DATA = 3;
    self.HAVE_ENOUGH_DATA = 4;

    self.getReadyState = function() {
        return readyState;
    };

    self.getSeeking = function() {
        // unimplemented
    };

    self.getCurrentTime = function() {
        if (media) {
            return media.getEstimatedTime();
        }
    };

    self.setCurrentTime = function(time) {
        if (media) {
            var seekRequest = new chrome.cast.media.SeekRequest();
            seekRequest.time = time;
            media.seek(seekRequest);
        }
    };

    self.getInitialTime = function() {
        // unimplemented
    };

    self.getDuration = function() {
        // unimplemented
    };

    self.getStartOffsetTime = function() {
        // unimplemented
    };

    self.getPaused = function() {
        if (media) {
            return media.playerState === chrome.cast.media.PlayerState.PAUSED;
        }
    };

    self.getDefaultPlaybackRate = function() {
        // unimplemented
    };

    self.setDefaultPlaybackRate = function() {
        // unimplemented
    };

    self.getPlaybackRate = function() {
        // unimplemented
    };

    self.setPlaybackRate = function() {
        // unimplemented
    };

    self.getPlayed = function() {
        // unimplemented
    };

    self.getSeekable = function() {
        // unimplemented
    };

    self.getEnded = function() {
        // unimplemented
    };

    self.getAutoplay = function() {
        // unimplemented
    };

    self.setAutoplay = function() {
        // unimplemented
    };

    self.getLoop = function() {
        // unimplemented
    };

    self.setLoop = function() {
        // unimplemented
    };

    self.play = function() {
        if (media) {
            var playRequest = new chrome.cast.media.PlayRequest();
            media.play(playRequest);
        } else {
            self.load();
        }
    };

    self.pause = function() {
        if (media) {
            var pauseRequest = new chrome.cast.media.PauseRequest();
            media.pause(pauseRequest);
        }
    };

    self.getControls = function() {
        // unimplemented
    };

    self.setControls = function() {
        // unimplemented
    };

    self.getVolume = function() {
        if (media) {
            return media.volume.level;
        }
    };

    self.setVolume = function(value) {
        if (media) {
            var volume = new chrome.cast.Volume(value, null);
            var volumeRequest = new chrome.cast.media.VolumeRequest(volume);
            media.setVolume(volumeRequest);
        }
    };

    self.getMuted = function() {
        if (media) {
            return media.volume.muted;
        }

        if (session) {
            return session.receiver.volume.muted;
        }
    };

    self.setMuted = function(value) {
        if (media) {
            var volume = new chrome.cast.Volume(null, value);
            var volumeRequest = new chrome.cast.media.VolumeRequest(volume);
            media.setVolume(volumeRequest);
        }
    };

    self.getDefaultMuted = function() {
        // unimplemented
    };

    self.setDefaultMuted = function() {
        // unimplemented
    };

    self.getQuality = function() {
        return 'auto';
    };

    self.setQuality = function() {
        // unimplemented
    };

    self.getQualities = function() {
        return ['auto'];
    };

    self.getChannel = function() {
        return channel;
    };

    self.setChannel = function(value) {
        channel = value;

        // Automatically load if we already have a session.
        if (session && channel) {
            loadChannel(channel);
        }
    };

    self.getVideo = function() {
        return null;
    };

    // TODO We currently don't support videos, so this unsets available.
    self.setVideo = function() {
        channel = null;
    };

    self.getStats = function() {
        // unimplemented
    };

    self.getStatsEnabled = function() {
        // unimplemented
    };

    self.setStatsEnabled = function() {
        // unimplemented
    };

    self.getVideoInfo = function() {
        // unimplemented
    };

    self.getCaption = function() {
        // unimplemented
    };

    self.getBackend = function() {
        return 'chromecast';
    };

    // Kills the current cast session.
    self.stop = function() {
        if (session) {
            session.stop();
        }
    };

    self.getDevice = function() {
        if (session && session.receiver) {
            return session.receiver.friendlyName;
        }
    };
})();
