import EventEmitter from 'event-emitter';
import { Deferred } from '../util/deferred';
import { ExtrapolatingTimer } from '../util/extrapolating-timer';
import * as flash from '../util/flash';
import * as Settings from '../settings';
import * as experiments from '../experiments';

// HTML5 Media Events
const EVENT_DURATION_CHANGE = 'durationchange';
const EVENT_SEEKING = 'seeking';
const EVENT_ENDED = 'ended';
const EVENT_ERROR = 'error';
const EVENT_VIDEO_PLAYING = 'playing';
const EVENT_WAITING = 'waiting';
const EVENT_VOLUME_CHANGE = 'volumechange';
const EVENT_VIDEO_LOADSTART = 'loadstart';

// Custom Video Events
const EVENT_PLAYER_INIT = 'init';
const EVENT_PLAYBACK_STATISTICS = 'playbackStatistics';
const EVENT_SPECTRE_PLAYLIST = 'spectrePlaylist';
const EVENT_CHANSUB_REQUIRED = 'chansubRequired';
const EVENT_STREAM_LOADED = 'streamLoaded';
const EVENT_VIDEO_LOADED = 'videoLoaded';
const EVENT_VIDEO_PAUSED = 'videoPaused';
const EVENT_VIDEO_FAILURE = 'videoFailure';
const EVENT_FORMATS = 'videoFormats';
const EVENT_FORMAT_CHANGED = 'videoFormatChanged';
const EVENT_TIME_CHANGE = 'timeChange';
const EVENT_BUFFER_CHANGE = 'bufferChange';
const EVENT_SEGMENT_CHANGE = 'segmentChange';
const EVENT_USHER_FAIL_ERROR = 'usherFail';
const EVENT_CAPTION_UPDATE = 'captions';

// Ad Events
const EVENT_AD_DISPLAY_STARTED = 'adDisplayStarted';
const EVENT_AD_DISPLAY_ENDED = 'adDisplayEnded';
const EVENT_AD_REQUEST = 'adRequested';
const EVENT_AD_REQUEST_DECLINED = 'adRequestDeclined';
const EVENT_AD_REQUEST_RESPONSE = 'adRequestResponse';
const EVENT_AD_REQUEST_ERROR = 'adRequestError';
const EVENT_AD_ERROR = 'adError';
const EVENT_COMPANION_RENDERED = 'adCompanionRendered';

const TIMEUPDATE_PERIOD = 250; //in milliseoconds

export function BackendFlash(analytics, options) {
    const self = this;

    // Assign a unique id for this instance.
    // Flash callbacks are static so we need to reference them based on id.
    const id = BackendFlash.counter++;
    BackendFlash.map[id] = self;

    // flashEvents are the incoming events from our Flash API.
    // videoEvents are the renamed events conforming to the html5 spec.
    const flashEvents = new EventEmitter();
    const videoEvents = new EventEmitter();

    const ready = new Deferred();

    const timer = new ExtrapolatingTimer();

    // HACK: Automatic Playback Statistics
    // updates from old flash backend.
    let isSendingStats = false;

    let initialized = false;
    let timeUpdateIntervalID = null;
    let readyState = 0;
    let networkState = 0;
    let channel = null;
    let video = null;
    let pendingChannel;
    let pendingVideo;
    let paused = true;
    let seeking;
    let duration = null;
    let volume;
    let unmuteVolume;
    let format;
    let formats;
    let caption;
    let stats = {
        bufferSize: 0,
        displayResolution: '',
        skippedFrames: 0,
        fps: 0,
        hlsLatencyBroadcast: 0,
        hlsLatencyEncoder: 0,
        memoryUsage: 0,
        playbackRate: 0,
        playerVolume: 0,
        videoResolution: '',
    };
    let buffer;
    let player;
    let error;
    let swfElementId = '';
    let ended = false;
    let adEnded = true;
    let statsEnabled = false;
    let autoPlay = false;

    // Cached spectre data.
    let isSpectre = false;

    // Playback State
    self.HAVE_NOTHING = 0;
    self.HAVE_METADATA = 1;
    self.HAVE_CURRENT_DATA = 2;
    self.HAVE_FUTURE_DATA = 3;
    self.HAVE_ENOUGH_DATA = 4;

    // Network Status
    self.NETWORK_EMPTY = 0;
    self.NETWORK_IDLE = 1;
    self.NETWORK_LOADING = 2;
    self.NETWORK_NO_SOURCE = 3;

    var init = function() {
        // Register our functions when the player starts.
        flashEvents.on(EVENT_PLAYER_INIT, onVideoInit);
        flashEvents.on(EVENT_VOLUME_CHANGE, onVideoVolumeChange);
        flashEvents.on(EVENT_DURATION_CHANGE, onVideoDurationChange);
        flashEvents.on(EVENT_FORMATS, onVideoFormats);
        flashEvents.on(EVENT_FORMAT_CHANGED, onVideoFormatChanged);
        flashEvents.on(EVENT_VIDEO_LOADSTART, onVideoLoadStart);
        flashEvents.on(EVENT_VIDEO_PLAYING, onVideoPlaying);
        flashEvents.on(EVENT_VIDEO_PAUSED, onVideoPaused);
        flashEvents.on(EVENT_VIDEO_FAILURE, onVideoFailure);
        flashEvents.on(EVENT_ENDED, onVideoEnded);
        flashEvents.on(EVENT_STREAM_LOADED, onVideoStreamLoaded);
        flashEvents.on(EVENT_VIDEO_LOADED, onVideoLoaded);
        flashEvents.on(EVENT_SEEKING, onVideoSeek);
        flashEvents.on(EVENT_TIME_CHANGE, onVideoTimeChange);
        flashEvents.on(EVENT_BUFFER_CHANGE, onVideoBufferChange);
        flashEvents.on(EVENT_WAITING, onVideoWaiting);
        flashEvents.on(EVENT_PLAYBACK_STATISTICS, onVideoStatistics);
        flashEvents.on(EVENT_CHANSUB_REQUIRED, onVideoChanSubRequired);
        flashEvents.on(EVENT_SEGMENT_CHANGE, onSegmentChange);
        flashEvents.on(EVENT_SPECTRE_PLAYLIST, onSpectrePlaylist);
        flashEvents.on(EVENT_ERROR, onVideoError);
        flashEvents.on(EVENT_USHER_FAIL_ERROR, onUsherFailError);
        flashEvents.on(EVENT_CAPTION_UPDATE, onCaptionUpdate);

        // Ad Events
        flashEvents.on(EVENT_AD_DISPLAY_STARTED, onVideoAdStarted);
        flashEvents.on(EVENT_AD_DISPLAY_ENDED, onVideoAdEnded);
        flashEvents.on(EVENT_COMPANION_RENDERED, onVideoAdCompanionRendered);
        flashEvents.on(EVENT_AD_REQUEST, onVideoAdRequest);
        flashEvents.on(EVENT_AD_REQUEST_DECLINED, onVideoAdRequestDeclined);
        flashEvents.on(EVENT_AD_REQUEST_RESPONSE, onVideoAdRequestResponse);
        flashEvents.on(EVENT_AD_REQUEST_ERROR, onVideoAdRequestError);
        flashEvents.on(EVENT_AD_ERROR, onVideoAdError);
    };

    /**
     * Called on video playback init.
     */
    var onVideoInit = function() {
        initialized = true;
        ready.resolve();
        videoEvents.emit('init');
    };

    /**
     * Called on caption event.
     *
     * @param {Object} data
     */
    var onCaptionUpdate = function(data) {
        caption = data;
        videoEvents.emit('captions');
    };

    /**
     * Called on video volume change.
     *
     * @param {Object} data
     */
    var onVideoVolumeChange = function(data) {
        // TODO Fix this on the Flash side.
        // Sometimes we get bad values like NaN.
        if (!_.isFinite(data.volume)) {
            return;
        }

        // Clamp the volume between 0 and 1
        volume = Math.min(Math.max(data.volume, 0), 1);
        videoEvents.emit('volumechange');
    };

    /**
     * Called on video duration change.
     *
     * @param {Object} data
     */
    var onVideoDurationChange = function(data) {
        duration = data.duration || null;
        videoEvents.emit('durationchange');
    };

    /**
     * Called on video formats list update.
     *
     * @param {Object} data
     */
    var onVideoFormats = function(data) {
        formats = data.formats;
        videoEvents.emit('qualitieschange');
    };

    /**
     * Called when the video format of the active video
     * stream has changed.
     *
     * @param {Object} data
     */
    var onVideoFormatChanged = function(data) {
        format = data.format;
        videoEvents.emit('qualitychange');
    };

    /**
     * The user agent begins looking for media data, as part
     * of the resource selection algorithm.
     *
     * @param {Object} data
     * @see http://www.w3.org/TR/html5/embedded-content-0.html#event-media-loadstart
     */
    var onVideoLoadStart = function(data) {
        // TODO Get the actual network status from Flash.
        if (data && data.format) {
            if (format !== data.format) {
                format = data.format;
                videoEvents.emit('qualitychange');
            }
        }

        networkState = self.NETWORK_LOADING;
        videoEvents.emit('loadstart');

        setTimerAndSendTimeUpdate(tryCall('getVideoTime'));
        pauseTimerAndPeriodicTimeUpdates();
    };

    /**
     * Called when the video stream has started or resumed playback.
     */
    var onVideoPlaying = function(data) {
        // Autoplay is supposed to keep paused true until video actually
        // starts playing.
        if (paused) {
            paused = false;
            videoEvents.emit('play');
        }

        // TODO: This is probably not needed, try to remove it.
        ended = false;

        // Set initial format selection
        format = data.format;

        setTimerAndSendTimeUpdate(tryCall('getVideoTime'));
        resumeTimerAndPeriodicTimeUpdates();

        if (seeking) {
            seeking = false;
            videoEvents.emit('seeked');
        }

        // streamLoaded always fires, so this is the first time we actually
        // know if a channel is online or not. videoLoaded fires normally,
        // so only fire this event if we haven't already.
        if (readyState < self.HAVE_METADATA) {
            videoEvents.emit('loadedmetadata');
        }

        // Always true but doesn't hurt to check.
        if (readyState < self.HAVE_CURRENT_DATA) {
            videoEvents.emit('loadeddata');
        }

        readyState = self.HAVE_FUTURE_DATA;
        videoEvents.emit('canplay');
        videoEvents.emit('playing');
    };

    /**
     * Called when the video stream has been paused.
     */
    var onVideoPaused = function() {
        paused = true;
        videoEvents.emit('pause');
        pauseTimerAndPeriodicTimeUpdates();
    };

    /**
     * Called on video playback failure.
     */
    var onVideoFailure = function() {
        videoEvents.emit('offline');

        if (!ended && readyState === self.HAVE_NOTHING) {
            ended = true;
            videoEvents.emit('ended');
        }

        isSpectre = false;

        ended = true;
        readyState = self.HAVE_NOTHING;
        pauseTimerAndPeriodicTimeUpdates();
    };

    /**
     * Called on video playback error.
     *
     * @param {Object} data
     */
    var onVideoError = function(data) {
        // TODO Support more possible error messages.
        if (pendingChannel) {
            error = Settings.channelError;
        } else if (pendingVideo) {
            error = Settings.videoError;
        } else {
            error = Settings.unknownError;
        }

        videoEvents.emit('error', data);
    };

    /**
     * Called when the stream or VOD has ended.
     */
    var onVideoEnded = function() {
        ended = true;
        isSpectre = false;
        paused = true;

        readyState = self.HAVE_NOTHING;
        videoEvents.emit('ended');

        pauseTimerAndPeriodicTimeUpdates();
    };

    /**
     * Called when we start trying to load a live stream.
     * Does NOT mean the channel is actually online.
     *
     * @param {Object} data
     */
    var onVideoStreamLoaded = function(data) {
        pauseTimerAndPeriodicTimeUpdates();
        setTimerAndSendTimeUpdate(0);

        // We don't know if the stream is online yet.
        channel = data.channel;
        video = null;
        videoEvents.emit('loadedchannel');

        //HACK TILL FLASH REFRESH
        if (isSendingStats) {
            deferCall('stopPlaybackStatistics');
        }

        isSendingStats = true;
        deferCall('startPlaybackStatistics');
    };

    /**
     * Called when a archive video has loaded.
     *
     * @param {Object} data
     */
    var onVideoLoaded = function(data) {
        pauseTimerAndPeriodicTimeUpdates();
        setTimerAndSendTimeUpdate(0);

        // Flash removes the "v", so we have to add it back.
        // ex. "12345" -> "v12345", but "c12345" -> "c12345"
        var videoId = data.videoId;
        if (!isNaN(videoId[0])) {
            videoId = `v${videoId}`;
        }

        video = videoId;
        channel = null;
        videoEvents.emit('loadedvideo');

        readyState = self.HAVE_METADATA;
        videoEvents.emit('loadedmetadata');

        //HACK TILL FLASH REFRESH
        if (isSendingStats) {
            deferCall('stopPlaybackStatistics');
        }

        isSendingStats = true;
        deferCall('startPlaybackStatistics');
    };

    /**
     * Called when the video stream has begun seeking
     * to a new offset in the stream.
     */
    var onVideoSeek = function() {
        pauseTimerAndPeriodicTimeUpdates();

        seeking = true;
        videoEvents.emit('seeking');

        // Flash automatically starts playing after seeking.
        paused = false;
        videoEvents.emit('play');
    };

    /**
     * Called when the video stream time has changed.
     *
     * @param {Object} data
     */
    var onVideoTimeChange = function() {
        // TODO Pass this value in as part of data.
        setTimerAndSendTimeUpdate(tryCall('getVideoTime'));
    };

    /**
     * Called when the video stream buffer has changed.
     *
     * @param {Object} data
     */
    var onVideoBufferChange = function(data) {
        buffer = data;
        videoEvents.emit('bufferedchange');
    };

    /**
     * Playback has stopped because the next frame is not available,
     * but the user agent expects that frame to become available in
     * due course.
     *
     * @param {Object} data
     * @see http://www.w3.org/TR/html5/embedded-content-0.html#event-media-waiting
     */
    var onVideoWaiting = function() {
        // readyState is equal to or less than HAVE_CURRENT_DATA, and paused is false.
        // Either seeking is true, or the current playback position is not contained
        // in any of the ranges in buffered. It is possible for playback to stop for
        // other reasons without paused being false, but those reasons do not fire this
        // event (and when those situations resolve, a separate playing event is not
        // fired either): e.g. the element is newly blocked on its media controller,
        // or playback ended, or playback stopped due to errors, or the element has
        // paused for user interaction or paused for in-band content.
        pauseTimerAndPeriodicTimeUpdates();

        readyState = self.HAVE_CURRENT_DATA;

        videoEvents.emit('waiting');
    };

    /**
     * Called when the video stream playback statistics have
     * been updated.
     *
     * @param {Object} data
     */
    var onVideoStatistics = function(data) {
        stats = data;
        videoEvents.emit('statschange');
    };

    /**
     * Called when the video stream playback statistics have
     * begun sending data to the UI on a set interval.
     *
     * @param {Object} data
     */
    var onVideoAdStarted = function(data) {
        adEnded = false;
        paused = false;

        // Pause time only for VOD
        if (video) {
            pauseTimerAndPeriodicTimeUpdates();
        }
        videoEvents.emit('adstart');

        analytics.trackEvent('video_ad_impression', data);
    };

    /**
     * Called when the video stream playback statistics have
     * stopped sending data to the UI on a set interval.
     *
     * @param {Object} data
     */
    var onVideoAdEnded = function(data) {
        adEnded = true;

        if (ended) {
            paused = true;
        }

        // Resume time only for VOD
        // Channel time would not have been paused
        if (video) {
            setTimerAndSendTimeUpdate(tryCall('getVideoTime'));
            resumeTimerAndPeriodicTimeUpdates();
        }
        videoEvents.emit('adend');

        analytics.trackEvent('video_ad_impression_complete', data);
    };

    /**
     * Called when a video ad has been requested.
     *
     * @param {Object} data
     */
    var onVideoAdRequest = function(data) {
        analytics.trackEvent('video_ad_request', data);
    };

    /**
     * Called when a video ad request has been declined.
     *
     * @param {Object} data
     */
    var onVideoAdRequestDeclined = function(data) {
        analytics.trackEvent('video_ad_request_declined', data);
    };

    /**
     * Called when a video ad request has been responded to.
     *
     * @param {Object} data
     */
    var onVideoAdRequestResponse = function(data) {
        analytics.trackEvent('video_ad_request_response', data);
    };

    /**
     * Called when a video ad request has encountered a error.
     *
     * @param {Object} data
     */
    var onVideoAdRequestError = function(data) {
        analytics.trackEvent('video_ad_error', data);
    };

    /**
     * Called when a video ad encountered a error.
     *
     * @param {Object} data
     */
    var onVideoAdError = function(data) {
        analytics.trackEvent('video_ad_error', data);
    };

    /**
     * Called on video stream advertisment companion rendered.
     *
     * @param {Object} data
     */
    var onVideoAdCompanionRendered = function(data) {
        // TODO Remove the data payload from this event.
        videoEvents.emit('adcompanionrendered', data);
    };

    /**
     * Called on video stream Channel Subscription required.
     */
    var onVideoChanSubRequired = function() {
        videoEvents.emit('restricted');
    };

    /**
     * Called on video stream Channel Subscription required.
     */
    var onUsherFailError = function() {
        videoEvents.emit('usherfail');
    };

    /**
     * Called on every video segment change.
     *
     * @param {Object} data
     */
    var onSegmentChange = function(data) {
        videoEvents.emit('segmentchange', data.name);
    };

    /**
     * Called when Spectre data has been acquired for the channel.
     *
     * @param {Object} data
     */
    var onSpectrePlaylist = function(data) {
        isSpectre = data.is_spectre;
        videoEvents.emit('isspectre', data.is_spectre);
    };

    /**
     * Emits a new timeupdate VideoEvent.
     */
    var emitTimeUpdateEvent = function() {
        videoEvents.emit('timeupdate', timer.extrapolateTimeStamp());
    };

    /**
     * Called when current playback position changes in interesting way.
     *
     * @param {String} time
     */
    var setTimerAndSendTimeUpdate = function(time) {
        timer.setCurrentTimeStamp(time);
        videoEvents.emit('timeupdate', timer.extrapolateTimeStamp());
    };

    /**
     * Used to update embed client. And implements html5 spec.
     */
    var resumeTimerAndPeriodicTimeUpdates = function() {
        if (timeUpdateIntervalID === null) {
            timeUpdateIntervalID = setInterval(emitTimeUpdateEvent, TIMEUPDATE_PERIOD);
            timer.resume();
        }
    };

    /**
     * Used to update embed client. And implements html 5 spec.
     * Called when playback position is paused.
     */
    var pauseTimerAndPeriodicTimeUpdates = function() {
        if (timeUpdateIntervalID !== null) {
            clearInterval(timeUpdateIntervalID);
            timeUpdateIntervalID = null;
            timer.pause();
        }
    };

    /**
     * Attaches a new playback container to the DOM.
     *
     * @param {Element} element
     */
    self.attach = function(element) {
        // Player Defaults
        var url = require('file?name=vendor/[name].[hash].[ext]!../../../vendor/TwitchPlayer.swf');
        url = `${Settings.playerHost}/${url}`; // Fixes inline embeds (web-client)
        var swfVersionStr = '10.2';
        var xiSwfUrlStr = 'playerProductInstall.swf';
        var width = '100%';
        var height = '100%';

        var flashvars = {};
        flashvars.eventsCallback = 'window._BackendFlash_emitEvents';
        flashvars.eventsContext = id;
        flashvars.initCallback = null;

        var params = {};
        params.bgcolor = '#000';
        params.allowscriptaccess = 'always';
        params.allowfullscreen = 'true';

        // We use wmode direct in Chrome/Pepper for best performance.
        // This mode will actually cover the UI in other browsers.
        if (flash.getFlashPlayerType() === 'ppapi') {
            params.wmode = 'direct';
        } else {
            params.wmode = 'transparent';
        }

        var attributes = {};
        attributes.align = 'middle';

        // Create a div for swfobject to replace.
        var temp = document.createElement('div');
        element.appendChild(temp);

        // Aassign a unique id so embedSWF can find the div.
        swfElementId = `swfobject-${id}`;
        temp.setAttribute('id', swfElementId);

        flash.embedSWF(url, swfElementId, width, height, swfVersionStr, xiSwfUrlStr, flashvars, params, attributes, function(e) {
            player = e.ref;
        });

        Promise.all([options.experiments.get(experiments.ABS_V1), ready.promise]).then(function(data) {
            var absV1Enabled = (data[0] === 'yes');
            tryCall('setABSV1Enabled', [absV1Enabled]);
        });

        Promise.all([options.experiments.get(experiments.PREROLL_ADS), ready.promise]).then(data => {
            var prerollsActive = (data[0] === 'yes');
            tryCall('setPrerollsActive', [prerollsActive]);
        });

        deferCall('setLastAdDisplay', [options.lastAdDisplay]);
        deferCall('setPlayerType', [options.player]);
    };

    /**
     * Calls an external interface Flash method with the given name and arguments.
     * Calling .apply doesn't work for Debug Flash FOR WHATEVER REASON.
     *
     * @param {String} name
     * @param {Array=} args
     */
    var flashCall = function(name, args) {
        args = args || [];

        if (options.debug) {
            console.log('flash call:', name, args);
        }

        // TODO UGH, this is a huge hack.
        switch (args.length) {
        case 0:
            return player[name]();
        case 1:
            return player[name](args[0]);
        case 2:
            return player[name](args[0], args[1]);
        default:
                // This will break for the Flash debugger
            console.log('WARNING, too many arguments passed to Flash');
            return player[name].apply(this, args);
        }
    };

    /**
     * Queues the given function call until the flash player has loaded.
     * Use tryCall instead if you need a blocking call (ie. return value).
     *
     * @param {String} func
     * @param {Array=} args
     */
    var deferCall = function(func, args) {
        // Queue the function call until the player has loaded.
        ready.promise.then(function() {
            // Don't block on the Flash call; it might take a while.
            setTimeout(function() {
                var result = flashCall(func, args);

                if (result && options.debug) {
                    console.log('flash return:', func, '=', result);
                }
            }, 0);
        });
    };

    /**
     * Call a function immediately iff the player has loaded.
     * If the player has not loaded yet, nothing happens.
     *
     * @param {String} func
     * @param {Array=} args
     */
    var tryCall = function(func, args) {
        if (initialized) {
            var result = flashCall(func, args);
            if (result && options.debug) {
                console.log('flash return:', func, '=', result);
            }

            return result;
        }
    };

    /**
     * Must be public and visible to _BackendFlash_emitEvents.
     *
     * @param {String} name
     * @param {Object} data
     */
    self._emitEvent = function(name, data) {
        // Defer the event in order to fix exception reporting.
        // No idea why this is still needed but it's a pain without it.
        setTimeout(function() {
            if (options.debug && name !== 'timeupdate' && name !== 'playbackStatistics') {
                console.log('flash event:', name, data);
            }

            flashEvents.emit(name, data);
        }, 0);
    };

    /**
     * Creates a new VideoEvents listener.
     *
     * @param {String} name
     * @param {function} callback
     */
    self.addEventListener = function(name, callback) {
        videoEvents.on(name, callback);
    };

    /**
     * Removes a VideoEvents listener.
     *
     * @param {String} name
     * @param {function} callback
     */
    self.removeEventListener = function(name, callback) {
        videoEvents.off(name, callback);
    };

    /**
     * Returns the current client network profile.
     */
    self.getNetworkProfile = function() {
        return tryCall('getNetworkProfile') || [];
    };

    /**
     * Returns the current playback error.
     */
    self.getError = function() {
        return error;
    };

    /**
     *
     */
    self.getNetworkState = function() {
        // One of the values defined above.
        // TODO Expose the actual network status from Flash.
        return networkState;
    };

    /**
     * Returns the current video buffer information.
     */
    self.getBuffered = function() {
        // Flash only returns a buffer of length 0 or 1.
        var length = (buffer ? 1 : 0);
        return {
            length: length,
            start(i) {
                if (i >= length) return;
                return buffer.start;
            },
            end(i) {
                if (i >= length) return;
                return buffer.end;
            },
        };
    };

    /**
     * Attempts to load the pending channel or video content stream
     * into the video playback container.
     */
    self.load = function() {
        if (!pendingChannel && !pendingVideo) {
            return;
        }

        readyState = self.HAVE_NOTHING;
        networkState = self.NETWORK_LOADING;
        videoEvents.emit('loadstart');

        if (pendingChannel) {
            deferCall('loadChannel', [pendingChannel]);
        } else if (pendingVideo) {
            deferCall('loadVideo', [pendingVideo]);
        }
    };

    /**
     * Returns the ready state.
     */
    self.getReadyState = function() {
        return readyState;
    };

    /**
     * Returns the seeking state.
     */
    self.getSeeking = function() {
        return seeking;
    };

    /**
     * Returns the current playback time of the content stream.
     */
    self.getCurrentTime = function() {
        return timer.extrapolateTimeStamp();
    };

    /**
     * Sets the current playback time in the content stream.
     *
     * @param {Number} time
     */
    self.setCurrentTime = function(time) {
        ended = false;

        // Most calls into the flash backend do not affect the timer
        // This one does because seeking requires us to know where
        // the scrubbing position wants to play at.
        pauseTimerAndPeriodicTimeUpdates();
        setTimerAndSendTimeUpdate(time);

        deferCall('videoSeek', [time]);
    };

    /**
     * Returns the duration of the content stream.
     */
    self.getDuration = function() {
        return duration;
    };

    /**
     * Returns the paused state.
     */
    self.getPaused = function() {
        return paused;
    };

    /**
     * Returns if the endlist has been located in the content stream.
     * Ended events are fired from EXT-X-ENDLIST tags in the HLS manifest.
     */
    self.getEnded = function() {
        if (!adEnded) {
            return adEnded;
        }

        return ended;
    };

    /**
     * Returns the autoplay state.
     */
    self.getAutoplay = function() {
        return autoPlay;
    };

    /**
     * Sets the autoplay state.
     *
     * @param {Boolean} value
     */
    self.setAutoplay = function(value) {
        autoPlay = value;
    };

    /**
     * Begin or resume playback of the loaded content stream.
     */
    self.play = function() {
        if (networkState === self.NETWORK_EMPTY) {
            // If we haven't loaded the content yet, load it.
            self.load();
        } else if (ended && adEnded && video) {
            // Restart the stream if ended; part of the spec.
            self.setCurrentTime(0);
        }

        paused = false;
        videoEvents.emit('play');

        deferCall('playVideo', null);
    };

    /**
     * Pauses playback of the content stream.
     */
    self.pause = function() {
        deferCall('pauseVideo');
    };

    /**
     * Returns the playback volume.
     */
    self.getVolume = function() {
        return volume;
    };

    /**
     * Sets the playback volume.
     *
     * @param {Number} value
     */
    self.setVolume = function(value) {
        if (value > 0) {
            unmuteVolume = value;
        }

        deferCall('setVolume', [value]);
    };

    /**
     * Returns the muted state.
     */
    self.getMuted = function() {
        var volume = self.getVolume();
        return volume === 0;
    };

    /**
     * Sets the volume muted state.
     *
     * @param {Boolean} muted
     */
    self.setMuted = function(muted) {
        var volume = muted ? 0 : (unmuteVolume || 0.5);
        self.setVolume(volume);
    };

    /**
     * Attempts to retrieve the playback video info objact from the Flash Backend.
     */
    self.getVideoInfo = function() {
        return tryCall('getVideoInfo') || {};
    };

    /**
     * Returns the current Twitch channel in playback.
     */
    self.getChannel = function() {
        return pendingChannel || channel;
    };

    /**
     * Sets a new Twitch channel for playback.
     *
     * @param {String} value
     */
    self.setChannel = function(value) {
        pendingChannel = value;
        pendingVideo = null;

        // Instantly load the new channel if autoplay is enabled or we
        // are currently playing another channel.
        if (autoPlay || networkState > self.NETWORK_EMPTY) {
            self.load();
        }
    };

    /**
     * Returns the current video ID.
     */
    self.getVideo = function() {
        return pendingVideo || video;
    };

    /**
     * Sets the Twitch VOD ID for playback
     *
     * @param {String} value
     */
    self.setVideo = function(value) {
        pendingVideo = value;
        pendingChannel = null;

        // Instantly load the new video if autoplay is enabled or we
        // are currently playing another video.
        if (autoPlay || networkState > self.NETWORK_EMPTY) {
            self.load();
        }
    };

    /**
     * Returns the current playback quality.
     */
    self.getQuality = function() {
        return format || '';
    };

    /**
     * Attempts to set a new preferred playback quality.
     *
     * @param {String} quality
     */
    self.setQuality = function(quality) {
        format = quality;
        deferCall('setQuality', [quality]);
    };

    /**
     * Returns the current list of playback qualities available
     * for the current content stream.
     */
    self.getQualities = function() {
        return formats || [];
    };

    /**
     * Returns the state of playback statistics output.
     */
    self.getStatsEnabled = function() {
        return statsEnabled;
    };

    /**
     * Attempts to turn on playback statistic output from the backend.
     *
     * @param {Boolean} value
     */
    self.setStatsEnabled = function(value) {
        statsEnabled = value;

        if (statsEnabled) {
            deferCall('startPlaybackStatistics');
        } else {
            deferCall('stopPlaybackStatistics');
        }
    };

    /**
     * Returns the most recent playback stats object.
     */
    self.getStats = function() {
        return stats;
    };

    /**
     * Returns the current Backend identifier.
     */
    self.getBackend = function() {
        return 'flash';
    };

    /**
     * Returns the current Flash Backend version.
     */
    self.getVersion = function() {
        return tryCall('getVersion');
    };

    /**
     * Returns if the current playack stream is Spectre content.
     */
    self.isSpectre = function() {
        return isSpectre;
    };

    /**
     * The mediagroup content attribute on media elements can be used to link multiple
     * media elements together by implicitly creating a MediaController. The value is
     * text; media elements with the same value are automatically linked by the user agent.
     */
    self.getMediaGroup = function() {
        return null;
    };

    /**
     * The controller attribute on a media element, on getting, must return the element's
     * current media controller, if any, or null otherwise.
     */
    self.getController = function() {
        return null;
    };

    /**
     *
     */
    self.getControls = function() {
        return false;
    };

    /**
     *
     */
    self.getLoop = function() {
        return false;
    };

    /**
     * The defaultMuted IDL attribute must reflect the muted content attribute.
     */
    self.getDefaultMuted = function() {
        return false;
    };

    /**
     * Returns an AudioTrackList object representing the audio tracks
     * available in the media resource.
     */
    self.getAudioTracks = function() {
        return null;
    };

    /**
     * Returns a VideoTrackList object representing the video
     * tracks available in the media resource.
     */
    self.getVideoTracks = function() {
        return null;
    };

    /**
     * The textTracks attribute of media elements must return a TextTrackList object
     * representing the TextTrack objects of the text tracks in the media element's list
     * of text tracks, in the same order as in the list of text tracks.
     */
    self.getTextTracks = function() {
        return null;
    };

    /**
     *
     */
    self.getDefaultPlaybackRate = function() {
        return 1;
    };

    /**
     *
     */
    self.getPlaybackRate = function() {
        return 1;
    };

    /**
     *
     */
    self.getInitialTime = function() {
        return 0;
    };

    /**
     *
     */
    self.getPreload = function() {
        return 'none';
    };

    /**
     * Creates and returns a new MutableTextTrack object, which is also added to the
     * media element's list of text tracks.
     */
    self.addTextTrack = function() {};

    /**
     * The defaultMuted IDL attribute must reflect the muted content attribute.
     */
    self.setDefaultMuted = function() {};

    /**
     *
     */
    self.setControls = function() {};

    /**
     * The loop attribute is a boolean attribute that, if specified, indicates that
     * the media element is to seek back to the start of the media resource upon reaching the end.
     */
    self.setLoop = function() {};

    /**
     * The playbackRate attribute gives the effective playback rate (assuming there is no
     * current media controller overriding it), which is the speed at which the media resource
     * plays, as a multiple of its intrinsic speed.
     */
    self.setPlaybackRate = function() {};

    /**
     * The played attribute must return a new static normalised TimeRanges object that
     * represents the ranges of points on the media timeline of the media resource reached through
     * the usual monotonic increase of the current playback position during normal playback, if
     * any, at the time the attribute is evaluated.
     */
    self.getPlayed = function() {};

    /**
     * The seekable attribute must return a new static normalised TimeRanges object that
     * represents the ranges of the media resource, if any, that the user agent is able to seek to,
     * at the time the attribute is evaluated.
     */
    self.getSeekable = function() {};

    /**
     * The defaultPlaybackRate attribute gives the desired speed at which the media resource
     * is to play, as a multiple of its intrinsic speed.
     */
    self.setDefaultPlaybackRate = function() {};

    /**
     * The startOffsetTime attribute must return a new Date object representing the
     * current timeline offset.
     */
    self.getStartOffsetTime = function() {};

    /**
     * Possible values: none, metadata, auto
     */
    self.setPreload = function() {};

    /**
     * The src content attribute on media elements gives the address
     * of the media resource (video, audio) to show.
     */
    self.getSrc = function() {};

    /**
     * The src content attribute on media elements gives the address of
     * the media resource (video, audio) to show.
     */
    self.setSrc = function() {};

    /**
     * Returns the address of the current media resource, if any.
     */
    self.getCurrentSrc = function() {};

    /**
     * Returns caption data from stream, if any.
     */
    self.getCaption = function() {
        return caption;
    };

    /**
     * Destroy the Flash Backend.
     */
    self.destroy = function() {
        // Stop intervals and timers
        pauseTimerAndPeriodicTimeUpdates();

        // Remove the SWF
        flash.removeSWF(swfElementId);

        player = null;
    };

    init();
}

// We need some static members to uniquely identify objects.
BackendFlash.map = {};
BackendFlash.counter = 0;

/**
 * Verify that the client is running a > Flash Player 10.2
 */
BackendFlash.canPlay = function() {
    return flash.hasFlashPlayerVersion('10.2');
};

/**
 * Flash fires this static callback on events.
 * ugh, this needs to be public so Flash can discover it.
 */
window._BackendFlash_emitEvents = function(context, es) { // eslint-disable-line camelcase
    // Map from the id to the BackendFlash object.
    var self = BackendFlash.map[context];

    // Loop over all of the events and fire them sequentially.
    for (var i = 0; i < es.length; i++) {
        var e = es[i];
        self._emitEvent(e.event, e.data);
    }
};
