import EventEmitter from 'event-emitter';
import { streamUrl } from '../api';

const EVENT_LOADED_METADATA = 'loadedmetadata';
const EVENT_ERROR = 'error';

export function BackendHls() {
    const self = this;

    const events = new EventEmitter();
    const videoElement = document.createElement('video');

    let channel;
    let video;
    let lastBuffer;

    // TODO Start calling on playing.
    var pollBuffer = function() {
        var buffer = self.getBuffered();
        if (buffer !== lastBuffer) {
            lastBuffer = buffer;
            events.emit('bufferchange');
        }

        setTimeout(pollBuffer, 1000);
    };

    self.attach = function(element) {
        $(element).append(videoElement);
        events.emit('init');

        videoElement.addEventListener(EVENT_ERROR, function() {
            if (videoElement.error === null) {
                // unsure if this can happen, but just in case...
                return;
            }

            // if the error from the video element is MEDIA_ERR_SRC_NOT_SUPPORTED,
            // then what most likely happened is that the master manifest 404'd,
            // which we should hide from downstream consumers and instead proxy
            // out as `ended`.
            if (videoElement.error.code === videoElement.error.MEDIA_ERR_SRC_NOT_SUPPORTED) {
                events.emit('offline');
                events.emit('ended');
            } else {
                events.emit(EVENT_ERROR);
            }
        });

        videoElement.addEventListener(EVENT_LOADED_METADATA, function() {
            events.emit('isspectre', self.isSpectre());
        });
    };

    self.destroy = function() {
        // empty
    };

    self.addEventListener = function(name, callback) {
        // Register the event on the video AND our custom events class.
        // This lets us provide a superset of the events.
        if (name !== EVENT_ERROR) {
            videoElement.addEventListener(name, callback);
        }
        events.on(name, callback);
    };

    self.removeEventListener = function(name, callback) {
        videoElement.removeEventListener(name, callback);
        events.off(name, callback);
    };

    self.getNetworkProfile = function() {
        // TODO implement network profile collection for this backend
        return [];
    };

    self.getError = function() {
        // if the error is MEDIA_ERR_SRC_NOT_SUPPORTED, then most likely the master
        // manifest 404'd, and that means the stream is simply not online. Hide
        // this from consumers.
        if (
            videoElement.error !== null &&
            videoElement.error.code !== videoElement.error.MEDIA_ERR_SRC_NOT_SUPPORTED
        ) {
            return videoElement.error;
        }

        return null;
    };

    self.getSrc = function() {
        return videoElement.src;
    };

    self.setSrc = function(value) {
        videoElement.src = value;
    };

    self.getCurrentSrc = function() {
        return videoElement.currentSrc;
    };

    self.NETWORK_EMPTY = videoElement.NETWORK_EMPTY;
    self.NETWORK_IDLE = videoElement.NETWORK_IDLE;
    self.NETWORK_LOADING = videoElement.NETWORK_LOADING;
    self.NETWORK_NO_SOURCE = videoElement.NETWORK_NO_SOURCE;

    self.getNetworkState = function() {
        return videoElement.networkState;
    };

    self.getPreload = function() {
        return videoElement.preload;
    };

    self.setPreload = function(value) {
        videoElement.preload = value;
    };

    self.getBuffered = function() {
        return videoElement.buffered;
    };

    self.load = function() {
        videoElement.load();
    };

    self.HAVE_NOTHING = videoElement.HAVE_NOTHING;
    self.HAVE_METADATA = videoElement.HAVE_METADATA;
    self.HAVE_CURRENT_DATA = videoElement.HAVE_CURRENT_DATA;
    self.HAVE_FUTURE_DATA = videoElement.HAVE_FUTURE_DATA;
    self.HAVE_ENOUGH_DATA = videoElement.HAVE_ENOUGH_DATA;

    self.getReadyState = function() {
        return videoElement.readyState;
    };

    self.getSeeking = function() {
        return videoElement.seeking;
    };

    self.getCurrentTime = function() {
        return videoElement.currentTime;
    };

    self.setCurrentTime = function(value) {
        videoElement.currentTime = value;
    };

    self.getInitialTime = function() {
        return videoElement.initialTime;
    };

    self.getDuration = function() {
        return videoElement.duration;
    };

    self.getStartOffsetTime = function() {
        return videoElement.startOffsetTime;
    };

    self.getPaused = function() {
        return videoElement.paused;
    };

    self.getDefaultPlaybackRate = function() {
        return videoElement.defaultPlaybackRate;
    };

    self.setDefaultPlaybackRate = function(value) {
        videoElement.defaultPlaybackRate = value;
    };

    self.getPlaybackRate = function() {
        return videoElement.playbackRate;
    };

    self.setPlaybackRate = function(value) {
        videoElement.playbackRate = value;
    };

    self.getPlayed = function() {
        return videoElement.played;
    };

    self.getSeekable = function() {
        return videoElement.seekable;
    };

    self.getEnded = function() {
        return (
            videoElement.ended ||
            // If the video element has an error and it is MEDIA_ERR_SRC_NOT_SUPPORTED,
            // then the most likely cause is that the master manifest 404'ed,
            // which we should communicate to downstream users as `ended`
            (
                videoElement.error !== null &&
                videoElement.error.code === videoElement.error.MEDIA_ERR_SRC_NOT_SUPPORTED
            )
        );
    };

    self.getAutoplay = function() {
        return videoElement.autoplay;
    };

    self.setAutoplay = function(value) {
        videoElement.autoplay = value;
    };

    self.getLoop = function() {
        return videoElement.loop;
    };

    self.setLoop = function(value) {
        videoElement.loop = value;
    };

    self.play = function() {
        videoElement.play();

        // TODO remove hacks on hacks; we should be setting autoplay at a higher
        // level, but for now, we are working around workarounds that make setting
        // autoplay a problem.
        if (videoElement.src === '') {
            videoElement.autoplay = true;
        }
    };

    self.pause = function() {
        videoElement.pause();
    };

    // TODO mediaGroup, controller, audioTracks, videoTracks, textTracks

    self.getControls = function() {
        return false;
    };

    self.setControls = function() {
        // nothing
    };

    self.getVolume = function() {
        return videoElement.volume;
    };

    self.setVolume = function(value) {
        videoElement.volume = value;
    };

    self.getMuted = function() {
        return videoElement.muted;
    };

    self.setMuted = function(value) {
        videoElement.muted = value;
    };

    self.getDefaultMuted = function() {
        return videoElement.defaultMuted;
    };

    self.setDefaultMuted = function(value) {
        videoElement.defaultMuted = value;
    };

    // Custom stuff
    self.getChannel = function() {
        return channel;
    };

    self.setChannel = function(value) {
        var urlRequest = streamUrl('channel', value, {
            allow_source: true, // eslint-disable-line camelcase
        });

        urlRequest.then(function(url) {
            self.setSrc(url);

            channel = value;
            video = null;
            events.emit('loadedchannel');
        });
    };

    self.getVideo = function() {
        return video;
    };

    self.setVideo = function(value) {
        var urlRequest = streamUrl('video', value.substring(1), {
            allow_source: true, // eslint-disable-line camelcase
        });

        urlRequest.then(function(url) {
            self.setSrc(url);

            video = value;
            channel = null;
            events.emit('loadedvideo');
        });
    };

    // TODO Properly implement quality switching.
    self.getQuality = function() {
        return 'auto';
    };

    self.setQuality = function() {
        // TODO
    };

    self.getQualities = function() {
        return ['auto'];
    };

    self.getStats = function() {
        return {
            bufferSize: 0,
            displayResolution: '',
            skippedFrames: 0,
            fps: 0,
            hlsLatencyBroadcast: 0,
            hlsLatencyEncoder: 0,
            memoryUsage: 0,
            playbackRate: 0,
            playerVolume: 0,
            videoResolution: '',
        };
    };

    self.getStatsEnabled = function() {
        // unimplemented
    };

    self.setStatsEnabled = function() {
        // unimplemented
    };

    self.getVideoInfo = function() {
        // unimplemented
    };

    self.getCaption = function() {
        // unimplemented
    };

    self.getBackend = function() {
        return 'hls_fallback';
    };

    self.getVersion = function() {
        // TODO
    };

    self.isSpectre = function() {
        // TODO
        return false;
    };
}

BackendHls.canPlay = function() {
    // Create a temporary video element for the check.
    const videoElement = document.createElement('video');

    if (_.isFunction(videoElement.canPlayType)) {
        return videoElement.canPlayType('application/vnd.apple.mpegURL');
    }

    return false;
};
