import EventEmitter from 'event-emitter';
import * as swfobject from '../util/flash';
import { Deferred } from '../util/deferred';
import * as Settings from '../settings';

export function BackendIMAFlash(root, options) {
    const self = this;

    // This backend communicates with a simple SWF to provide video ads.
    // It is special because we only attach the SWF when we want to play
    // ads, and remove it when a block of ads has finished. This means we
    // must cache all backend methods (ex. setVolume) and reapply them
    // for each new ad request.

    // The interface with the SWF is rather simple; it emulates the html5
    // video spec but over ExternalInterface. To request an ad, set the src
    // to the playlist url and all events/methods will work like normal.
    // To remove the ad, set the src to empty string.

    const id = BackendIMAFlash.counter++;
    BackendIMAFlash.map[id] = self;

    const events = new EventEmitter();

    // Keep track of the Flash initialization state.
    let ready;
    let player;
    let initialized;

    // Properties that are set and don't depend on the current state.
    // We initialize each new SWF with these values.
    let src;
    let preload;
    let autoplay;
    let loop;
    let controls;
    let defaultPlaybackRate;
    let playbackRate;
    let volume;
    let muted;
    let defaultMuted;
    let channel;
    let video;

    // Properties that are dependent on the current ad playing.
    let error;

    var init = function() {
        events.on('init', function() {
            initialized = true;
            ready.resolve();
        });

        events.on('error', function() {
            error = tryCall('getError');
        });

        initPlayer();
    };

    var initPlayer = function() {
        // Create a new promise that will fire when the player is ready.
        ready = new Deferred();
        initialized = false;
        player = null;

        ready.promise.then(null, function(reason) {
            error = reason;
            events.emit('error');
        });

        // Copy over the current parameters when the player is ready.
        // The user will still need to call `load` to request an ad.
        ready.promise.then(function() {
            tryCall('setPreload', [preload]);
            tryCall('setAutoplay', [autoplay]);
            tryCall('setLoop', [loop]);
            tryCall('setControls', [controls]);
            tryCall('setDefaultPlaybackRate', [defaultPlaybackRate]);
            tryCall('setPlaybackRate', [playbackRate]);
            tryCall('setVolume', [volume]);
            tryCall('setMuted', [muted]);
            tryCall('setDefaultMuted', [defaultMuted]);
            tryCall('setSrc', [src]);
        });
    };

    var attach = function() {
        // Prevent duplicates.
        if (player) {
            return;
        }

        // Insert IMAPlayer.swf into the DOM using swfobject.
        var url = `${Settings.playerHost}/vendor/IMAPlayer.swf`;
        var swfVersionStr = '10.2';
        var xiSwfUrlStr = 'playerProductInstall.swf';
        var width = '100%';
        var height = '100%';

        var flashvars = {};
        flashvars.eventCallback = 'window._BackendIMAFlash_emitEvent';
        flashvars.eventCallbackArg = id;

        var params = {};
        params.bgcolor = '#000';
        params.allowscriptaccess = 'always';
        params.allowfullscreen = 'true';
        params.wmode = 'transparent';

        var attributes = {};
        attributes.align = 'middle';

        var temp = document.createElement('div');
        root.appendChild(temp);

        // Aassign a unique id so embedSWF can find the div.
        var elementId = `swfobject-ima-${id}`;
        temp.setAttribute('id', elementId);

        // Reject the promise if anything goes wrong.
        // Flash will fire the "init" event when it has fully initialized.
        swfobject.embedSWF(url, elementId, width, height, swfVersionStr, xiSwfUrlStr, flashvars, params, attributes, function(e) {
            if (e.success) {
                player = e.ref;
            } else {
                ready.reject('failed to create flash element');
            }
        });

        setTimeout(function() {
            ready.reject('failed to initialize flash');
        }, Settings.flashTimeout);
    };

    var detach = function() {
        if (!player) {
            return;
        }

        root.removeChild(player);

        // Recreate the ready promise so we can attach again.
        initPlayer();
    };

    // Calls an external interface Flash method with the given name and arguments.
    // Calling .apply doesn't work for Debug Flash FOR WHATEVER REASON.
    var flashCall = function(name, args) {
        args = args || [];

        if (options.debug) {
            console.log('flash ima call:', name, args);
        }

        if (!player[name]) {
            console.error(`${name} is not a valid Flash function`);
            return;
        }

        // TODO UGH, this is a huge hack.
        switch (args.length) {
        case 0:
            return player[name]();
        case 1:
            return player[name](args[0]);
        case 2:
            return player[name](args[0], args[1]);
        default:
                // This will break for the Flash debugger
            console.log('WARNING, too many arguments passed to Flash');
            return player[name].apply(this, args);
        }
    };

    // Queues the given function call until the flash player has loaded.
    // Use tryCall instead if you need a blocking call (ie. return value).
    var deferCall = function(func, args, callback) {
        // Queue the function call until the player has loaded.
        ready.promise.then(function() {
            var result = tryCall(func, args);

            // Return the result via a callback.
            if (callback) {
                callback(result);
            }
        });
    };

    // Call a function immediately iff the player has loaded.
    // If the player has not loaded yet, nothing happens.
    var tryCall = function(func, args) {
        if (!initialized) {
            return;
        }

        var result = flashCall(func, args);
        if (result && options.debug) {
            console.log('flash ima return:', func, '=', result);
        }

        return result;
    };

    self.addEventListener = function(name, callback) {
        events.on(name, callback);
    };

    self.removeEventListener = function(name, callback) {
        events.off(name, callback);
    };

    // Flash will call this function when it emits events.
    // It must be public otherwise Flash cannot access it.
    self._emitEvent = function(name) {
        if (options.debug) {
            console.log('flash ima event:', name);
        }

        events.emit(name);
    };

    self.destroy = function() {
        detach();
    };

    self.getError = function() {
        return error;
    };

    self.getSrc = function() {
        return src;
    };

    self.setSrc = function(value) {
        src = value;

        // Setting src to empty string will remove the element from the DOM.
        if (value) {
            attach();
        } else {
            detach();
        }

        tryCall('setSrc', [value]);
    };

    self.getCurrentSrc = function() {
        return tryCall('getCurrentSrc');
    };

    self.NETWORK_EMPTY = 0;
    self.NETWORK_IDLE = 1;
    self.NETWORK_LOADING = 2;
    self.NETWORK_NO_SOURCE = 3;

    self.getNetworkState = function() {
        return tryCall('getNetworkState');
    };

    self.getPreload = function() {
        return preload;
    };

    self.setPreload = function(value) {
        preload = value;
        tryCall('setPreload', [value]);
    };

    self.getBuffered = function() {
        return tryCall('getBuffered');
    };

    self.load = function() {
        deferCall('load');
    };

    self.HAVE_NOTHING = 0;
    self.HAVE_METADATA = 1;
    self.HAVE_CURRENT_DATA = 2;
    self.HAVE_FUTURE_DATA = 3;
    self.HAVE_ENOUGH_DATA = 4;

    self.getReadyState = function() {
        return tryCall('getReadyState');
    };

    self.getSeeking = function() {
        return tryCall('getSeeking');
    };

    self.getCurrentTime = function() {
        return tryCall('getCurrentTime');
    };

    self.setCurrentTime = function(value) {
        deferCall('setCurrentTime', [value]);
    };

    self.getInitialTime = function() {
        return tryCall('getInitialTime');
    };

    self.getDuration = function() {
        return tryCall('getDuration');
    };

    self.getStartOffsetTime = function() {
        return tryCall('getStartOffsetTime');
    };

    self.getPaused = function() {
        return tryCall('getPaused');
    };

    self.getDefaultPlaybackRate = function() {
        return defaultPlaybackRate;
    };

    self.setDefaultPlaybackRate = function(value) {
        defaultPlaybackRate = value;
        tryCall('setDefaultPlaybackRate', [value]);
    };

    self.getPlaybackRate = function() {
        return playbackRate;
    };

    self.setPlaybackRate = function(value) {
        playbackRate = value;
        tryCall('setPlaybackRate', [value]);
    };

    self.getPlayed = function() {
        return tryCall('getPlayed');
    };

    self.getSeekable = function() {
        return tryCall('getSeekable');
    };

    self.getEnded = function() {
        return tryCall('getEnded');
    };

    self.getAutoplay = function() {
        return autoplay;
    };

    self.setAutoplay = function(value) {
        autoplay = value;
        tryCall('setAutoplay', [value]);
    };

    self.getLoop = function() {
        return loop;
    };

    self.setLoop = function(value) {
        loop = value;
        tryCall('setLoop', [value]);
    };

    self.play = function() {
        deferCall('play');
    };

    self.pause = function() {
        deferCall('pause');
    };

    self.getControls = function() {
        return controls;
    };

    self.setControls = function(value) {
        controls = value;
        tryCall('setControls', [value]);
    };

    self.getVolume = function() {
        return volume;
    };

    self.setVolume = function(value) {
        volume = value;
        tryCall('setVolume', [value]);
    };

    self.getMuted = function() {
        return muted;
    };

    self.setMuted = function(value) {
        muted = value;
        tryCall('setMuted', [value]);
    };

    self.getDefaultMuted = function() {
        return defaultMuted;
    };

    self.setDefaultMuted = function(value) {
        defaultMuted = value;
        tryCall('setDefaultMuted', [value]);
    };

    self.getQuality = function() {
        return 'auto';
    };

    self.setQuality = function() {
        // unimplemented
    };

    self.getQualities = function() {
        return ['auto'];
    };

    self.getChannel = function() {
        return channel;
    };

    self.setChannel = function(value) {
        channel = value;
        video = null;
    };

    self.getVideo = function() {
        return video;
    };

    self.setVideo = function(value) {
        channel = null;
        video = value;
    };

    self.getStats = function() {
        // unimplemented
    };

    self.getStatsEnabled = function() {
        // unimplemented
    };

    self.setStatsEnabled = function() {
        // unimplemented
    };

    self.getCaption = function() {
        // unimplemented
    };

    self.getPlaylistLength = function() {
        return tryCall('getPlaylistLength');
    };

    self.getPlaylistDuration = function() {
        return tryCall('getPlaylistDuration');
    };

    self.getAdInfo = function() {
        return tryCall('getAdInfo');
    };

    init();
}

// We need some static members to uniquely identify objects.
BackendIMAFlash.map = {};
BackendIMAFlash.counter = 0;

BackendIMAFlash.canPlay = function() {
    return swfobject.hasFlashPlayerVersion('10.2');
};

// Flash fires this static callback on events.
// ugh, this needs to be public so Flash can discover it.
window._BackendIMAFlash_emitEvent = function(context, e) { // eslint-disable-line camelcase
    // Map from the id to the BackendIMAFlash object.
    var self = BackendIMAFlash.map[context];
    self._emitEvent(e);
};
