/* global google */
import EventEmitter from 'event-emitter';

export function BackendIMAHtml5(root) {
    const self = this;

    // This backend implements the HTML5 IMA SDK:
    // https://developers.google.com/interactive-media-ads/docs/sdks/html5/

    // The IMA SDK is fairly simple but there's a lot of options.
    // You start by creating an AdsDisplayContainer and appending it
    // to the DOM. We create a new overlay for each request to improve
    // stream FPS, at the trade-off of longer initialization times.
    // Once you have a container, you create an AdsLoader and give it a
    // request URL. Eventually, it will fire a callback with an AdsManager.

    // The AdsManager object is responsible for playing multiple ads
    // within the response. The AdsManager interface is fairly custom so
    // we emulate the Backend interface instead. It handles the VAST
    // tracking and has a number of different callbacks we use.

    const events = new EventEmitter();

    // The IMA SDK requires a video object used to schedule VMAP midrolls.
    // We don't use midrolls, so we make an empty player to fool it.
    // TODO Determine if this causes any problems.
    const fakePlayer = document.createElement('video');

    let volume;
    let muted;
    let channel;
    let video;

    let paused;
    let ended;
    let error;
    let src;
    let autoplay;
    let duration;

    let networkState;
    let readyState;

    let adsDisplay;
    let adsLoader;
    let adsManager;

    let playlistLength;
    let playlistDuration;
    let adInfo;

    var init = function() {
        ended = true;
        paused = false;

        networkState = self.NETWORK_EMPTY;
        readyState = self.HAVE_NOTHING;
    };

    var attach = function() {
        if (adsDisplay) {
            return;
        }

        // Creates an iframe on the page containing the IMA player.
        adsDisplay = new google.ima.AdDisplayContainer(root, fakePlayer);

        // TODO "Must be done as the result of a user action on mobile"
        adsDisplay.initialize();

        // This AdsLoader it used to request the ads url.
        adsLoader = new google.ima.AdsLoader(adsDisplay);
        adsLoader.addEventListener(google.ima.AdsManagerLoadedEvent.Type.ADS_MANAGER_LOADED, onAdsManagerLoaded, false);
        adsLoader.addEventListener(google.ima.AdErrorEvent.Type.AD_ERROR, onAdError, false);

        // Disable auto ad breaks and JS VPAID ads.
        var settings = adsLoader.getSettings();
        settings.setAutoPlayAdBreaks(false);
        settings.setVpaidMode(google.ima.ImaSdkSettings.VpaidMode.DISABLED);
    };

    var detach = function() {
        if (!adsDisplay) {
            return;
        }

        adsDisplay.destroy();
        adsLoader.destroy();

        adsDisplay = null;
        adsLoader = null;
    };

    var loadAd = function() {
        // Request video ads.
        var adsRequest = new google.ima.AdsRequest();
        adsRequest.adTagUrl = src;
        adsRequest.setAdWillAutoPlay(true);

        // "Specify the linear and nonlinear slot sizes. This helps the SDK
        // to select the correct creative if multiple are returned."
        // We don't have nonLinear ads but give it the element size anyway.
        adsRequest.linearAdSlotWidth = root.offsetWidth;
        adsRequest.linearAdSlotHeight = root.offsetHeight;
        adsRequest.nonLinearAdSlotWidth = root.offsetWidth;
        adsRequest.nonLinearAdSlotHeight = root.offsetHeight;

        adsLoader.requestAds(adsRequest);
    };

    var onAdsManagerLoaded = function(adsManagerLoadedEvent) {
        // Tear down any previous ads manager.
        // This will also cancel any ads in progress.
        if (adsManager) {
            adsManager.destroy();
        }

        // Get the ads manager using our fakePlayer.
        adsManager = adsManagerLoadedEvent.getAdsManager(fakePlayer);
        adsManager.setVolume(muted ? 0 : volume);

        // Add listeners to the all of the fired events.
        // We don't use all of these but I left them in for completion.
        adsManager.addEventListener(google.ima.AdErrorEvent.Type.AD_ERROR, onAdError);
        adsManager.addEventListener(google.ima.AdEvent.Type.AD_BREAK_READY, onAdBreakReady);
        adsManager.addEventListener(google.ima.AdEvent.Type.AD_METADATA, onAdMetadata);
        adsManager.addEventListener(google.ima.AdEvent.Type.ALL_ADS_COMPLETED, onAllAdsCompleted);
        adsManager.addEventListener(google.ima.AdEvent.Type.COMPLETE, onComplete);
        adsManager.addEventListener(google.ima.AdEvent.Type.CONTENT_PAUSE_REQUESTED, onContentPauseRequested);
        adsManager.addEventListener(google.ima.AdEvent.Type.CONTENT_RESUME_REQUESTED, onContentResumeRequested);
        adsManager.addEventListener(google.ima.AdEvent.Type.LOADED, onLoaded);
        adsManager.addEventListener(google.ima.AdEvent.Type.LOG, onLog);
        adsManager.addEventListener(google.ima.AdEvent.Type.STARTED, onStarted);
        adsManager.addEventListener(google.ima.AdEvent.Type.IMPRESSION, onImpression);
        adsManager.addEventListener(google.ima.AdEvent.Type.VOLUME_CHANGED, onVolumeChanged);
        adsManager.addEventListener(google.ima.AdEvent.Type.VOLUME_MUTED, onVolumeChanged);

        try {
            // "Initialize the ads manager. Ad rules playlist will start at this time."
            adsManager.init(-1, -1, google.ima.ViewMode.NORMAL);
            adsManager.start();
        } catch (adError) {
            // An error may be thrown if there was a problem with the VAST response.
            error = adError.getMessage();
            events.emit('error');
        }
    };

    var onAdError = function(adErrorEvent) {
        var err = adErrorEvent.getError();

        error = err.getMessage();
        events.emit('error');

        if (adsManager) {
            adsManager.destroy();
            adsManager = null;
        }
    };

    var onAdBreakReady = function() {
        // TODO Not sure when this is fired.
    };

    var onAdMetadata = function() {
        readyState = self.HAVE_METADATA;
        events.emit('loadedmetadata');
    };

    var onAllAdsCompleted = function() {
        duration = 0;
        ended = true;

        readyState = self.HAVE_NOTHING;
        networkState = self.NETWORK_EMPTY;

        events.emit('ended');

        adsManager.destroy();
        adsManager = null;
    };

    var onComplete = function() {
        events.emit('adend');
        adInfo = {};
    };

    var onContentPauseRequested = function() {
        // Not used but required.
        // We use onStarted instead.
    };

    var onContentResumeRequested = function() {
        // Not used but required.
        // We use onAllAdsCompleted instead.
    };

    var onImpression = function() {};

    var onLoaded = function(adEvent) {
        var ad = adEvent.getAd();
        var adPod = ad.getAdPodInfo();

        duration = ad.getDuration();
        playlistLength = adPod.getTotalAds();
        playlistDuration = adPod.getMaxDuration();

        /* eslint-disable camelcase */
        // Create a blob of ad information, mostly for tracking.
        adInfo = {
            ad_id: ad.getAdId(),
            ad_system: ad.getAdSystem(),
            ad_framework: 'VAST',

            video_type: ad.getContentType(),
            video_duration: ad.getDuration(),
            video_linear: ad.isLinear(),
            video_skippable: ad.isSkippable(),

            playlist_length: adPod.getTotalAds(),
            playlist_position: adPod.getAdPosition(),
        };
        /* eslint-enable camelcase */

        if (readyState < self.HAVE_CURRENT_DATA) {
            readyState = self.HAVE_CURRENT_DATA;
            events.emit('loadeddata');
        }
    };

    var onLog = function(adEvent) {
        var data = adEvent.getAdData();
        var adError = data.adError;

        if (adError) {
            console.warn('non-fatal ad error occurred:', adError.getMessage());
        }
    };

    var onStarted = function() {
        if (readyState < self.HAVE_FUTURE_DATA) {
            readyState = self.HAVE_FUTURE_DATA;
            events.emit('canplay');
        }

        events.emit('adstart');
        events.emit('playing');
    };

    var onVolumeChanged = function() {
        var value = adsManager.getVolume();

        if (value) {
            volume = value;
        } else {
            muted = true;
        }

        events.emit('volumechange');
    };

    self.addEventListener = function(name, callback) {
        events.on(name, callback);
    };

    self.removeEventListener = function(name, callback) {
        events.off(name, callback);
    };

    self.destroy = function() {
        detach();
    };

    self.getError = function() {
        return error;
    };

    self.getSrc = function() {
        return src;
    };

    self.setSrc = function(value) {
        src = value;

        if (value) {
            attach();
        } else {
            detach();
        }

        if (autoplay && src) {
            self.load();
        }
    };

    self.getCurrentSrc = function() {
        // unimplemented
    };

    self.NETWORK_EMPTY = 0;
    self.NETWORK_IDLE = 1;
    self.NETWORK_LOADING = 2;
    self.NETWORK_NO_SOURCE = 3;

    self.getNetworkState = function() {
        return networkState;
    };

    self.getPreload = function() {
        // unimplemented
    };

    self.setPreload = function() {
        // unimplemented
    };

    self.getBuffered = function() {
        return {
            length: 0,
        };
    };

    self.load = function() {
        ended = false;
        loadAd();
    };

    self.HAVE_NOTHING = 0;
    self.HAVE_METADATA = 1;
    self.HAVE_CURRENT_DATA = 2;
    self.HAVE_FUTURE_DATA = 3;
    self.HAVE_ENOUGH_DATA = 4;

    self.getReadyState = function() {
        return readyState;
    };

    self.getSeeking = function() {
        // unimplemented
    };

    self.getCurrentTime = function() {
        if (duration && adsManager) {
            var remaining = adsManager.getRemainingTime();
            return duration - remaining;
        } else {
            return 0;
        }
    };

    self.setCurrentTime = function() {
        // unimplemented
    };

    self.getInitialTime = function() {
        // unimplemented
    };

    self.getDuration = function() {
        if (duration) {
            return duration;
        } else {
            return 0;
        }
    };

    self.getStartOffsetTime = function() {
        // unimplemented
    };

    self.getPaused = function() {
        return paused;
    };

    self.getDefaultPlaybackRate = function() {
        // unimplemented
    };

    self.setDefaultPlaybackRate = function() {
        // unimplemented
    };

    self.getPlaybackRate = function() {
        // unimplemented
    };

    self.setPlaybackRate = function() {
        // unimplemented
    };

    self.getPlayed = function() {
        // unimplemented
    };

    self.getSeekable = function() {
        // unimplemented
    };

    self.getEnded = function() {
        if (adsManager) {
            return adsManager.getRemainingTime() === -1;
        } else {
            return true;
        }
    };

    self.getAutoplay = function() {
        return autoplay;
    };

    self.setAutoplay = function(value) {
        autoplay = value;

        if (autoplay && src) {
            self.load();
        }
    };

    self.getLoop = function() {
        // unimplemented
    };

    self.setLoop = function() {
        // unimplemented
    };

    self.play = function() {
        if (ended) {
            self.load();
        }

        if (adsManager) {
            adsManager.resume();
        }

        paused = false;
        events.emit('play');
    };

    self.pause = function() {
        if (adsManager) {
            adsManager.pause();
        }

        paused = true;
        events.emit('pause');
    };

    self.getControls = function() {
        // unimplemented
    };

    self.setControls = function() {
        // unimplemented
    };

    self.getVolume = function() {
        return volume;
    };

    self.setVolume = function(value) {
        volume = value;

        if (adsManager) {
            adsManager.setVolume(muted ? 0 : value);
        }

        events.emit('volumechange');
    };

    self.getMuted = function() {
        return muted;
    };

    self.setMuted = function(value) {
        muted = value;

        if (adsManager) {
            adsManager.setVolume(muted ? 0 : volume);
        }

        events.emit('volumechange');
    };

    self.getDefaultMuted = function() {
        // unimplemented
    };

    self.setDefaultMuted = function() {
        // unimplemented
    };

    self.getQuality = function() {
        return 'auto';
    };

    self.setQuality = function() {
        // unimplemented
    };

    self.getQualities = function() {
        return ['auto'];
    };

    self.getChannel = function() {
        return channel;
    };

    self.setChannel = function(value) {
        channel = value;
        video = null;
    };

    self.getVideo = function() {
        return video;
    };

    self.setVideo = function(value) {
        channel = null;
        video = value;
    };

    self.getStats = function() {
        // unimplemented
    };

    self.getStatsEnabled = function() {
        // unimplemented
    };

    self.setStatsEnabled = function() {
        // unimplemented
    };

    self.getCaption = function() {
        //unimplemented
    };

    self.getPlaylistLength = function() {
        return playlistLength;
    };

    self.getPlaylistDuration = function() {
        return playlistDuration;
    };

    self.getAdInfo = function() {
        return adInfo;
    };

    init();
}

// Crude adblock detection.
BackendIMAHtml5.canPlay = function() {
    // IMA SDK could not load, probably adblock.
    return !!(window.google && window.google.ima);
};
