import EventEmitter from 'event-emitter';
import { BackendIMAFlash } from './ima-flash';
import { BackendIMAHtml5 } from './ima-html5';
import { videoInfo, channelInfo } from '../api';
import * as Settings from '../settings';
import * as UniqueId from '../util/unique-id';

export function BackendIMA(root, analytics, options) {
    const self = this;

    // This backend handles the dynamic switching between Flash and HTML5.
    // Both implementations use the IMA SDK and provide the same interface.
    // We default to the Flash implementation because it supports VPAID ads,
    // and failover to the HTML5 implementation if Flash is busted or not
    // supported. The failover is triggered by the "error" event.

    const events = new EventEmitter();

    let backend;
    let backendType;
    let backendName;
    let channel;
    let video;

    // Required for tracking.
    let requestedType;
    let requestedDuration;
    let sessionId;

    var init = function() {
        if (BackendIMAFlash.canPlay()) {
            backendType = BackendIMAFlash;
            backendName = 'ima_flash';
        } else if (BackendIMAHtml5.canPlay()) {
            backendType = BackendIMAHtml5;
            backendName = 'ima_html5';
        } else {
            backendType = null;
            backendName = 'unsupported';
        }

        if (backendType) {
            createBackend();
        }
    };

    var createBackend = function() {
        // Create a new backend with the given type.
        var newBackend = new backendType(root, options);

        newBackend.addEventListener('ended', function() {
            // Unsetting the src will remove the element from the DOM.
            // This improves the frame rate when an ad isn't playing, but
            // requires re-initializing the ads backend each time..
            newBackend.setSrc('');
        });

        newBackend.addEventListener('error', function() {
            // Cancel and remove the ad if it errors for any reason.
            backend.setSrc('');

            // Fail-over to the HTML5 ad player if currently using Flash.
            if (backendType === BackendIMAFlash && BackendIMAHtml5.canPlay()) {
                // Track the failover player with a different name.
                backendType = BackendIMAHtml5;
                backendName = 'ima_html5_failover';

                // Create the new backend.
                createBackend();

                // Request ads again.
                self.requestAds(requestedType, requestedDuration);
            }
        });

        _.each(Settings.allEvents, function(name) {
            newBackend.addEventListener(name, function() {
                events.emit(name);
            });
        });

        // Register the event listeners before we start using the backend.
        newBackend.addEventListener('loadeddata', function() {
            /* eslint-disable camelcase */
            // loadeddata is the first time we get playlist length/duration.
            trackEvent('video_ad_request_response', {
                playlist_length: newBackend.getPlaylistLength(),
                time_break: newBackend.getPlaylistDuration(),
                time_break_requested: requestedDuration,
            });
            /* eslint-enable camelcase */
        });

        newBackend.addEventListener('adstart', function() {
            var adInfo = newBackend.getAdInfo();
            trackEvent('video_ad_impression', adInfo);
        });

        newBackend.addEventListener('adend', function() {
            var adInfo = newBackend.getAdInfo();
            trackEvent('video_ad_impression_complete', adInfo);
        });

        newBackend.addEventListener('error', function() {
            var adInfo = newBackend.getAdInfo();
            if (!adInfo) {
                // No ad info means it was a request issue.
                trackEvent('video_ad_request_error', {
                    time_break: requestedDuration, // eslint-disable-line camelcase
                });
            } else {
                trackEvent('video_ad_impression_timeout', adInfo);
            }
        });

        if (backend) {
            // Copy over any settings from the previous backend.
            // This is so we keep volume/autoplay/etc settings.
            copyBackend(backend, newBackend);

            // Destroy the previous backend now that we no longer need it.
            backend.destroy();
        }

        backend = newBackend;
    };

    var copyBackend = function(oldBackend, newBackend) {
        // Copy over all of the old backend settings to the new one.
        newBackend.setPreload(oldBackend.getPreload());
        newBackend.setDefaultPlaybackRate(oldBackend.getDefaultPlaybackRate());
        newBackend.setPlaybackRate(oldBackend.getPlaybackRate());
        newBackend.setAutoplay(oldBackend.getAutoplay());
        newBackend.setLoop(oldBackend.getLoop());
        newBackend.setControls(oldBackend.getControls());
        newBackend.setVolume(oldBackend.getVolume());
        newBackend.setMuted(oldBackend.getMuted());
        newBackend.setDefaultMuted(oldBackend.getDefaultMuted());
        newBackend.setQuality(oldBackend.getQuality());

        if (oldBackend.getChannel()) {
            newBackend.setChannel(oldBackend.getChannel());
        } else {
            newBackend.setVideo(oldBackend.getVideo());
        }

        newBackend.setSrc(oldBackend.getSrc());
    };

    // Ugh.
    // We need to construct a playlist URL for the ads request.
    // The problem is that it depends on multiple pieces of information
    // from the API, so we construct a Promise instead.
    // We also never want to return an error because it would block ads.
    var getUrlPromise = function(type, duration) {
        var channel = backend.getChannel();
        var channelPromise;

        if (!channel) {
            // We need to make an API request to get the channel for a VOD.
            var video = backend.getVideo();
            channelPromise = videoInfo(video).then(function(info) {
                return info.channel;
            }, function() {
                // Default to empty string to prevent errors.
                return '';
            });
        } else {
            channelPromise = Promise.resolve(channel);
        }

        var channelInfoPromise = channelPromise.then(function(channel) {
            if (!channel) {
                // Default to empty object to prevent errors.
                return {};
            }

            // Get the channel info, and cast errors to empty object.
            return channelInfo(channel).then(null, function() {
                return {};
            });
        });

        return channelInfoPromise.then(function(channelInfo) {
            // FINALLY, call getUrl with fetched information.
            return getUrl(type, duration, channelInfo);
        });
    };

    // Constructs a doubleclick playlist URL.
    // TODO Add support for other providers.
    var getUrl = function(type, duration, channelInfo) {
        var url = Settings.doubleClickUrl;

        if (!duration) {
            duration = 30;
        }

        var typeId;
        if (type === 'preroll') {
            typeId = 1;
        } else if (type === 'midroll') {
            typeId = 2;
        } else if (type === 'postroll') {
            typeId = 3;
        }

        var customParams = {
            partner: channelInfo.partner,
            noAd: false,
            game: channelInfo.game,
            chan: channelInfo.name,
            embed: false, // TODO
            playerType: 'site', // TODO
            mature: channelInfo.mature,
            pos: typeId,
            timebreak: duration,
        };

        if (window.Krux) {
            customParams.kuid = window.Krux.user;
        }

        // We can't use $.param because it will be double escaped.
        var customSerialized = _.map(customParams, function(value, key) {
            return `${key}=${value}`;
        }).join('&');

        /* eslint-disable camelcase */
        var params = {
            iu: `/3576121/twitch/channels/${channelInfo.name}`,
            ciu_szs: '300x250',
            sz: '640x480',
            impl: 's',
            gdfp_req: '1',
            env: 'vp',
            output: 'vast',
            cust_params: customSerialized,
            unviewed_position_start: '1',
            url: window.location.href,
            correlator: (new Date()).getTime() / 1000,
        };
        /* eslint-enable camelcase */

        return `${url}?${$.param(params)}`;
    };

    // Request ads with the possible types "preroll", "midroll", "postroll"
    self.requestAds = function(type, duration) {
        // 30 seconds is defacto ad duration.
        if (!duration) {
            duration = 30;
        }

        requestedType = type;
        requestedDuration = duration;
        sessionId = UniqueId.generate();

        trackEvent('video_ad_request', {
            time_break: requestedDuration, // eslint-disable-line camelcase
        });

        if (!backend) {
            // ads_backend = "unsupported", which is probably adblock.
            trackEvent('video_ad_request_error', {
                time_break: requestedDuration, // eslint-disable-line camelcase
            });

            return;
        }

        // Fetches channel information and constructs a playlist url.
        var urlPromise = getUrlPromise(type, duration);
        urlPromise.then(function(url) {
            backend.setSrc(url);
        });
    };

    var trackEvent = function(name, properties) {
        /* eslint-disable camelcase */
        properties = _.extend({}, properties, {
            ad_session_id: sessionId,
            ad_backend: backendName,
            roll_type: requestedType,
        });
        /* eslint-enable camelcase */

        analytics.trackEvent(name, properties);
    };

    self.addEventListener = function(name, callback) {
        events.on(name, callback);
    };

    self.removeEventListener = function(name, callback) {
        events.off(name, callback);
    };

    self.destroy = function() {
        backend.destroy();
    };

    self.getError = function() {
        return backend.getError();
    };

    self.getSrc = function() {
        return backend.getSrc();
    };

    self.setSrc = function(value) {
        backend.setSrc(value);
    };

    self.getCurrentSrc = function() {
        return backend.getCurrentSrc();
    };

    self.NETWORK_EMPTY = 0;
    self.NETWORK_IDLE = 1;
    self.NETWORK_LOADING = 2;
    self.NETWORK_NO_SOURCE = 3;

    self.getNetworkState = function() {
        return backend.getNetworkState();
    };

    self.getPreload = function() {
        return backend.getPreload();
    };

    self.setPreload = function(value) {
        backend.setPreload(value);
    };

    self.getBuffered = function() {
        return backend.getBuffered();
    };

    self.load = function() {
        backend.load();
    };

    self.HAVE_NOTHING = 0;
    self.HAVE_METADATA = 1;
    self.HAVE_CURRENT_DATA = 2;
    self.HAVE_FUTURE_DATA = 3;
    self.HAVE_ENOUGH_DATA = 4;

    self.getReadyState = function() {
        return backend.getReadyState();
    };

    self.getSeeking = function() {
        return backend.getSeeking();
    };

    self.getCurrentTime = function() {
        return backend.getCurrentTime();
    };

    self.setCurrentTime = function(value) {
        backend.setCurrentTime(value);
    };

    self.getInitialTime = function() {
        return backend.getInitialTime();
    };

    self.getDuration = function() {
        return backend.getDuration();
    };

    self.getStartOffsetTime = function() {
        return backend.getStartOffsetTime();
    };

    self.getPaused = function() {
        return backend.getPaused();
    };

    self.getDefaultPlaybackRate = function() {
        return backend.getDefaultPlaybackRate();
    };

    self.setDefaultPlaybackRate = function(value) {
        backend.setDefaultPlaybackRate(value);
    };

    self.getPlaybackRate = function() {
        return backend.getPlaybackRate();
    };

    self.setPlaybackRate = function(value) {
        backend.setPlaybackRate(value);
    };

    self.getPlayed = function() {
        return backend.getPlayed();
    };

    self.getSeekable = function() {
        return backend.getSeekable();
    };

    self.getEnded = function() {
        return backend.getEnded();
    };

    self.getAutoplay = function() {
        return backend.getAutoplay();
    };

    self.setAutoplay = function(value) {
        backend.setAutoplay(value);
    };

    self.getLoop = function() {
        return backend.getLoop();
    };

    self.setLoop = function(value) {
        backend.setLoop(value);
    };

    self.play = function() {
        backend.play();
    };

    self.pause = function() {
        backend.pause();
    };

    self.getControls = function() {
        return backend.getControls();
    };

    self.setControls = function(value) {
        backend.setControls(value);
    };

    self.getVolume = function() {
        return backend.getVolume();
    };

    self.setVolume = function(value) {
        backend.setVolume(value);
    };

    self.getMuted = function() {
        return backend.getMuted();
    };

    self.setMuted = function(value) {
        backend.setMuted(value);
    };

    self.getDefaultMuted = function() {
        return backend.getDefaultMuted();
    };

    self.setDefaultMuted = function(value) {
        backend.setDefaultMuted(value);
    };

    self.getQuality = function() {
        return backend.getQuality();
    };

    self.setQuality = function(value) {
        backend.setQuality(value);
    };

    self.getQualities = function() {
        return backend.getQualities();
    };

    self.getChannel = function() {
        return channel;
    };

    self.setChannel = function(value) {
        channel = value;

        if (backend) {
            backend.setChannel(value);
        }
    };

    self.getVideo = function() {
        return video;
    };

    self.setVideo = function(value) {
        video = value;

        if (backend) {
            backend.setVideo(value);
        }
    };

    self.getStats = function() {
        return backend.getStats();
    };

    self.getStatsEnabled = function() {
        return backend.getStatsEnabled();
    };

    self.setStatsEnabled = function() {
        return backend.setStatsEnabled();
    };

    self.getVideoInfo = function() {
        return backend.getVideoInfo();
    };

    self.getCaption = function() {
        return backend.getCaption();
    };

    init();
}

// Crude adblock detection.
BackendIMA.canPlay = function() {
    return BackendIMAFlash.canPlay() || BackendIMAHtml5.canPlay();
};
