import EventEmitter from 'event-emitter';
import * as Settings from '../settings';

// Switches automatically between two backends automatically.
// This is used to prioritize ads and chromecast over the normal player.
// Switching is based on the `canplay` and `ended` events.
export function BackendMulti(primary, secondary) {
    var self = this;

    var events = new EventEmitter();
    var queue = [];

    var current;
    var backends = [primary, secondary];

    // Prevent the duplicate events: loadstart, loadedmetadata, loadeddata.
    var duplicateEvents = {};

    var init = function() {
        if (primary.getReadyState() >= primary.HAVE_CURRENT_DATA && !primary.getEnded()) {
            current = primary;
        } else {
            current = secondary;
        }

        // Switch from secondary to primary when primary starts.
        primary.addEventListener('canplay', function() {
            if (current === primary) {
                return;
            }

            switchBackend(secondary, primary);
        });

        // Switch from primary to secondary when primary ends.
        primary.addEventListener('ended', function() {
            if (current !== primary) {
                return;
            }

            switchBackend(primary, secondary);
        });

        // Forward events from both backends.
        _.each(backends, function(backend) {
            _.each(Settings.allEvents, function(name) {
                backend.addEventListener(name, function() {
                    var args = Array.prototype.slice.call(arguments);
                    queueEvent(backend, name, args);
                });
            });
        });
    };

    var queueEvent = function(backend, name, args) {
        // Add the event name to the args to make things easier.
        args.unshift(name);

        // Prevent firing "ended" unless the secondary is ended also.
        if (name === 'ended' && backend === primary && !secondary.getEnded()) {
            return;
        }

        // We can't fire both backends events at once because the UI would
        // get confused. Rather than drop the non-active backend events,
        // we queue them and fire them all on backend activation.
        if (current === backend) {
            handleEvent(args);
        } else {
            queue.push(args);
        }
    };

    var flushEvents = function() {
        _.each(queue, function(args) {
            handleEvent(args);
        });

        queue = [];
    };

    var handleEvent = function(args) {
        var name = args[0];

        // These events should only be fired once, so prevent duplicates.
        var duplicateKeys = ['loadstart', 'loadedmetadata', 'loadeddata'];
        var duplicate = false;

        if (_.contains(duplicateKeys, name)) {
            duplicate = _.has(duplicateEvents, name);
            duplicateEvents[name] = true;
        }

        if (duplicate) {
            return;
        }

        // Clear duplicate events when we fully end.
        if (name === 'ended') {
            duplicateEvents = [];
        }

        events.emit.apply(events, args);
    };

    var switchBackend = function(previous, next) {
        previous.pause();

        current = next;
        flushEvents();

        if (previous.getDuration() !== next.getDuration()) {
            events.emit('durationchange');
        }

        if (previous.getCurrentTime() !== next.getCurrentTime()) {
            events.emit('timeupdate');
        }

        // TODO This will always be true because buffered is an object.
        if (previous.getBuffered() !== next.getBuffered()) {
            events.emit('bufferedchange');
        }

        if (previous.getQuality() !== next.getQuality()) {
            events.emit('qualitychange');
        }

        // TODO This will always be true because qualities is an array.
        if (previous.getQualities() !== next.getQualities()) {
            events.emit('qualitieschange');
        }

        next.play();
    };

    self.destroy = function() {
        primary.destroy();
        secondary.destroy();
    };

    self.addEventListener = function(name, func) {
        events.on(name, func);
    };

    self.removeEventListener = function(name, func) {
        events.off(name, func);
    };

    self.getNetworkProfile = function() {
        return current.getNetworkProfile();
    };

    self.getError = function() {
        return current.getError();
    };

    self.getSrc = function() {
        return current.getSrc();
    };

    self.setSrc = function(value) {
        _.each(backends, function(backend) {
            backend.setSrc(value);
        });
    };

    self.getCurrentSrc = function() {
        return current.getCurrentSrc();
    };

    self.NETWORK_EMPTY = 0;
    self.NETWORK_IDLE = 1;
    self.NETWORK_LOADING = 2;
    self.NETWORK_NO_SOURCE = 3;

    self.getNetworkState = function() {
        return current.getNetworkState();
    };

    self.getPreload = function() {
        return current.getPreload();
    };

    self.setPreload = function(value) {
        _.each(backends, function(backend) {
            backend.setPreload(value);
        });
    };

    self.getBuffered = function() {
        return current.getBuffered();
    };

    self.load = function() {
        current.load();
    };

    self.HAVE_NOTHING = 0;
    self.HAVE_METADATA = 1;
    self.HAVE_CURRENT_DATA = 2;
    self.HAVE_FUTURE_DATA = 3;
    self.HAVE_ENOUGH_DATA = 4;

    self.getReadyState = function() {
        return current.getReadyState();
    };

    self.getSeeking = function() {
        return current.getSeeking();
    };

    self.getCurrentTime = function() {
        return current.getCurrentTime();
    };

    self.setCurrentTime = function(value) {
        _.each(backends, function(backend) {
            return backend.setCurrentTime(value);
        });
    };

    self.getInitialTime = function() {
        return current.getCurrentTime();
    };

    self.getDuration = function() {
        return current.getDuration();
    };

    self.getStartOffsetTime = function() {
        return current.getStartOffsetTime();
    };

    self.getPaused = function() {
        return current.getPaused();
    };

    self.getDefaultPlaybackRate = function() {
        return current.getDefaultPlaybackRate();
    };

    self.setDefaultPlaybackRate = function(value) {
        _.each(backends, function(backend) {
            backend.setDefaultPlaybackRate(value);
        });
    };

    self.getPlaybackRate = function() {
        return current.getPlaybackRate();
    };

    self.setPlaybackRate = function(value) {
        _.each(backends, function(backend) {
            return backend.setPlaybackRate(value);
        });
    };

    self.getPlayed = function() {
        return current.getPlayed();
    };

    self.getSeekable = function() {
        return current.getSeekable();
    };

    self.getEnded = function() {
        return current.getEnded();
    };

    self.getAutoplay = function() {
        return current.getAutoplay();
    };

    self.setAutoplay = function(value) {
        _.each(backends, function(backend) {
            backend.setAutoplay(value);
        });
    };

    self.getLoop = function() {
        return current.getLoop();
    };

    self.setLoop = function(value) {
        _.each(backends, function(backend) {
            backend.setLoop(value);
        });
    };

    self.play = function() {
        current.play();
    };

    self.pause = function() {
        current.pause();
    };

    self.getControls = function() {
        return current.getControls();
    };

    self.setControls = function(value) {
        _.each(backends, function(backend) {
            backend.setControls(value);
        });
    };

    self.getVolume = function() {
        return current.getVolume();
    };

    self.setVolume = function(value) {
        _.each(backends, function(backend) {
            backend.setVolume(value);
        });
    };

    self.getMuted = function() {
        return current.getMuted();
    };

    self.setMuted = function(value) {
        _.each(backends, function(backend) {
            backend.setMuted(value);
        });
    };

    self.getDefaultMuted = function() {
        return current.getDefaultMuted();
    };

    self.setDefaultMuted = function(value) {
        _.each(backends, function(backend) {
            backend.setDefaultMuted(value);
        });
    };

    self.getQuality = function() {
        return current.getQuality();
    };

    self.setQuality = function(value) {
        _.each(backends, function(backend) {
            backend.setQuality(value);
        });
    };

    self.getQualities = function() {
        return current.getQualities();
    };

    self.getChannel = function() {
        return current.getChannel();
    };

    self.setChannel = function(value) {
        _.each(backends, function(backend) {
            backend.setChannel(value);
        });
    };

    self.getVideo = function() {
        return current.getVideo();
    };

    self.getBackend = function() {
        return current.getBackend();
    };

    self.setVideo = function(value) {
        _.each(backends, function(backend) {
            backend.setVideo(value);
        });
    };

    self.getStats = function() {
        return current.getStats();
    };

    self.getStatsEnabled = function() {
        return current.getStatsEnabled();
    };

    self.setStatsEnabled = function(value) {
        _.each(backends, function(backend) {
            backend.setStatsEnabled(value);
        });
    };

    self.getVideoInfo = function() {
        return current.getVideoInfo();
    };

    self.isSpectre = function() {
        return current.isSpectre();
    };

    self.getCaption = function() {
        return current.getCaption();
    };

    init();
}
