import EventEmitter from 'event-emitter';
import { BRIDGE_STATE_UPDATE, BRIDGE_HOST_READY, BRIDGE_REQ_SUBSCRIBE,
         BRIDGE_PLAYER_EVENT, BRIDGE_HOST_NAMESPACE, BRIDGE_CLIENT_NAMESPACE } from './host';
import { DEFAULT_EMBED_STATE } from '../state-tracker';

const HOST_WAIT_TIMEOUT = 15000; //milliseconds

export const EVENT_EMBED_READY = 'ready';

export class EmbedClient {
    constructor(hostFrame) {
        this._host = hostFrame;
        this._eventEmitter = new EventEmitter();
        this._playerState = DEFAULT_EMBED_STATE;
        this._onHostReady = this._getHostReady();
        this._send(BRIDGE_REQ_SUBSCRIBE);
    }

    _getHostReady() {
        return new Promise((resolve, reject) => {
            const listenerFunc = hostListener.bind(this);
            window.addEventListener('message', listenerFunc);
            // 15 second timeout for bridge to be established
            setTimeout(reject, HOST_WAIT_TIMEOUT);
            function hostListener(message) {
                if (this._isClientMessage(message) && message.data.method === BRIDGE_HOST_READY) {
                    window.removeEventListener('message', listenerFunc);
                    window.addEventListener('message', this);
                    this._eventEmitter.emit(EVENT_EMBED_READY);
                    resolve();

                }
            }
        });
    }

    _send(method, ...args) {
        // Set the namespace so only servers can read this message.
        // This prevents us from accidentally talking to ourselves.
        const payload = {
            namespace: BRIDGE_HOST_NAMESPACE,
            method: method,
            args: args,
        };

        this._host.postMessage(payload, '*');
    }

    _deferSend(...args) {
        this._onHostReady.then(
            () => this._send.apply(this, args)
        );
    }

    _isClientMessage(message) {
        // Other components use this API so we must check if the message
        // is actually intended for the player.
        // It is possible to have a client and server listening on the same
        // window, so we must also avoid talking to ourself.

        return Boolean(message.data) && message.data.namespace === BRIDGE_CLIENT_NAMESPACE;
    }

    handleEvent(event) {
        // This function is fired any time the window receives a message.

        if (!this._isClientMessage(event)) {
            return;
        }

        switch (event.data.method) {
        case BRIDGE_STATE_UPDATE:
            this._playerState = event.data.args[0];
            break;
        case BRIDGE_PLAYER_EVENT:
            this._eventEmitter.emit(event.data.args[0]);
            break;
        }
    }

    getPlayerState() {
        return this._playerState;
    }

    addEventListener(name, callback) {
        this._eventEmitter.on(name, callback);
    }

    removeEventListener(name, callback) {
        this._eventEmitter.off(name, callback);
    }

    callPlayerMethod(name, argument) {
        this._deferSend(name, argument);
    }
}
