import { toString as paramsToString } from '../util/params';
import { playerHost as settingsPlayerHost } from '../settings';
import { EmbedClient } from './client';
import { PLAYBACK_PAUSED, PLAYBACK_ENDED } from '../state-tracker';
import { METHOD_SET_CHANNEL, METHOD_SET_MUTE, METHOD_PAUSE,
         METHOD_PLAY, METHOD_SET_QUALITY, METHOD_SEEK,
         METHOD_SET_VIDEO, METHOD_SET_VOLUME, METHOD_DESTROY } from './host';

// Embed Version 0.0

export class PlayerEmbed {
    constructor(root, options) {
        this._root = root;
        // Convert the options into a query param.
        const params = paramsToString(_.omit(options, 'width', 'height'));
        const url = `${settingsPlayerHost}/?${params}`;

        this._iframe = document.createElement('iframe');
        this._iframe.setAttribute('src', url);

        if (options.width) {
            this._iframe.setAttribute('width', options.width);
        }

        if (options.height) {
            this._iframe.setAttribute('height', options.height);
        }

        this._iframe.setAttribute('frameBorder', '0');
        this._iframe.setAttribute('allowFullScreen', '');
        this._iframe.setAttribute('scrolling', 'no');

        if (_.isString(this._root)) {
            // Support passing in an ID of the root.
            this._root = document.getElementById(this._root);
        }

        this._root.appendChild(this._iframe);

        this._bridge = new EmbedClient(this._iframe.contentWindow);
    }

    // Embed API Function Calls

    // Embed Playback Controls Functions
    play() {
        this._bridge.callPlayerMethod(METHOD_PLAY);
    }

    pause() {
        this._bridge.callPlayerMethod(METHOD_PAUSE);
    }

    seek(time) {
        this._bridge.callPlayerMethod(METHOD_SEEK, time);
    }

    setVolume(volume) {
        this._bridge.callPlayerMethod(METHOD_SET_VOLUME, volume);
    }

    setMuted(muted) {
        this._bridge.callPlayerMethod(METHOD_SET_MUTE, muted);
    }

    setChannel(channel) {
        this._bridge.callPlayerMethod(METHOD_SET_CHANNEL, channel);
    }

    setVideo(video) {
        this._bridge.callPlayerMethod(METHOD_SET_VIDEO, video);
    }

    setQuality(quality) {
        this._bridge.callPlayerMethod(METHOD_SET_QUALITY, quality);
    }

    // Embed Event Handling Functions

    addEventListener(name, callback) {
        this._bridge.addEventListener(name, callback);
    }

    removeEventListener(name, callback) {
        this._bridge.removeEventListener(name, callback);
    }

    // Embed Player Status Functions

    getChannel() {
        return this._bridge.getPlayerState().channelName;
    }

    getCurrentTime() {
        return this._bridge.getPlayerState().currentTime;
    }

    getDuration() {
        return this._bridge.getPlayerState().duration;
    }

    getEnded() {
        return this._bridge.getPlayerState().playback === PLAYBACK_ENDED;
    }

    getMuted() {
        return this._bridge.getPlayerState().muted;
    }

    getPlaybackStats() {
        return this._bridge.getPlayerState().stats;
    }

    isPaused() {
        return this._bridge.getPlayerState().playback === PLAYBACK_PAUSED;
    }

    getQuality() {
        return this._bridge.getPlayerState().quality;
    }

    getQualities() {
        return this._bridge.getPlayerState().qualitiesAvailable;
    }

    getViewers() {
        return this._bridge.getPlayerState().viewers;
    }

    getVideo() {
        return this._bridge.getPlayerState().videoID;
    }

    getVolume() {
        return this._bridge.getPlayerState().volume;
    }

    // Properties specifically for Dashboard

    get channel() {
        return this.getChannel();
    }

    get muted() {
        return this.getMuted();
    }

    get viewers() {
        return this.getViewers();
    }

    // Misc

    destroy() {
        this._bridge.callPlayerMethod(METHOD_DESTROY);
        this._root.removeChild(this._iframe);
    }
}

// Make the class global as 'Twitch.embed.Player' just for v0
window.Twitch = window.Twitch || {};
window.Twitch.embed = window.Twitch.embed || {};
window.Twitch.embed.Player = PlayerEmbed;
