import { toString as paramsToString } from '../util/params';
import { playerHost as settingsPlayerHost } from '../settings';
import { EmbedClient } from './client';
import { PLAYBACK_PAUSED, PLAYBACK_ENDED } from '../state-tracker';
import { METHOD_SET_CHANNEL, METHOD_SET_MUTE, METHOD_PAUSE,
         METHOD_PLAY, METHOD_SET_QUALITY, METHOD_SEEK,
         METHOD_SET_VIDEO, METHOD_SET_VOLUME } from './host';

// Embed Version 1.0
// API Here: https://github.com/justintv/Twitch-API/blob/master/embed-video.md

export class PlayerEmbed {
    constructor(root, options) {
        this._root = root;
        // Convert the options into a query param.
        const params = paramsToString(_.omit(options, 'width', 'height'));
        const url = `${settingsPlayerHost}/?${params}`;

        // Create our iframe.
        const iframe = document.createElement('iframe');
        iframe.setAttribute('src', url);

        if (options.width) {
            iframe.setAttribute('width', options.width);
        }

        if (options.height) {
            iframe.setAttribute('height', options.height);
        }

        iframe.setAttribute('frameBorder', '0');
        iframe.setAttribute('allowFullScreen', '');
        iframe.setAttribute('scrolling', 'no');

        if (_.isString(this._root)) {
            // Support passing in an ID of the root.
            this._root = document.getElementById(this._root);
        }

        this._root.appendChild(iframe);

        this._bridge = new EmbedClient(iframe.contentWindow);
    }

    // Embed API Function Calls

    // Embed Playback Controls Functions
    play() {
        this._bridge.callPlayerMethod(METHOD_PLAY);
    }

    pause() {
        this._bridge.callPlayerMethod(METHOD_PAUSE);
    }

    seek(time) {
        this._bridge.callPlayerMethod(METHOD_SEEK, time);
    }

    setVolume(volume) {
        this._bridge.callPlayerMethod(METHOD_SET_VOLUME, volume);
    }

    setMuted(muted) {
        this._bridge.callPlayerMethod(METHOD_SET_MUTE, muted);
    }

    setChannel(channel) {
        this._bridge.callPlayerMethod(METHOD_SET_CHANNEL, channel);
    }

    setVideo(video) {
        this._bridge.callPlayerMethod(METHOD_SET_VIDEO, video);
    }

    setQuality(quality) {
        this._bridge.callPlayerMethod(METHOD_SET_QUALITY, quality);
    }

    // Embed Event Handling Functions

    addEventListener(name, callback) {
        this._bridge.addEventListener(name, callback);
    }

    removeEventListener(name, callback) {
        this._bridge.removeEventListener(name, callback);
    }

    // Embed Player Status Functions

    getChannel() {
        return this._bridge.getPlayerState().channelName;
    }

    getCurrentTime() {
        return this._bridge.getPlayerState().currentTime;
    }

    getDuration() {
        return this._bridge.getPlayerState().duration;
    }

    getEnded() {
        return this._bridge.getPlayerState().playback === PLAYBACK_ENDED;
    }

    getMuted() {
        return this._bridge.getPlayerState().muted;
    }

    getPlaybackStats() {
        return this._bridge.getPlayerState().stats;
    }

    isPaused() {
        return this._bridge.getPlayerState().playback === PLAYBACK_PAUSED;
    }

    getQuality() {
        return this._bridge.getPlayerState().quality;
    }

    getQualities() {
        return this._bridge.getPlayerState().qualitiesAvailable;
    }

    getViewers() {
        return this._bridge.getPlayerState().viewers;
    }

    getVideo() {
        return this._bridge.getPlayerState().videoID;
    }

    getVolume() {
        return this._bridge.getPlayerState().volume;
    }
}

// Make the class global as 'Twitch.Player'
window.Twitch = window.Twitch || {};
window.Twitch.Player = PlayerEmbed;
