import { EVENT_STATE_UPDATE, EVENT_PLAYER_UPDATE } from '../state-tracker';

// Available Bridge this._player Method Calls
export const METHOD_PLAY = 'play';
export const METHOD_PAUSE = 'pause';
export const METHOD_SET_CHANNEL = 'channel';
export const METHOD_SET_VIDEO = 'video';
export const METHOD_SEEK = 'seek';
export const METHOD_SET_QUALITY = 'quality';
export const METHOD_SET_MUTE = 'mute';
export const METHOD_SET_VOLUME = 'volume';
export const METHOD_DESTROY = 'destroy';

// Intra Bridge Communication
export const BRIDGE_REQ_SUBSCRIBE = 'subscribe';
export const BRIDGE_HOST_READY = 'ready';
export const BRIDGE_STATE_UPDATE = 'bridgestateupdate';
export const BRIDGE_PLAYER_EVENT = 'bridgeplayerevent';
export const BRIDGE_DOCUMENT_EVENT = 'bridgedocumentevent';
export const BRIDGE_HOST_NAMESPACE = 'player.embed.host';
export const BRIDGE_CLIENT_NAMESPACE = 'player.embed.client';

// Document events to be passed through bridge
const EMBED_DOCUMENT_EVENTS = [
    'blur',
    'focus',
    'keydown',
    'keypress',
    'keyup',
    'resize',
];

export class EmbedHost {
    constructor(player, state) {
        this._player = player;
        this._clients = [];
        this._initWindowEvents();

        // Pass all state updates from statetracker to clients
        state.addEventListener(EVENT_STATE_UPDATE, this._sendPlayerState.bind(this));
        state.addEventListener(EVENT_PLAYER_UPDATE, this._sendPlayerEvent.bind(this));

        // Start listening for any events.
        window.addEventListener('message', this);

        // If we were opened as an iframe or popup, inform the client.
        const frame = window.opener || window.parent;
        if (frame && frame !== window) {
            this._addClient(frame);
        }
    }

    _initWindowEvents() {
        // Forward certain document events to the client.
        // Mostly use this for key press events, otherwise inaccessable.
        _.each(EMBED_DOCUMENT_EVENTS, name => {
            // NOTE: document not root otherwise we miss events.
            document.addEventListener(name, event => {
                // Remove any complicated and non-clonable types.
                const payload = _.omit(event, _.isObject);
                this._sendAll(BRIDGE_DOCUMENT_EVENT, [name, payload]);
            });
        });
    }

    _addClient(client) {
        this._clients.push(client);
        this._send(client, BRIDGE_HOST_READY);
    }

    _send(client, method, args = []) {
        if (!Array.isArray(args)) {
            args = [args];
        }

        // Set the namespace so only clients can read this message.
        // This prevents us from accidentally talking to ourselves.

        const payload = {
            namespace: BRIDGE_CLIENT_NAMESPACE,
            method: method,
            args: args,
        };

        client.postMessage(payload, '*');
    }

    _sendAll(method, args) {
        _.each(this._clients, client => {
            this._send(client, method, args);
        });
    }

    handleEvent(event) {
        // This function is fired any time the window receives a message.
        // Other components use this API so we must check if the message
        // is actually intended for the player.
        // It is possible to have a client and server listening on the same
        // window, so we must also avoid talking to ourselves.

        // Make sure we can parse the message.
        if (!_.isObject(event.data)) {
            return;
        }

        // Only handle messages targeting the host.
        if (event.data.namespace !== BRIDGE_HOST_NAMESPACE) {
            return;
        }

        switch (event.data.method) {
        case BRIDGE_REQ_SUBSCRIBE:
            this._addClient(event.source);
            break;
        case METHOD_PLAY:
            this._player.play();
            break;
        case METHOD_PAUSE:
            this._player.pause();
            break;
        case METHOD_SET_CHANNEL:
            this._player.setChannel(event.data.args[0]);
            break;
        case METHOD_SET_VIDEO:
            this._player.setVideo(event.data.args[0]);
            break;
        case METHOD_SEEK:
            this._player.setCurrentTime(parseFloat(event.data.args[0]));
            break;
        case METHOD_SET_QUALITY:
            this._player.setQuality(event.data.args[0]);
            break;
        case METHOD_SET_MUTE:
            this._player.setMuted(!!event.data.args[0]);
            break;
        case METHOD_SET_VOLUME:
            this._player.setVolume(parseFloat(event.data.args[0]));
            break;
        case METHOD_DESTROY:
            this._player.destroy();
            break;
        }
    }

    _sendPlayerState(playerState) {
        this._sendAll(BRIDGE_STATE_UPDATE, playerState.toJSON());
    }

    _sendPlayerEvent(event) {
        this._sendAll(BRIDGE_PLAYER_EVENT, event);
    }
}
