import i18next from 'i18next';
import XHR from 'i18next-xhr-backend';
import { playerHost } from './settings';

const LANG_TO_COUNTRY_CODE = {
    ar: 'ar-SA',
    bg: 'bg-BG',
    cs: 'cs-CZ',
    da: 'da-DK',
    de: 'de-DE',
    el: 'el-GR',
    en: 'en-US',
    es: 'es-ES',
    fi: 'fi-FI',
    fr: 'fr-FR',
    hi: 'hi-IN',
    hu: 'hu-HU',
    it: 'it-IT',
    ja: 'ja-JP',
    ko: 'ko-KR',
    nl: 'nl-NL',
    no: 'no-NO',
    pl: 'pl-PL',
    pt: 'pt-PT',
    ro: 'ro-RO',
    ru: 'ru-RU',
    sk: 'sk-SK',
    sv: 'sv-SE',
    th: 'th-TH',
    tr: 'tr-TR',
    vi: 'vi-VN',
    zh: 'zh-CN',
};

i18next.use(XHR).init({
    fallbackLng: 'en-US',
    load: 'currentOnly',
    backend: {
        crossDomain: true,
        loadPath: `${playerHost}/localization/{{lng}}/strings.json`,
    },
    nsSeparator: false,
    keySeparator: false,
    // TODO we cannot pre-seed the en-US translations at this time, because i18next
    // will incorrectly prevent us from requesting other languages lazily.
    // See https://github.com/i18next/i18next/issues/617
    /*
    resources: {
        'en-US': {
            'translation': require('../../localization/en-US/strings.json')
        }
    }
    */
});

class i18n {
    constructor(translate) {
        this._translate = translate;
    }

    translate(msgid, options = {}) {
        let translation = this._translate(msgid, options);
        if (options.replacements) {
            Object.keys(options.replacements).map(function(key) {
                translation = translation.replace(key, options.replacements[key]);
            });
        }

        return translation;
    }
}

export class Nulli18n {
    translate(msgid, options = {}) { // eslint-disable-line no-unused-vars
        return msgid;
    }
}

export function getI18N(language) {
    language = LANG_TO_COUNTRY_CODE[language] || language;

    return new Promise(function(resolve, reject) {
        i18next.changeLanguage(language, (err, t) => {
            if (err) {
                reject(err);
            } else {
                resolve(new i18n(t));
            }
        });
    });
}
