require('es6-promise').polyfill();

import { Analytics } from './analytics/analytics';
import { AnalyticsTracker } from './analytics/tracker';
import * as params from './util/params';
import { isTwitchEmbed } from './util/twitch-embed';
import * as Settings from './settings';
import * as store from 'store';
import * as PlayerType from './util/player-type';
import * as cookie from 'cookie_js';
import * as DeviceId from './util/device-id';
import * as expClient from './experiments';
import { State } from './state-tracker';
import * as Timestamp from './util/timestamp';
import { Video } from './video';
import { forwardProperties } from './util/forward-properties';
import { EmbedHost, METHOD_PAUSE, METHOD_PLAY } from './embed/host';
import { EmbedClient } from './embed/client';
import { PlayerHotkeys } from './hotkeys';
import { PlayerUI } from './ui/main';
import { BackendMse } from './backend/mse';
import { BackendHls } from './backend/hls';
import { PubSub } from './pubsub';
import { userInfo } from './api';
import { FullScreen } from './util/fullscreen';

require('../sass/player.sass');
require('jquery-ui/ui/widgets/slider');
// TODO VP-365 Remove this monkey patch after the cutoff date
require('./util/https').patch(store);

export function Player(root, options) {
    var self = this;

    let embedClient;
    let experiments;
    let fullscreen;
    let pubSub;
    let state;
    let tracker;
    let video;
    let playerUI;

    function getLanguage(fallbackLang) {
        return userInfo().then(function (userInfo) {
            store.set('playerLanguage', userInfo.received_language);
            return userInfo.received_language;
        }, function () {
            return (
                cookie.get('language', null) ||
                (navigator.languages && navigator.languages[0]) ||
                navigator.language ||
                navigator.userLanguage ||
                fallbackLang
            );
        });
    }

    var init = function() {
        // Root DOM element.
        if (typeof root === 'string' || root instanceof String) {
            // Support passing in an ID of the root.
            root = document.getElementById(root);
        }

        // Initialize the options to their defaults.
        options = initDefaults();

        // Initialize the experiments service client
        experiments = expClient.createClient({
            login: cookie.get('login') || null,
            deviceID: DeviceId.get(false),
        });

        // PubSub Client
        pubSub = new PubSub(options);
        pubSub.addEventListener('online', onChannelOnline);

        // Create the analytics tracker for sending events to spade/mixpanel
        tracker = new AnalyticsTracker(options);

        // Create fullscreen utility
        fullscreen = new FullScreen(root);

        // Create the video object.
        video = new Video(root, tracker, experiments, pubSub, options);

        // Create the state tracker, pass in backend.
        state = new State(video, pubSub, fullscreen, experiments, options);
        if (options.debug) {
            window.state = state;
        }

        // Forward any calls on this Player to the Video object.
        //   ex. player.setVolume -> video.setVolume
        // TODO Explicitly define the interface instead of forwarding.
        forwardProperties(self, video);

        // Use a seperate module to track video events.
        new Analytics(video, tracker, experiments, options);

        // Once the analytics module is ready, start playing the video.
        // We do this before the UI has loaded because 1) the UI is more
        // likely to contain a syntax error and 2) the UI is async anyway.
        initVideo(options);

        // Create the embed API that uses postMessage.
        new EmbedHost(video, state);

        // Listens for key presses and sends commands to the player.
        new PlayerHotkeys(video, root, options);

        // Check if we are a popup.
        if (window.opener) {
            // This allows us to control our parent.
            embedClient = new EmbedClient(window.opener);

            // Pause the parent and resume it when we exit.
            // TODO Maybe only pause when the video has actually started.
            embedClient.callPlayerMethod(METHOD_PAUSE);

            window.onbeforeunload = function() {
                embedClient.callPlayerMethod(METHOD_PLAY);
            };
        }

        if (options.controls) {
            // Create the HTML UI last.
            playerUI = new PlayerUI(video, root, tracker, experiments, state, options);
        }

        // Use language from local storage temporarily until we get a response from /api/me
        const storedLang = store.get('playerLanguage');
        if (storedLang) {
            state.setLang(storedLang);
        }

        getLanguage(options.lang).then(function (language) {
            if (language !== storedLang) {
                state.setLang(language);
            }
        });

    };

    var onChannelOnline = function() {
        video.onChannelOnline();
    };

    // Initialize the options to their defaults.
    var initDefaults = function() {
        if (_.isString(options)) {
            options = params.parse(options);
        }
        if (!isTwitchEmbed()) {
            // for non-Twitch embeds, accept only a whitelist of properties
            options = _.pick(options, Settings.embedParameters);
        }

        // Force the backend based on some overrides.
        if (options.flash) {
            options.backend = 'flash';
        } else if (options.mse) {
            options.backend = 'mse';
        } else if (options.hls) {
            options.backend = 'hls';
        }

        // The HTML5 option uses either HLS or MSE if supported.
        if (options.html5 && BackendHls.canPlay()) {
            options.backend = 'hls';
        } else if (options.html5 && BackendMse.canPlay()) {
            options.backend = 'mse';
        }

        // Rename some parameters for backwards compatibility.
        options = _.defaults(options, {
            time: options.t,
        });

        // Start by using localstorage values if available.
        options = _.defaults(options, {
            volume: store.get('volume'),
            muted: store.get('muted'),
            quality: store.get('quality'),
            backend: store.get('backend'),
            player: PlayerType.getPlayerType(),
        });

        // Some default values.
        options = _.defaults(options, {
            volume: 0.5,
            muted: false,
            controls: true,
            autoplay: true,
            quality: 'medium',
            time: null,

            leaveDialogEnabled: Settings.leaveDialog.enabled,
            leaveDialogViewerThreshold: Settings.leaveDialog.viewerThreshold,
            refreshWarningEnabled: Settings.leaveDialog.enabled,

            lang: Settings.defaultLanguage,
        });

        // Add the player to the tracking events too.
        options.tracking = _.defaults({}, options.tracking, {
            player: options.player,
        });

        return options;
    };

    // Set up the video given parameters.
    var initVideo = function(options) {
        video.setVolume(options.volume);
        video.setMuted(options.muted);

        if (options.quality) {
            video.setQuality(options.quality);
        }

        if (options.channel) {
            video.setChannel(options.channel);
        } else if (options.video) {
            video.setVideo(options.video);
        }

        if (options.autoplay) {
            video.play();
        }

        if (options.time) {
            var time = Timestamp.parse(options.time);
            if (video.readyState > video.HAVE_NOTHING) {
                video.setCurrentTime(time);
            } else {
                video.addEventListener('loadedmetadata', function() {
                    video.setCurrentTime(time);
                });
            }
        }

        if (options.debug) {
            _.each(Settings.allEvents, function(name) {
                if (!_.contains(Settings.debugIgnoreEvents, name)) {
                    video.addEventListener(name, function() {
                        console.log('video event: ', name);
                    });
                }
            });

        }

        video.addEventListener('error', function() {
            console.error('video error:', video.getError());
        });
    };

    self.setLanguage = function(language) {
        state.setLang(language);
    };

    init();

    self.destroy = function() {
        playerUI.destroy();
        video.destroy();
        state.destroy();
    };
}

window.Twitch = window.Twitch || {};
window.Twitch.video = window.Twitch.video || {};
window.Twitch.video.Player = Player;
window.Twitch.Player = Player;
