import EventEmitter from 'event-emitter';
import PubSubClient from 'pubsub-js-client';
import * as Settings from './settings';

export const EVENT_ONLINE = 'online';
export const EVENT_OFFLINE = 'offline';
export const EVENT_VIEWCOUNT = 'viewerschange';
export const EVENT_TOS = 'tos';
export const AVAILABLE_EVENTS = [EVENT_ONLINE, EVENT_OFFLINE, EVENT_VIEWCOUNT, EVENT_TOS];

export class PubSub {
    constructor(options) {
        this._eventEmitter = new EventEmitter();
        this._options = options;
        this._isListening = false;
        this._onMessage = msg => {
            this.onPubSubMessage(msg);
        };

        // PubSub is only required on live content playback at the moment.
        if (options.video) {
            return;
        }

        this._pubSub = PubSubClient.getInstance(Settings.pubSubEnviroment);
        this._pubSub.Listen({
            topic: `video-playback.${options.channel}`,
            success: () => {
                this._isListening = true;
                if (options.debug) {
                    console.info(`PubSub: successfully listened to video-playback.${options.channel}`);
                }
            },
            failure: err => {
                console.error(`PubSub: error listening: ${err}`);
            },
            message: this._onMessage,
        });
    }

    /**
     * Called on new message from Twitch PubSub connection.
     *
     * @param {Object} msg
     */
    onPubSubMessage(msg) {
        msg = JSON.parse(msg);

        if (this._options.debug) {
            console.info(`PubSub: ${JSON.stringify(msg)}`);
        }

        switch (msg.type) {
        case 'stream-up':
            this._eventEmitter.emit(EVENT_ONLINE);
            break;

        case 'stream-down':
            this._eventEmitter.emit(EVENT_OFFLINE);
            break;

        case 'viewcount':
            this._eventEmitter.emit(EVENT_VIEWCOUNT, {
                viewers: msg.viewers,
            });
            break;

        case 'tos-strike':
            this._eventEmitter.emit(EVENT_TOS);
            break;

        default:
            if (this._options.debug) {
                console.warn(`PubSub: Invalid message - ${JSON.stringify(msg)}`);
            }
        }
    }

    /**
     * Add event listener
     */
    addEventListener(event, callback) {
        if (_.contains(AVAILABLE_EVENTS, event)) {
            this._eventEmitter.addListener(event, callback);
        }
    }

    /**
     * Remove event listener
     */
    removeEventListener(event, callback) {
        this._eventEmitter.removeListener(event, callback);
    }

    /**
     * Destroy the PubSub client.
     */
    destroy() {
        if (!this._isListening) {
            return;
        }

        // Unlisten the PubSubClient
        this._pubSub.Unlisten({
            topic: `video-playback.${this._options.channel}`,
            // Callback when the Driver has successfully unlistened on the topic
            success: () => {
                this._isListening = false;
                if (this._options.debug) {
                    console.info(`PubSub: successfully unlistened to video-playback.${this._options.channel}`);
                }
            },
            // callback when there was an error unlistening
            failure: err => {
                console.error('PubSub: error unlistening: %o', err);
            },
            // A reference to the callback used in the initial Listen(), to specify which callback to remove
            message: this._onMessage,
        });
    }

}
