import EventEmitter from 'event-emitter';
import { TimelineMetadata } from './timeline';
import { MARKERS } from './experiments';
import * as Settings from './settings';
import { getI18N, Nulli18n } from './i18n';
import { BackendChromecast } from './backend/chromecast';
import * as PlayerApi from './api';
import * as Store from 'store';
import SessionStorage from './util/session-storage-json';
import { EVENT_ONLINE, EVENT_OFFLINE, EVENT_VIEWERS_CHANGE } from './pubsub';
import { FULLSCREEN_CHANGE } from './util/fullscreen';

// TODO VP-365 Remove this monkey patch after the cutoff date
require('./util/https').patch(Store);

// Types of events outputted by state tracker
export const EVENT_STATE_UPDATE = 'stateupdate';
export const EVENT_PLAYER_UPDATE = 'player';
export const EVENT_LANGUAGE_CHANGED = 'languagechanged';

// States of player playback
export const PLAYBACK_PAUSED = 'paused';
export const PLAYBACK_PLAYING = 'playing';
export const PLAYBACK_ENDED = 'ended';

// States of Chromecast
export const CHROMECAST_UNAVAILABLE = 'unavailable';
export const CHROMECAST_AVAILABLE = 'available';
export const CHROMECAST_CONNECTING = 'connecting';
export const CHROMECAST_CONNECTED = 'connected';

// Default state for Embed
export const DEFAULT_EMBED_STATE = {
    channelName: '',
    currentTime: 0,
    duration: 0,
    muted: false,
    playback: '',
    quality: '',
    qualitiesAvailable: [],
    stats: {}, //TODO: export default stat object from video backend
    videoID: '',
    viewers: 0,
    volume: 0,
};

// Browser Storage Related Keys
const KEY_MATURE = 'mature';
const KEY_VOD_RESUME_TIMES = 'vodResumeTimes';
const KEY_VOD_RESUME_WATCHEDS = 'vodResumeWatcheds';
const KEY_SESSION_LEAVE_DATA = 'leaveData';

// State tracker events available to be listened to
const AVAILABLE_EVENTS = [EVENT_STATE_UPDATE, EVENT_PLAYER_UPDATE, EVENT_LANGUAGE_CHANGED];

// Backend events used by state tracker
export const EVENT_AD_START = 'adstart';
export const EVENT_AD_END = 'adend';
const EVENT_LOADED_CHANNEL = 'loadedchannel';
const EVENT_LOADED_METADATA = 'loadedmetadata';
const EVENT_LOAD_START = 'loadstart';
const EVENT_PLAYING = 'playing';
const EVENT_USHER_FAIL = 'usherfail';
export const EVENT_CAPTION_UPDATE = 'captions';

// Internal logging events
const EVENT_MARKERS_CHANGE = 'markerschange';

/**
 * State
 * Keeps track of the state of the player and passes through backend event.
 * One stop shop for all your state needs!
 * @constructor
 */
export class State {
    constructor(backend, pubSub, fullscreen, experiments, options) {
        this._options = options;
        this._backend = backend;
        this._experiments = experiments;
        this._pubSub = pubSub;
        this._timelineMetadata = new TimelineMetadata();
        this._fullscreen = fullscreen;

        this._eventEmitter = new EventEmitter();
        this._i18n = new Nulli18n();

        // Properties not proxied, derived, or retrieved from kraken API
        this._adrunning = false;

        this._theatreModeEnabled = false;
        this._isVODRestricted = false;
        this._hasCaptions = false;
        this._captionsEnabled = false;

        // Channel Online status
        this._online = false;
        this._viewers = 0;
        this._markers = [];
        this._mutedSegments = [];

        // Listen to all events from backend
        _.each(Settings.allEvents, function(event) {
            backend.addEventListener(event, this.handleEvent.bind(this, event));
        }, this);

        fullscreen.addEventListener(FULLSCREEN_CHANGE, this.handleEvent.bind(this, FULLSCREEN_CHANGE));

        // Listen to all configured PubSub events
        pubSub.addEventListener(EVENT_VIEWERS_CHANGE, this._onViewersChange.bind(this));
        pubSub.addEventListener(EVENT_ONLINE, this.handleEvent.bind(this, EVENT_ONLINE));
        pubSub.addEventListener(EVENT_OFFLINE, this.handleEvent.bind(this, EVENT_OFFLINE));
    }

    // All state properties, (State Machines, derived, and proxied) defined at end of file
    onViewersChange(data) {
        this._viewers = data.viewers;
        if (this._backend.onViewersChange) {
            this._backend.onViewersChange(data);
        }
        this._eventEmitter.emit(EVENT_STATE_UPDATE, this);
    }

    // Getter & Setter Functions

    get isAdRunning() {
        return this._adrunning;
    }

    get autoplayEnabled() {
        return this._backend.getAutoplay();
    }

    get buffered() {
        return this._backend.getBuffered();
    }

    get castingDevice() {
        return BackendChromecast.getDevice();
    }

    get castingState() {
        if (BackendChromecast.getReadyState() !== BackendChromecast.HAVE_NOTHING) {
            return CHROMECAST_CONNECTED;
        }

        switch (BackendChromecast.getNetworkState()) {
        case BackendChromecast.NETWORK_EMPTY:
            return CHROMECAST_UNAVAILABLE;
        case BackendChromecast.NETWORK_IDLE:
            return CHROMECAST_AVAILABLE;
        default:
            return CHROMECAST_CONNECTED;
        }
    }

    get channelName() {
        return this._backend.getChannel();
    }

    get currentBackend() {
        return this._backend.getBackend();
    }

    get currentTime() {
        return this._backend.getCurrentTime();
    }

    get duration() {
        return this._backend.getDuration();
    }

    get error() {
        return this._backend.error;
    }

    get i18n() {
        return this._i18n;
    }

    canFullScreen() {
        return this._fullscreen.canFullScreen();
    }

    isFullScreen() {
        return this._fullscreen.isFullScreen();
    }

    get isLoading() {
        return (this.playback === PLAYBACK_PLAYING && this._backend.getReadyState() <= this._backend.HAVE_CURRENT_DATA);
    }

    getMarkers() {
        return this._markers;
    }

    isMature() {
        return Store.get(KEY_MATURE, false);
    }

    setIsMature(isMature) {
        Store.set(KEY_MATURE, isMature);
    }

    get muted() {
        return this._backend.getMuted();
    }

    getMutedSegments() {
        return this._mutedSegments;
    }

    get online() {
        return this._online;
    }

    get playback() {
        if (this._backend.getPaused()) {
            return PLAYBACK_PAUSED;
        } else if (this._backend.getEnded()) {
            return PLAYBACK_ENDED;
        } else {
            return PLAYBACK_PLAYING;
        }
    }

    get quality() {
        return this._backend.getQuality();
    }

    get qualitiesAvailable() {
        return this._backend.getQualities();
    }

    get sessionLeaveData() {
        return SessionStorage.getItem(KEY_SESSION_LEAVE_DATA);
    }

    setSessionLeaveData(leaveData) {
        SessionStorage.setItem(KEY_SESSION_LEAVE_DATA, leaveData);
    }

    get isSeeking() {
        return this._backend.getSeeking();
    }

    get isSpectre() {
        return this._backend.isSpectre();
    }

    get stats() {
        return this._backend.getStats();
    }

    get supportedBackends() {
        return this._backend.getSupportedBackends();
    }

    get theatreEnabled() {
        return this._theatreModeEnabled;
    }

    setTheatreEnabled(isEnabled) {
        this._theatreModeEnabled = isEnabled;
        this._eventEmitter.emit(EVENT_STATE_UPDATE, this);
    }

    get viewers() {
        return this._viewers;
    }

    get videoID() {
        return this._backend.getVideo();
    }

    get videoURL() {
        return this._backend.getVideoURL();
    }

    // TODO: This information needs to be resurfaced correctly; it is now disabled
    // due to conflicts with a duplicate method below.
    /*
    get videoInfo() {
        return this._backend.getVideoInfo();
    }
    */

    getCaption() {
        if (this.getCaptionsEnabled() && !this._adrunning) {
            return this._backend.getCaption();
        }
    }

    hasCaptions() {
        return this._hasCaptions;
    }

    isVODRestricted() {
        return this._isVODRestricted;
    }

    get vodResumeTimes() {
        return Store.get(KEY_VOD_RESUME_TIMES, {});
    }

    get vodResumeWatcheds() {
        return Store.get(KEY_VOD_RESUME_WATCHEDS, {});
    }

    setVodResumeTimes(resumeTimes) {
        Store.set(KEY_VOD_RESUME_TIMES, resumeTimes);
    }

    setVodResumeWatcheds(resumeWatcheds) {
        Store.set(KEY_VOD_RESUME_WATCHEDS, resumeWatcheds);
    }

    get volume() {
        return this._backend.getVolume();
    }

    // Kraken API Calls - Returns Promise
    // TODO: All of this will be encompassed in our polymorphic object approach to
    // storing video/stream info, instead of separate repeated calls to our Player API (kraken calls)

    getChannelInfo(channelName) {
        return PlayerApi.channelInfo(channelName);
    }

    getStreamInfo(channelName) {
        return PlayerApi.streamInfo(channelName);
    }

    getVideoInfo(videoID) {
        return PlayerApi.videoInfo(videoID);
    }

    getChannelViewerInfo(channelName) {
        return PlayerApi.channelViewerInfo(channelName);
    }

    getOfflinePlaylistInfo(channelName) {
        return this.getChannelInfo(channelName).then(function(data) {
            return PlayerApi.offlinePlaylistInfo(data._id);
        });
    }

    setCaptionsEnabled(bool) {
        this._captionsEnabled = bool;
        this._eventEmitter.emit(EVENT_STATE_UPDATE, this);
    }

    getCaptionsEnabled() {
        return this._captionsEnabled;
    }

    // TODO: Change event emitter structure to include payload so this internal method is not needed
    _onViewersChange(data) {
        this._viewers = data.viewers;
        this._eventEmitter.emit(EVENT_STATE_UPDATE, this);
    }

    /**
     * Update state of player based on events from backend
     * @private
     */
    handleEvent(event) {
        // TODO: Currently only a small handful states have been implemented. Need MOAR states.

        this._logStateChange(event);

        switch (event) {
        case EVENT_AD_START:
            this._adrunning = true;
            break;
        case EVENT_AD_END:
            this._adrunning = false;
            break;
        case EVENT_LOADED_CHANNEL:
                // Ensure loaded channels are always `online`.
                // PubSub will not indicate to the player that a stream is
                // online pending it was init'ed while the broadcast was
                // already running.
            this._online = true;
            break;
        case EVENT_ONLINE:
            this._online = true;
            break;
        case EVENT_OFFLINE:
            this._online = false;
            break;
        case EVENT_LOADED_METADATA:
            this._updateIsVODRestricted();
            break;
        case EVENT_LOAD_START:
            // Paywalled VODs do not emit EVENT_LOADED_METADATA from backend
            this._updateIsVODRestricted();
            break;
        case EVENT_CAPTION_UPDATE:
            this._hasCaptions = true;
            break;
        case EVENT_PLAYING:
            if (this.videoID) {
                // Dark Launch Experiment, controls retrieval of markers
                this._experiments.get(MARKERS).then(retrieveMarkers => {
                    if (retrieveMarkers === 'no') {
                        return;
                    }
                    // Dark Launch hardcodes the retrieval of 'v54109226' markers.
                    this._timelineMetadata.getMarkers(this.videoID).then(markers => {
                        this._markers = markers;
                        this._logStateChange(EVENT_MARKERS_CHANGE);
                        this._eventEmitter.emit(EVENT_STATE_UPDATE, this);
                    });
                });
            }
            break;
        case EVENT_USHER_FAIL:
            this._updateIsVODRestricted();
            break;
        }

        // Pass through backend event
        this._eventEmitter.emit(EVENT_PLAYER_UPDATE, event);

        // Update all listeners with new state
        this._eventEmitter.emit(EVENT_STATE_UPDATE, this);
    }

    /**
     * Add event listener
     */
    addEventListener(event, callback) {
        if (_.contains(AVAILABLE_EVENTS, event)) {
            this._eventEmitter.addListener(event, callback);
        }
    }

    /**
     * Remove event listener
     */
    removeEventListener(event, callback) {
        this._eventEmitter.removeListener(event, callback);
    }

    /**
     * ONLY FOR EMBED API
     * Post messaging only accepts regular JS objects.
     * Creates JS object for embed API with the public state properties
     * @return {Object} A javascript object with the public state properties
     */
    toJSON() {
        return {
            channelName: this.channelName,
            currentTime: this.currentTime,
            duration: this.duration,
            muted: this.muted,
            online: this.online, // TODO: will be eliminated
            playback: this.playback,
            quality: this.quality,
            qualitiesAvailable: this.qualitiesAvailable,
            stats: this.stats,
            videoID: this.videoID,
            viewers: this.viewers,
            volume: this.volume,
        };
    }

    /**
     * Logs state changes.
     */
    _logStateChange(event = null) {
        if (!this._options.debug || _.contains(Settings.debugIgnoreEvents, event) || event === null) {
            return;
        }

        console.groupCollapsed('state change: %s', event);
        console.log('partial state: %o', this.toJSON());
        console.groupEnd();
    }

    setLang(language) {
        getI18N(language).then(i18n => {
            this._i18n = i18n;
            this._eventEmitter.emit(EVENT_LANGUAGE_CHANGED, this);
        });
    }

    _updateIsVODRestricted() {
        this._isVODRestricted = false;
        this._eventEmitter.emit(EVENT_STATE_UPDATE, this);
        // TODO: Should only check videoID. Currently
        // flash backend does not null out videoID when it loads a
        // new channel while it is playing a VOD. VP-424
        if (this.videoID === null || this.channelName !== null) {
            return;
        }
        const videoInfo  = PlayerApi.videoInfo(this.videoID);
        const viewerInfo = videoInfo.then(info => PlayerApi.channelViewerInfo(info.channel.name));
        Promise.all([videoInfo, viewerInfo]).then(([videoInfo, viewerInfo]) => {
            this._isVODRestricted = PlayerApi.isVODRestricted(viewerInfo, videoInfo);
            this._eventEmitter.emit(EVENT_STATE_UPDATE, this);
        });
    }

    destroy() {
        AVAILABLE_EVENTS.map(event => {
            this._eventEmitter.removeEvent(event);
        });
    }
}
