import { isTwitchEmbed } from '../util/twitch-embed';
import * as PlayerType from '../util/player-type';
import Clipboard from 'clipboard/dist/clipboard';
import * as Settings from '../settings';
import { extendTimeout } from '../util/extend-timeout';
import * as Params from '../util/params';
import * as Timestamp from '../util/timestamp';
import { EVENT_STATE_UPDATE } from '../state-tracker';

export function PlayerUIControls(player, root, state, analytics, options) {
    let selectingVolume;
    let unmuteVolume;
    let manualQualityChange;
    let previousQuality;
    let hovering;
    let menuOpen;
    let mouseX;
    let mouseY;

    var init = function() {
        initDefaults();
        initPlayer();
        initDom();

        // Show the controls for 5 seconds.
        showControls(Settings.initialControlsDelay);

        // Set the initial fullscreen state.
        var fullscreenEnabled = player.getFullscreenEnabled();
        if (fullscreenEnabled) {
            var fullscreen = player.getFullscreen();
            $(root).attr('data-fullscreen', fullscreen);
        } else {
            $(root).attr('data-fullscreen', 'disabled');
        }

        $(root).attr('data-branding', options.branding);
        $(root).attr('data-advertisement', false);
        $(root).attr('data-theatre', false);
        $(root).attr('data-showtheatre', options.showtheatre);

        // We might have missed some events, so set up the initial state.
        onVolumeChange();
        onCastingChange();
        onBackendChange();

        state.addEventListener(EVENT_STATE_UPDATE, onStateUpdate);
    };

    var initDefaults = function() {
        // Force some values for non-twitch websites.
        if (!isTwitchEmbed()) {
            options = _.extend(options, {
                branding: true,
            });
        }
    };

    var initPlayer = function() {
        player.addEventListener('loadstart', onLoadStart);
        player.addEventListener('loadedmetadata', onLoadedMetadata);
        player.addEventListener('playing', onVideoPlaying);
        player.addEventListener('volumechange', onVolumeChange);
        player.addEventListener('qualitychange', onQualityChange);
        player.addEventListener('qualitieschange', onQualitiesChange);
        player.addEventListener('castingchange', onCastingChange);
        player.addEventListener('fullscreenchange', onFullscreenChange);
        player.addEventListener('theatrechange', onToggleTheatre);
        player.addEventListener('restricted', onRestricted);
        player.addEventListener('seeking', onSeeking);

        player.addEventListener('adstart', onAdStart);
        player.addEventListener('adend', onAdEnd);
    };

    var initDom = function() {
        if (options.player !== PlayerType.PLAYER_FRONTPAGE) {
            $(root).on('dblclick', '.js-control-fullscreen-overlay', function(e) {
                e.preventDefault();
                toggleFullscreen();
            });
        }

        $(root).on('click', '.js-control-playpause-button', togglePlay);
        $(root).on('click', '.js-control-play-button', togglePlay);
        $(root).on('click', '.js-control-fullscreen', toggleFullscreen);
        $(root).on('click', '.js-control-theatre', toggleTheatre);
        $(root).on('click', '.js-control-volume', toggleMute);

        $(root).on('click', '.js-chromecast-button', function() {
            var casting = player.getCasting();

            if (casting === 'available') {
                player.startCast();
            } else {
                player.stopCast();
            }
        });

        $(root).on('click', '.js-player-alert__close', function() {
            $(this).closest('.js-player-alert').attr('data-active', false);
        });

        $(root).on('mousemove', function(e) {
            // We keep track of the last position so we can ignore events with no movement.
            // This fixes a Chrome bug triggered by chat scrolling.
            if (mouseX === e.screenX && mouseY === e.screenY) {
                return;
            }

            mouseX = e.screenX;
            mouseY = e.screenY;

            showControls(Settings.hoverControlsDelay);
        });

        // Only auto-hide the controls if we are not hovering over them.
        $(root).on('mouseenter', '.js-controls-top, .js-controls-bottom', function() {
            hovering = true;
        });

        $(root).on('mouseleave', '.js-controls-top, .js-controls-bottom', function() {
            hovering = false;
        });

        $(root).on('mouseleave', function() {
            // Immediately hide controls.
            hideControls();
        });

        $(root).on('click', '.js-meta-url', function() {
            // They just opened a new window, pause the video.
            player.pause();
        });

        $('.js-volume-slider', root).slider({
            range: 'min',
            value: 100,
            min: 0,
            max: 100,
            slide(e, pos) {
                // Unmute the player if it is muted.
                var muted = player.getMuted();
                if (muted) {
                    player.setMuted(false);
                }

                var volume = pos.value / 100;
                player.setVolume(volume);
            },
            start() {
                selectingVolume = true;

                // Store the current volume in case we drag to 0.
                // If that happens, we want unmute to restore volume.
                unmuteVolume = player.getVolume();
            },
            stop(e, pos) {
                selectingVolume = false;

                // We don't need unmuteVolume unless we stopped at 0.
                var volume = pos.value / 100;
                if (volume > 0) {
                    unmuteVolume = null;
                }
            },
        });

        // Remove volume slider from the tab flow
        $('.js-volume-slider .ui-slider-handle').attr('tabindex', '-1');

        $(root).on('click', '.js-popout-player', function(e) {
            e.preventDefault();
            openPopout();
        });

        $(root).on('click', '.js-menu-button', toggleMenu);

        // On 'Copy URL at Current Time' mouse click.
        new Clipboard('.js-copy-url', {
            text() {
                // Create the deeplink URL to this video at the current playback time.
                const url = `${player.getVideoURL()}?t=${Timestamp.toURLString(player.getCurrentTime())}`;

                hideMenu();

                return url;
            },
        });

        // If a click event propagates all the way up, hide the menu.
        $(root).on('click', function(e) {
            // Hide the menu unless the menu itself was clicked.
            if (!$(e.target).closest('.js-menu, .js-menu-button').length) {
                hideMenu();
            }
        });

        // Hide the menu if our iframe loses focus.
        $(window).on('blur', hideMenu);

        // Change the quality.
        $(root).on('change', '.js-quality', function() {
            manualQualityChange = true;

            var quality = $('.js-quality').val();
            setQuality(quality);
        });

        // Submit the video issue report.
        $(root).on('submit', '.js-video-issue-form', submitVideoIssueReport);

        $(root).on('click', '.js-player-product-close', function(e) {
            e.preventDefault();
            hideProductOverlay();
        });

        $(root).on('click', '.js-html5-slider', onHtml5Toggle);
    };

    var onStateUpdate = function() {

    };

    var hideControlsTimeout = null;
    var showControls = function(duration) {
        $(root).attr('data-controls', true);

        if (duration) {
            // This is a custom function defined in util/extend-timeout.js
            // Schedules the new timeout instead if it will occur later.
            hideControlsTimeout = extendTimeout(hideControlsTimeout, hideControls, duration);
        }
    };

    var hideControls = function() {
        // Don't hide controls in some situations.
        if (hovering) return;
        if (menuOpen) return;

        // Do not hide the controls if Chromecasting
        var casting = player.getCasting();
        if (casting === 'connected' || casting === 'connecting') return;

        $(root).attr('data-controls', false);
    };

    var togglePlay = function() {
        if (player.getEnded() || player.getPaused()) {
            player.play();
        } else {
            player.pause();
        }
    };

    var toggleMute = function() {
        var muted = player.getMuted();

        // If we saved the volume to restore, explicitly set it.
        // Without this, dragging to 0 would cause unmute to use 0.001
        if (muted && unmuteVolume) {
            player.setVolume(unmuteVolume);
        } else {
            player.setMuted(!muted);
        }
    };

    var toggleFullscreen = function() {
        var fullscreen = player.getFullscreen();
        player.setFullscreen(!fullscreen);
    };

    var onLoadStart = function() {
        resetVideoIssueReport();
    };

    var onLoadedMetadata = function() {
        onQualitiesChange();

        // Display the copy VOD URL option pending this is VOD content.
        toggleCopyVODURLOption(Boolean(options.video));
    };

    var onVideoPlaying = function() {
        // Ensure playback quality is set in Quality Selector element.
        $('.js-quality', root).val(player.getQuality());
    };

    var showMenu = function() {
        menuOpen = true;
        $('.js-menu', root).attr('data-state', 'open');
        $('.js-menu-button', root).attr('data-state', 'menu-open');
    };

    var hideMenu = function() {
        menuOpen = false;
        $('.js-menu', root).attr('data-state', 'closed');
        $('.js-menu-button', root).attr('data-state', 'menu-closed');
    };

    var toggleMenu = function() {
        var open = $('.js-menu', root).attr('data-state') === 'open';
        if (open) {
            hideMenu();
        } else {
            showMenu();
        }
    };

    var setQuality = function(value) {
        if (!player.getPaused()) {
            // Only display the loading spinner pending the player isn't paused
            $('.js-quality-display-contain', root).attr('data-q', 'loading');
        }

        // Set the newly selected quality.
        player.setQuality(value);

        // Close the menu after selecting a quality.
        hideMenu();
    };

    var onQualityChange = function() {
        var quality = player.getQuality();

        previousQuality = quality;

        // Auto select the active quality
        $('.js-quality', root).val(quality);

        // Flash the selected quality
        var qualityText = state.i18n.translate(Settings.qualityText[quality]) || quality;
        $('.js-quality-display', root).text(qualityText);

        // Show the new quality animation.
        var display = $('.js-quality-display-contain', root);
        display.attr('data-q', 'show');

        setTimeout(function() {
            display.attr('data-q', 'hide');
        }, Settings.qualityChangeDuration);

        // Show the controls for the same amount of time
        showControls(Settings.qualityChangeDuration);

        manualQualityChange = false;
    };

    var onQualitiesChange = function() {
        var qualities = player.getQualities();
        var dropdown = $('.js-quality', root).empty();

        for (var i = 0; i < qualities.length; i++) {
            var quality = qualities[i];
            var qualityText = state.i18n.translate(Settings.qualityText[quality]) || quality;

            $('<option>').val(quality).text(qualityText).appendTo(dropdown);
        }

        var currentQuality = player.getQuality();
        dropdown.val(currentQuality);

        dropdown.prop('disabled', qualities.length < 2);
    };

    // Submit a Video Issue Report to the correct receiver
    var submitVideoIssueReport = function(e) {
        e.preventDefault();

        // Convert the form inputs into an object with name -> value.
        // TODO Move this to a helper file.
        var inputsArray = $(e.target).serializeArray();
        var inputs = _.reduce(inputsArray, function(fields, input) {
            fields[input.name] = input.value;
            return fields;
        }, {});

        if (!inputs.issue) {
            // abort if no issue selected
            return;
        }

        // Allocate the VideoInfo data and append the issue reason
        var properties = {
            issue: inputs.issue,
        };
        properties = _.extend(properties, player.getVideoInfo());

        // Send the tracking event to mixpanel/spade.
        analytics.trackEvent('vid_issue_report', properties);

        // Mark the form as complete and eventually hide the menu.
        $('.js-video-issue', root).attr('data-complete', true);
        setTimeout(hideMenu, Settings.reportHideDelay);
    };

    // Set the form back to it's original state.
    var resetVideoIssueReport = function() {
        $('.js-video-issue', root).attr('data-complete', false);
        $('.js-video-issue-form', root).get(0).reset();
    };

    var onHtml5Toggle = function() {
        var backend = player.getBackend();

        if (backend === 'flash') {
            var backends = player.getSupportedBackends();
            var nonFlash = _.without(backends, 'flash');

            // Switch to the first non-flash supported backend.
            backend = _.first(nonFlash);
        } else {
            backend = 'flash';
        }

        player.setBackend(backend);

        // TODO Should the player fire this as an event?
        onBackendChange();
    };

    var onBackendChange = function() {
        var backend = player.getBackend();
        var value = (backend === 'flash') ? 'off' : 'on';
        $('.js-html5-slider', root).attr('data-value', value);
    };

    // Flash cannot keep up with our volume changing because it queues events.
    // So we ignore the volumechange event and keep track of it ourselves.
    var onVolumeChange = function() {
        var muted = player.getMuted();
        var volume = player.getVolume();

        if (muted) {
            volume = 0;
        } else if (volume === 0) {
            muted = true;
        }

        $(root).attr('data-muted', muted);

        if (!selectingVolume) {
            // Only update the volume slider if we aren't dragging.
            $('.js-volume-slider', root).slider('value', volume * 100);
        }
    };

    var toggleTheatre = function() {
        var theatre = player.getTheatre();
        player.setTheatre(!theatre);
    };

    var onToggleTheatre = function() {
        var theatre = player.getTheatre();
        var tooltip = theatre ? 'Theater Mode' : 'Exit Theater Mode';

        $('.js-control-theatre .js-control-tip', root).attr('data-tip', tooltip);
        $(root).attr('data-theatre', !theatre);

        hovering = false;
        showControls(Settings.hoverControlsDelay);
    };

    var onFullscreenChange = function() {
        var fullscreen = player.getFullscreen();
        $(root).attr('data-fullscreen', fullscreen);

        var text = fullscreen ? 'Exit Fullscreen' : 'Fullscreen';
        $('.js-control-fullscreen .js-control-tip', root).attr('data-tip', text);
    };

    var onSeeking = function() {
        showControls(Settings.hoverControlsDelay);
    };

    var onCastingChange = function() {
        var casting = player.getCasting();

        // TODO: Make an icon/message for Chromecast errors.
        if (casting === 'error') {
            casting = 'available';
        }

        // Possible values: available unavailable connecting connected error
        $(root).attr('data-casting', casting);

        if (casting === 'connected') {
            // Set the clients Chromecast device name in the UI
            var device = player.getCastDevice();
            device = _.isString(device) ? device : 'Chromecast';
            $('.js-chromecast-device', root).text(device);
        }

        if (casting === 'connected' || casting === 'connecting') {
            $(root).attr('data-chromecast', 'true');

            showControls();
        } else {
            $(root).attr('data-chromecast', 'false');

            showControls(Settings.hoverControlsDelay);
        }
    };

    var onAdStart = function() {
        $(root).attr('data-advertisement', true);
    };

    var onAdEnd = function() {
        $(root).attr('data-advertisement', false);
    };

    // Opens a new popout to the current channel/video.
    var openPopout = function() {
        var options = {};
        options.volume = player.getVolume();

        var channel = player.getChannel();
        var video = player.getVideo();

        if (channel) {
            options.channel = channel;
        } else {
            options.video = video;

            var time = player.getCurrentTime();
            options.time = Timestamp.toURLString(time);
        }

        // Ensure initial playback is paused before opening
        // The popout container with the same stream content.
        player.pause();

        // Defaults to 480p
        const width = Settings.popoutSize.width;
        const height = Settings.popoutSize.height;

        const url = `${Settings.playerHost}/?${Params.toString(options)}`;
        let args = `width=${width},height=${height}`;

        // note: The address bar will still show if the origin is different.
        args += ',toolbar=no,menubar=no,scrollbars=no,location=no,status=no';

        window.open(url, '_blank', args);

        // note: The popout uses the embed API to control the player.
    };

    var onRestricted = function() {
        // This avoids showing the product screen on stream start.
        if (manualQualityChange) {
            showLiveQualityChansubOverlay();
            $('.js-quality', root).val(previousQuality);
            var display = $('.js-quality-display-contain', root);
            display.attr('data-q', 'hide');
            return;
        }

        // We weren't allowed to switch quality, show the old one.
        onQualityChange();
    };

    var showLiveQualityChansubOverlay = function() {
        $('.js-player-product-overlay', root).attr('data-active', true);
        $('.js-player-product-close', root).attr('data-active', true);

        let message = state.i18n.translate('This video quality is only available to subscribers. Subscribe now to watch and support %s.');
        message = message.replace('%s', options.channel);
        $('.js-player-product p', root).text(message);
        $('.js-player-product .purchase_button', root).attr('href', `http://www.twitch.tv/${options.channel}/subscribe?ref=chansub_overlay_subscribe`);
    };

    var hideProductOverlay = function() {
        $('.js-player-product-overlay', root).attr('data-active', false);
    };

    /**
     * Toggles the display state of the 'Copy VOD URL at Current Time'
     * option in the playback control menu.
     *
     * @param {Boolean} isVOD If the current playback is VOD.
     */
    var toggleCopyVODURLOption = function(isVOD) {
        $('.js-copyurl-contain', root).attr('data-active', isVOD);
    };

    /**
     *
     */
    this.destroy = function() {
        $(root).off('click');
        $(root).off('dblclick');
        $(root).off('mousemove');
        $(root).off('mouseleave');
        $(root).off('mouseenter');
        $(root).off('change');
        $(root).off('submit');
        $(window).off('blur');
    };

    init();
}
