import * as store from 'store';
import { Deferred } from '../util/deferred';
import * as PlayerType from '../util/player-type';
import { channelInfo, videoInfo, offlinePlaylistInfo,
         streamInfo, channelUrl, videoUrl } from '../api';
import * as Settings from '../settings';
import * as Timestamp from '../util/timestamp';
import { EVENT_STATE_UPDATE } from '../state-tracker';

// TODO VP-365 Remove this monkey patch after the cutoff date
require('../util/https').patch(store);

// Default image resoltuion to request for video background display.
const defaultVideoBackgroundResolution = '1280x720';

export function PlayerUIInfo(player, root, state, options) {
    // The video url requires an API request to figure out the channel.
    // TODO This will bug out if the channel is ever switched.
    const streamUrl = new Deferred();

    // Determine if the user has clicked on the mature button.
    let mature = store.get('mature');

    let online = false;
    let viewers = 0;

    // Whether we need to unpause the video after mature overlay disappears.
    let mutedSegments;
    let videoBanner;
    let videoThumbnail;

    // We need to know whether the channel was online to switch to spectre transitions.
    let channelWasOnline;
    let isSpectre;

    var init = function() {
        $(root).attr('data-showinfo', options.showInfo);

        $(root).on('click', '.js-player-mature-accept', function(e) {
            e.preventDefault();
            setMature();

            // Resume the playback.
            player.play();
            player.setMuted(false);
        });

        var playerType = options.player;
        $(root).attr('data-playertype', playerType);

        if (playerType === PlayerType.PLAYER_FRONTPAGE) {
            $(root).on('click', '.js-control-fullscreen-overlay', function() {
                // Wait for the URL to be calculated from the API response.
                streamUrl.promise.then(function(url) {
                    window.top.location.href = url;
                });
            });
        }

        $(root).on('click', '.js-watch-twitch', function(e) {
            e.preventDefault();

            streamUrl.promise.then(function(url) {
                // Append the current timestamp to URL for VODs only.
                var time = player.getCurrentTime();
                if (player.video && time) {
                    url += `?t=${Timestamp.toURLString(Math.round(time))}`;
                }

                window.open(url, '_blank');

                // Pause the player to avoid two copies playing.
                player.pause();
            });
        });

        player.addEventListener('loadedchannel', onLoadedChannel);
        player.addEventListener('loadedvideo', onLoadedVideo);
        player.addEventListener('ended', onEnded);
        player.addEventListener('durationchange', onDurationChange);
        player.addEventListener('isspectre', onIsSpectre);
        player.addEventListener('loadeddata', onLoadedData);
        player.addEventListener('playing', onPlaying);
        player.addEventListener('error', onError);

        state.addEventListener(EVENT_STATE_UPDATE, onStateUpdate);

        // The player may have already errored because we asynchronously
        // load the DOM, so handle the initial state.
        if (player.error) {
            onError();
        }

        if (player.channel) {
            onLoadedChannel();
        } else if (player.video) {
            onLoadedVideo();
        }
    };

    var onStateUpdate = function() {
        const playerState = state.toJSON();

        // Check online state update
        if (online && playerState.online === false) {
            online = false;
        } else if (!online && playerState.online === true) {
            online = true;
            onOnline();
        }

        // Check for viewercount updates
        if (viewers !== playerState.viewers) {
            viewers = playerState.viewers;
            onViewersChange(viewers);
        }
    };

    var onIsSpectre = function(spectreActive) {
        isSpectre = spectreActive;
        $(root).attr('data-playlist', isSpectre);
    };

    var onLoadedData = function() {
        channelWasOnline = true;
    };

    var onPlaying = function() {
        // TODO: Fix this when UI load ordering is resolved
        // This ensures that the video stays paused when the 'mature' overlay is displayed.
        if ($(root).attr('data-mature') === 'true') {
            player.pause();
            player.setMuted(true);
        }

        hideVideoBackgroundImage();
    };

    // TODO Put this in onLoadedMetadata instead.
    var onOnline = function() {
        var channel = player.getChannel();
        var cacheSpectre = isSpectre;

        channelInfo(channel).then(function(channelData) {
            return offlinePlaylistInfo(channelData._id);
        }).then(function(spectreData) {
            if (channelWasOnline && !cacheSpectre && spectreData.enabled) {
                $(root).attr('data-playlist', 'pending');
            }
        });
    };

    var onError = function() {
        const error = state.i18n.translate(player.error);
        $('.js-player-error', root).text(error);
        $(root).attr('data-error', true);
    };

    var showMatureOverlay = function() {
        $(root).attr('data-mature', true);
    };

    var hideMatureOverlay = function() {
        $(root).attr('data-mature', false);
    };

    var setMature = function() {
        // Save the mature overlay setting for the future.
        mature = true;
        store.set('mature', true);

        hideMatureOverlay();
    };

    var onViewersChange = function(viewers) {
        if (_.isNumber(viewers)) {
            $(root).attr('data-viewers', viewers);

            // Use the localized display for the number of viewers.
            var viewersText = Number(viewers).toLocaleString('en');
            $('.js-meta-viewers', root).text(viewersText);
            $('.js-viewers-label').text(state.i18n.translate(' viewer', {
                count: viewers,
            }));
        } else {
            $(root).removeAttr('data-viewers');
        }
    };

    var onLoadedChannel = function() {
        var channel = player.getChannel();

        $(root).attr('data-channel', channel);

        $(root).removeAttr('data-video');
        $(root).removeAttr('data-game');

        // Calculate the url for the channel
        var url = channelUrl(channel);
        streamUrl.resolve(url);

        // Request the channel information from the API.
        var channelRequest = channelInfo(channel);
        channelRequest.then(onChannelInfo);

        // Request Kraken stream info from the API.
        var streamRequest = streamInfo(channel);
        streamRequest.then(onStreamInfo);
    };

    var onLoadedVideo = function() {
        var video = player.getVideo();
        $(root).attr('data-video', video);

        $(root).removeAttr('data-channel');
        $(root).removeAttr('data-game');

        // Request the video information from the API.
        var videoRequest = videoInfo(video);
        videoRequest.then(function(data) {
            var channel = data.channel.name;

            // Once we get the channel name, we can calculate the url.
            var url = videoUrl(channel, video);
            streamUrl.resolve(url);

            // Video information isn't enough, we also need the channel.
            var channelRequest = channelInfo(channel);
            channelRequest.then(onChannelInfo);

            // Call onVideoInfo after gathering all other data
            onVideoInfo(data);
        });
    };

    var onEnded = function() {
        var channel = player.getChannel();
        if (!channel) {
            // Ignore VODs.
            return;
        }

        channelInfo(channel).then(function(channelData) {
            return offlinePlaylistInfo(channelData._id);
        }).then(function(spectreData) {
            if (channelWasOnline && !isSpectre && spectreData.enabled) {
                $(root).attr('data-playlist', 'pending');
            } else {
                $(root).attr('data-playlist', false);
                showOfflineBanner();
                $(root).attr('data-loading', false);
            }
        });
    };

    var onChannelInfo = function(data) {
        var channel = player.getChannel();

        // Show the mature overlay if the channel is marked.
        if (!mature && data.mature) {
            showMatureOverlay();
        } else {
            hideMatureOverlay();
        }

        // Save the video banner for when we go offline.
        videoBanner = data.video_banner;

        // Set the offline banner if the channel is already over.
        var ended = player.getEnded();
        if (ended) {
            showOfflineBanner();
        }

        // Channel name.
        $('.js-meta-name', root).text(data.display_name);
        // Sets or removes the avatar source.
        $('.js-meta-picture', root).attr('src', data.logo || null);

        // Make the avatar clickable.
        streamUrl.promise.then(function(url) {
            $('.js-meta-url', root).attr('href', url);
        });

        // The remaining data only used for live channels.
        // Otherwise, we would override the video-specific data.
        if (channel) {
            $('.js-meta-title', root).text(data.status);

            // Set and link to the game being played.
            var url = `${Settings.gamePath}/${encodeURIComponent(data.game)}`;
            $('.js-meta-game', root).text(data.game).attr('href', url);

            $(root).attr('data-game', data.game);
        }
    };

    /**
     * Called on stream info allocation from Kraken.
     *
     * @param {Object} data
     */
    var onStreamInfo = function(data) {
        var channel = player.getChannel();

        // The remaining data only used for live channels.
        // Otherwise, we would override the video-specific data.
        if (channel && data.stream) {
            // Live video thumbail image
            videoThumbnail = data.stream.preview.template.replace('{width}x{height}', defaultVideoBackgroundResolution);

            // Show the background image if we haven't tried to load the
            // stream yet.
            if (player.getPaused() && player.getReadyState() === player.HAVE_NOTHING) {
                showVideoBackgroundImage();
            }
        }
    };

    var onVideoInfo = function(data) {
        if (data.preview) {
            // Live video thumbail image
            videoThumbnail = data.preview.replace('320x240', defaultVideoBackgroundResolution);

            // Check autoplay status
            if (!player.getAutoplay()) {
                showVideoBackgroundImage();
            }
        }

        mutedSegments = data.muted_segments;
        updateMutedSegments();

        if (mutedSegments && mutedSegments.length > 0) {
            showAlert(Settings.mutedSegmentsMessage);
        }

        $('.js-meta-title', root).text(data.title);

        // TODO Videos don't have game information.
    };

    var onDurationChange = function() {
        updateMutedSegments();
    };

    var updateMutedSegments = function() {
        var duration = player.getDuration();
        var container = $('.js-seek-slider .js-muted-segments-container', root);

        // always remove all muted segments - when this fn is called, there's no
        // easy way to update existing segments to conform to the updated
        // segments
        container.empty();

        if (!duration || duration === Infinity || !mutedSegments) {
            // duration required to compute the fractional offset, and need
            // mutedSegments to map to DOM elements
            return;
        }

        mutedSegments = combineOverlappingSegments(mutedSegments);

        var baseSegment = $('<span></span>').addClass('player-slider__muted');
        for (var i = 0; i < mutedSegments.length; i++) {
            var segment = baseSegment.clone().css({
                width: `${(mutedSegments[i].duration * 100 / duration)}%`,
                left: `${(mutedSegments[i].offset * 100 / duration)}%`,
            });

            container.append(segment);
        }
    };

    /*
     *  Combines any overlapping segments in an array
     * @param {Object} segments
     */
    var combineOverlappingSegments = function(segments) {
        if (segments.length <= 1) {
            return segments;
        }

        var sortedSegments = segments.slice(0).sort(function(a, b) {
            return a.offset - b.offset;
        });

        return _.reduce(sortedSegments, function(resultArr, currentSegment) {
            var lastSegment = resultArr[resultArr.length - 1];
            var lastSegmentEnd = lastSegment.offset + lastSegment.duration;
            var currentSegmentEnd = currentSegment.offset + currentSegment.duration;

            if (lastSegmentEnd >= currentSegment.offset) {
                if (lastSegmentEnd < currentSegmentEnd) {
                    lastSegment.duration = currentSegmentEnd - lastSegment.offset;
                }
            } else {
                resultArr.push(currentSegment);
            }
            return resultArr;
        }, [sortedSegments[0]]);
    };

    // Display an alert message to the user
    // TODO This should be shared amongst ui modules.
    var showAlert = function(message) {
        var alertElement = $('.js-player-alert', root);

        alertElement.find('.js-player-alert__message').text(message);
        alertElement.attr('data-active', true);
    };

    var showOfflineBanner = function() {
        // Only load the banner for channels.
        var channel = player.getChannel();
        if (videoBanner && channel) {
            $('.js-offline-banner', root).css('background-image', `url('${videoBanner}')`);
        }
    };

    /**
     * Displays the video snapshot background image.
     */
    var showVideoBackgroundImage = function() {
        if (videoThumbnail) {
            $(root).attr('data-background', true);

            $('.js-video-background-banner', root).css('background-image', `url('${videoThumbnail}')`);
        }
    };

    /**
     * Hides the video snapshot background image.
     */
    var hideVideoBackgroundImage = function() {
        $(root).attr('data-background', false);
    };

    /**
     *
     */
    this.destroy = function() {
        $(root).off('click');
    };

    init();
}
