import * as Timestamp from '../util/timestamp';
import { EVENT_STATE_UPDATE } from '../state-tracker';

export function PlayerUISeek(player, root, state) {
    let seeking;
    let channel;

    var init = function() {
        // Uses a ms scale.
        $('.js-seek-slider', root).slider({
            range: 'min',
            value: 0,
            min: 0,
            max: 0, // overwritten by durationchange
            stop(e, pos) {
                var time = pos.value / 1000;
                player.setCurrentTime(time);
                seeking = false;
            },
            slide() {
                seeking = true;
            },
            create() {
                // Don't allow using arrow keys to move the slider handle
                $(this).find('.ui-slider-handle').unbind('keydown');
            },
        });

        $(root).on('mousemove', '.js-seek-slider', onSeekMove);

        player.addEventListener('loadedmetadata', onLoadedMetadata);

        // These events cause currentTime to behave differently.
        player.addEventListener('pause', onTimeChange);
        player.addEventListener('playing', onTimeChange);
        player.addEventListener('waiting', onTimeChange);
        player.addEventListener('timeupdate', onTimeChange);

        player.addEventListener('durationchange', onDurationChange);
        player.addEventListener('bufferedchange', onBufferChange);
        state.addEventListener(EVENT_STATE_UPDATE, onStateUpdate);
    };

    var onStateUpdate = function() {

    };

    var onLoadedMetadata = function() {
        channel = player.getChannel();
    };

    // Set the timestamp for the current mouse position.
    var onSeekMove = function(e) {
        var offset = $(this).offset();
        var relx = e.pageX - offset.left;
        var percent = relx / $(this).width();

        if (percent > 1) {
            percent = 1;
        } else if (percent < 0) {
            percent = 0;
        }

        var formatted = '';
        var duration = player.getDuration();

        // If duration is unknown, we can't show anything.
        if (_.isFinite(duration)) {
            var time = percent * duration;
            formatted = Timestamp.toString(time, false);
        }

        // TODO Don't store the text in a data attribute.
        $('.js-slider-tip-container', root).css({
            left: `${relx}px`,
        });
        $('.js-slider-tip', root).attr('data-tip', formatted);
    };

    // Update the video position every second.
    var updateTimeSecondsTimeout = null;
    var updateTimeSeconds = function() {
        var delay = 0;

        var time = player.getCurrentTime();
        var duration = player.getDuration();
        if (_.isFinite(duration) && time > duration) {
            time = duration;
        }

        if (_.isFinite(time)) {
            var formatted = Timestamp.toString(time, false);
            $('.js-seek-currenttime', root).text(formatted);

            // Delay until the exact millisecond the second changes.
            delay = (Math.ceil(time) - time) * 1000;
            if (delay <= 0) delay = 1;
        }

        // Don't go faster than 60fps (1000 / 60 = 16.66)
        delay = Math.max(delay, 16);

        updateTimeSecondsTimeout = setTimeout(updateTimeSeconds, delay);
    };

    // Update the video slider depending on the duration and width.
    var updateTimeSliderInterval = null;
    var updateTimeSlider = function() {
        var inner = function() {
            if (seeking) {
                // Don't override the user's choice while they drag.
                return;
            }

            var time = player.getCurrentTime();
            if (!_.isFinite(time)) {
                // Not supposed to happen but just to be safe.
                return;
            }

            $('.js-seek-slider', root).slider('value', time * 1000);
        };

        var delay = 0;
        var duration = player.getDuration();

        // If the duration is unknown, use the default delay.
        if (_.isFinite(duration)) {
            // Calculate the amount of time it takes to move one pixel.
            var width = $(root).width();
            delay = duration * 1000 / width;
        }

        // Don't go faster than 60fps (1000 / 60 = 16.66)
        delay = Math.max(delay, 16);

        // Trigger this event to fire at the start of the next second.
        inner();
        updateTimeSliderInterval = setInterval(inner, delay);
    };

    var onTimeChange = function() {
        // Cancel any running timeout loops.
        clearTimeout(updateTimeSecondsTimeout);
        clearInterval(updateTimeSliderInterval);

        // Ignore live streams.
        if (channel) {
            return;
        }

        // Ignore non-seekable streams.
        var duration = player.getDuration();
        if (!_.isFinite(duration)) {
            return;
        }

        // Don't update paused streams for performance reasons.
        if (player.getPaused()) {
            return;
        }

        // These are treated seperately for performance reasons.
        // timeSeconds updates at exact second intervals.
        // timeSlider updates each pixel of the screen.
        updateTimeSeconds();
        updateTimeSlider();
    };

    var onDurationChange = function() {
        var duration = player.getDuration();

        if (_.isFinite(duration)) {
            var formatted = Timestamp.toString(duration, false);

            $('.js-seek-totaltime', root).text(formatted);
            $('.js-seek-slider', root).slider('option', 'max', duration * 1000);
        }

        onTimeChange();
        onBufferChange();
    };

    var onBufferChange = function() {
        // Ignore live stream time updates.
        var duration = player.getDuration();
        var buffered = player.getBuffered();

        if (buffered.length && _.isFinite(duration)) {
            // TODO Support multiple buffer ranges.
            var buffer = buffered.end(0);
            var percent = 100 * buffer / duration;
            if (percent > 100) percent = 100;

            $('.js-seek-buffer', root).css('width', `${percent}%`);
        }
    };

    /**
     *
     */
    this.destroy = function() {
        $(root).off('mousemove');

        // Cancel any running timeout loops.
        clearTimeout(updateTimeSecondsTimeout);
        clearInterval(updateTimeSliderInterval);
    };

    init();
}
