import { EVENT_STATE_UPDATE } from '../state-tracker';
import * as Timestamp from '../util/timestamp';

export class PlayerUISeekBarMarkers {
    constructor(player, root, state) {
        this._player = player;
        this._root = root;
        this._state = state;
        this._dotsContainer = $('.js-marker-dots-container', this._root);
        this._popupContainer = $('.js-popup-container', this._root);
        this._state.addEventListener(EVENT_STATE_UPDATE, this.handleEvent.bind(this));
    }

    handleEvent() {
        if (this._state.getMarkers().length === 0) {
            return;
        }

        if (this._dotsContainer.children().length === 0 && this._state.duration !== null) {
            this._createDots();
            this._createMarkers();
        }

        this._updateDotsPassed();
    }

    _createDots() {
        const baseDot = $('<span></span>').addClass('player-slider__marker-dot js-marker-dot');

        this._dotsContainer.append(this._state.getMarkers().map((marker, index) => {
            const dot = baseDot.clone().css('left', `${(marker.startTime / this._state.duration) * 100}%`);
            dot.attr('data-time', marker.startTime);
            dot.attr('marker-index', index);
            dot.attr('data-passed', false);
            return dot;
        }));

        // On mouse click, marker dots will seek player
        this._dotsContainer.on('click', '.js-marker-dot', event => {
            const markerDotElement = $(event.target);
            this._player.setCurrentTime(parseFloat(markerDotElement.attr('data-time')));
        });

        this._updateDotsPassed();
    }

    _updateDotsPassed() {
        const currentTime = this._state.currentTime;

        $('.js-marker-dot', this._root).each((index, dot) => {
            const dotElement = $(dot);
            const isPassed = Number(dotElement.attr('data-time')) <= currentTime;
            dotElement.attr('data-passed', isPassed);
        });
    }

    _createMarkers() {
        // Preload sprite image into browser cache
        const spriteImage = new Image();
        spriteImage.src = this._state.getMarkers()[0].imageURL;

        // Define hover behavior of marker dots
        this._dotsContainer.on('mouseenter', '.js-marker-dot', event => {
            const currentDot = $(event.target);
            const popupLeftOffset = this._getPopupLeftOffset(currentDot);
            this._populatePopupContainer(currentDot);
            this._popupContainer.css('left', popupLeftOffset);

            // Shows popup, hides mouse timestamp
            $('.js-seek-slider', this._root).attr('show-popup', true);
        });

        this._dotsContainer.on('mouseleave', '.js-marker-dot', () => {
            // Hides popup, shows mouse timestamp
            $('.js-seek-slider', this._root).attr('show-popup', false);
        });
    }

    _populatePopupContainer(currentDot) {
        const markerDot = currentDot;
        const markerIndex = Number(markerDot.attr('marker-index'));
        const marker = this._state.getMarkers()[markerIndex];

        $('.js-popup-title', this._root).text(marker.title);
        $('.js-popup-info', this._root).text(marker.info);
        $('.js-popup-timestamp', this._root).text(Timestamp.toString(marker.startTime, false));
        $('.js-popup-marker-thumbnail', this._root).css({
            height: marker.height,
            width: marker.width,
            'background-image': `url(${marker.imageURL})`,
            'background-position' : `-${marker.x}px -${marker.y}px`,
        });
    }

    _getPopupLeftOffset(dotElement) {
        const dotLeftPos = parseFloat(dotElement.css('left'));
        const dotWidth = parseFloat(dotElement.css('width'));
        const popupWidth = parseFloat(this._popupContainer.css('width'));
        // Assuming dotContainer has same width as player
        const playerWidth = parseFloat(this._dotsContainer.css('width'));
        const offset = dotLeftPos + (0.5 * dotWidth) - (0.5 * popupWidth);

        // Bound left offset between 0 and (playerWidth - popupWidth)
        return Math.max(0, Math.min(offset, playerWidth - popupWidth));
    }

    /**
     *
     */
    destroy() {
        this._dotsContainer.off('mouseenter');
        this._dotsContainer.off('mouseleave');
        this._dotsContainer.off('click');
    }
}
