import { channelInfo, offlinePlaylistInfo, videoInfo } from '../api';
import { EVENT_STATE_UPDATE } from '../state-tracker';

const ANIMATION_DURATION = 6000;
const BLACK_SCREEN_DURATION = 3500;
const FADE_OUT_DURATION = 2000;

export function PlayerUIUpnext(player, root, state) {
    let upNextTimer;
    let noSegmentFoundTimer;
    let backupTimer;
    let resetAnimationTimer;
    let activeVodIndex;
    let nextVodIndex;
    let playlistLength;
    let spectreInfo;
    let remainingTime;
    let nextVideoInfo;
    let transitionSegmentHash = {};
    let isSpectre;
    let channelPromise;
    let activeVodId;
    let nextVodInfoPromise;

    var init = function() {
        player.addEventListener('offline', onOffline);
        player.addEventListener('qualitychange', onQualityChange);
        player.addEventListener('segmentchange', onSegmentChange);
        player.addEventListener('pause', onPause);
        player.addEventListener('isspectre', onSpectreConfirmation);
        player.addEventListener('online', onOnline);
        player.addEventListener('waiting', onWaiting);
        state.addEventListener(EVENT_STATE_UPDATE, onStateUpdate);
    };

    var onStateUpdate = function() {

    };

    var onOnline = function() {
        if (isSpectre) {
            clearAllTimers();
        }

        isSpectre = false;
        player.load();
    };

    var onOffline = function() {
        clearAllTimers();
    };

    var onWaiting = function() {
        clearAllTimers();
    };

    var onSpectreConfirmation = function(spectreConfirmation) {
        isSpectre = spectreConfirmation;
        if (isSpectre) {
            startAnimation();
        }
    };

    var clearAllTimers = function() {
        clearTimeout(noSegmentFoundTimer);
        clearTimeout(upNextTimer);
        clearTimeout(resetAnimationTimer);
        clearTimeout(backupTimer);
    };

    var onPause = function() {
        clearAllTimers();
    };

    var resetAnimation = function() {
        $('.js-transition', root).attr('data-stage', '0');
    };

    var hideAnimationDiv = function() {
        $(root).attr('data-transition', false);
    };

    var resetTransitionHash = function() {
        transitionSegmentHash = {};
        if (player.quality && spectreInfo && spectreInfo.playhead) {
            var vods = spectreInfo.playhead.vods;
            for (var i = 0; i < vods.length; i++) {
                var transitionSegmentInfo = vods[i].transition_segment;
                if (transitionSegmentInfo) {
                    var transitionSegment = transitionSegmentInfo.uris[player.quality];
                    transitionSegmentHash[transitionSegment] = i;
                }
            }
        }
    };

    var onQualityChange = function() {
        resetTransitionHash();
    };

    var onSegmentChange = function(currentSegment) {
        if (!isNaN(transitionSegmentHash[currentSegment])) {
            // Now that we hit a transition segment, we know exactly which vod is playing.
            activeVodIndex = transitionSegmentHash[currentSegment];
            setNextVodIndex();
            resetNextVodInfo();
            var transitionSegmentInfo = spectreInfo.playhead.vods[activeVodIndex].transition_segment;

            remainingTime = transitionSegmentInfo.remaining_seconds * 1000;
            if (remainingTime >= ANIMATION_DURATION) {
                upNextTimer = setTimeout(upnextAnimation, remainingTime - ANIMATION_DURATION);
            }
            $(root).attr('data-transition', true); // This doesn't mean it's actually transitioning yet.
            setTimeout(resetAnimation, remainingTime + BLACK_SCREEN_DURATION);
            setTimeout(hideAnimationDiv, remainingTime + BLACK_SCREEN_DURATION + FADE_OUT_DURATION);

            backupTimer = setTimeout(refreshBackupTimer, remainingTime);

            clearTimeout(noSegmentFoundTimer);
            clearTimeout(resetAnimationTimer);

        }
    };

    var refreshBackupTimer = function() {
        updateVodIndices();
        updatePlayerInfo();
        setNoSegmentFoundTimer();
        resetNextVodInfo();
    };

    var setNoSegmentFoundTimer = function() {
        clearTimeout(noSegmentFoundTimer);
        var estimatedTimeLeft = spectreInfo.playhead.vods[activeVodIndex].duration * 1000;
        noSegmentFoundTimer = setTimeout(refreshBackupTimer, estimatedTimeLeft);

        resetAnimationTimer = setTimeout(resetAnimation, estimatedTimeLeft);
    };

    var startAnimation = function() {
        hideAnimationDiv();
        setupInitialVariables();
    };

    var setupInitialVariables = function() {
        var channel = player.getChannel();
        channelPromise = channelInfo(channel);
        channelPromise.then(function(channelData) {
            if (!channelData && !channelData._id) {
                return;
            }

            offlinePlaylistInfo(channelData._id).then(function(spectreData) {
                spectreInfo = spectreData;

                resetTransitionHash();
                playlistLength = spectreInfo.playhead.vods.length;

                activeVodIndex = spectreInfo.playhead.active_vod_index;
                activeVodId = spectreInfo.playhead.vods[activeVodIndex].id;
                var remainingTimeInActiveVod = spectreInfo.playhead.remaining_seconds_in_active_vod;
                updatePlayerInfo();

                nextVodIndex = activeVodIndex === playlistLength - 1 ? 0 : activeVodIndex + 1;
                var nextVodId = spectreInfo.playhead.vods[nextVodIndex].id;
                nextVodInfoPromise = videoInfo(`v${nextVodId}`);
                nextVodInfoPromise.then(function(vodInfo) {
                    nextVideoInfo = vodInfo;
                });
                backupTimer = setTimeout(refreshBackupTimer, remainingTimeInActiveVod * 1000);
            });
        });
    };

    var updateVodIndices = function() {
        // This gets called when we want to set the current vod index to be the next one, and the next vod index to be the one after that.
        activeVodIndex = activeVodIndex === playlistLength - 1 ? 0 : activeVodIndex + 1;
        setNextVodIndex();
    };

    var setNextVodIndex = function() {
        nextVodIndex = activeVodIndex === playlistLength - 1 ? 0 : activeVodIndex + 1;
    };

    var resetNextVodInfo = function() {
        var nextVodId = spectreInfo.playhead.vods[nextVodIndex].id;
        videoInfo(`v${nextVodId}`).then(function(videoData) {
            if (!videoData) {
                return;
            }
            nextVideoInfo = videoData;
        });
    };

    var updatePlayerInfo = function() {
        activeVodId = spectreInfo.playhead.vods[activeVodIndex].id;
        videoInfo(`v${activeVodId}`).then(function(vodInfo) {
            if (vodInfo) {
                $('.js-currentvod-title', root).text(vodInfo.title);
            }
        });
    };

    var upnextAnimation = function() {
        $('.js-next-video-title', root).text(nextVideoInfo.title);
        $('.js-next-video-thumbnail', root).attr('src', nextVideoInfo.preview);
        $('.js-transition', root).attr('data-stage', '1');
    };

    /**
     *
     */
    this.destroy = function() {
    };

    init();
}
