/**
 * Extrapolating Timer
 * @constructor
 * Initialized with the timestamp at 0s and paused.
 */
export function ExtrapolatingTimer() {
    this._lastKnownTimeStamp = 0; //in ms
    this._paused = true;
    this._lastTimeUpdated = new Date().getTime(); //in ms
}

/**
 * Pause the timer
 */
ExtrapolatingTimer.prototype.pause = function() {
    this._updateLastKnownTimeStamp();
    this._paused = true;
};

/**
 * Resume the timer
 */
ExtrapolatingTimer.prototype.resume = function() {
    this._updateLastKnownTimeStamp();
    this._paused = false;
};

/**
 * Extrapolate the current timestamp based off the last known timestamp
 * @return {Number} extrapolated current time stamp in seconds
 */
ExtrapolatingTimer.prototype.extrapolateTimeStamp = function() {
    if (this._paused) {
        return this._lastKnownTimeStamp / 1000;
    } else {
        var currentTime = new Date().getTime();
        return (this._lastKnownTimeStamp + (currentTime - this._lastTimeUpdated)) / 1000;
    }
};

/**
 * Set the timestamp that the timer will use to extrapolate
 * future timestamps. Makes the time more accurate.
 *
 * @param {Number} currentTimestamp : in seconds
 */
ExtrapolatingTimer.prototype.setCurrentTimeStamp = function(currentTimeStamp) {
    if (!isNaN(currentTimeStamp)) {
        this._lastKnownTimeStamp = currentTimeStamp * 1000;
        this._lastTimeUpdated = new Date().getTime();
    }
};

/**
 * Updates _lastKnownTimeStamp and _lastTimeUpdated.
 * @private
 */
ExtrapolatingTimer.prototype._updateLastKnownTimeStamp = function() {
    var currentTime = new Date().getTime();
    this._lastKnownTimeStamp += (currentTime - this._lastTimeUpdated);
    this._lastTimeUpdated = currentTime;
};
