import BigScreen from 'bigscreen/bigscreen';
import EventEmitter from 'event-emitter';

export const FULLSCREEN_CHANGE = 'fullscreenchange';

/**
 * Fullscreen Utility
 * Surfaces the functionality for full screening the player.
 * @class
 */
export class FullScreen {
    /**
     * @constructor
     */
    constructor(root) {
        this._root = root;
        this._eventEmitter = new EventEmitter();
    }

    /**
     * To add an event listener to listen to any events from FullScreen
     * @param {String} name - name of event to listen to
     * @param {Function} callback - function to call when the event is fired
     */
    addEventListener(name, callback) {
        this._eventEmitter.addListener(name, callback);
    }

    /**
     * To remove an event listener listening to an event from FullScreen
     * @param {String} name - name of event for callback to not listen to
     * @param {Function} callback - function to remove from being callbacked
     */
    removeEventListener(name, callback) {
        this._eventEmitter.removeListener(name, callback);
    }

    /**
     * Checks if player is able to fullscreen.
     * @return {Boolean}
     */
    canFullScreen() {
        return BigScreen.enabled;
    }

    /**
     * Checks if player is fullscreen
     * @return {Boolean}
     */
    isFullScreen() {
        return BigScreen.element === this._root;
    }

    /**
     * Sets player to either full screen or not full screen
     * @param {Boolean} isFullScreen - defines if player should be fullscreen
     */
    setFullScreen(isFullScreen) {
        if (this.isFullScreen() !== isFullScreen) {
            BigScreen.toggle(this._root, this._onFullScreenChange.bind(this), this._onFullScreenChange.bind(this));
        }
    }

    /**
     * Called on any changes to the fullscreen state of the player
     * @private
     */
    _onFullScreenChange() {
        this._eventEmitter.emit(FULLSCREEN_CHANGE);
    }
}
