import * as cookie from 'cookie_js';
import * as Settings from '../settings';

export function patch(store) {
    // Redefine store.get and store.set to preserve values across the
    // http -> https transition
    var isHTTPS     = (location.protocol === 'https:');
    var oldStoreGet = store.get;
    var oldStoreSet = store.set;

    var preserveKey = {
        mature:  'salt_mature',
        muted:   'salt_muted',
        quality: 'salt_quality',
        volume:  'salt_volume',
    };
    // idiotic implementation of store.js uses `store.get` internally to
    // determine membership of the key... use a sentinel value as the default,
    // so the original `get` can also determine membership/existence
    var sentinelValue = {};

    /**
     * Monkey patch over store.js's default `store.get` method, which augments
     * the retrieval method to also consult the cookie (if it exists) to yield
     * the correct value for the key after the https transition.
     *
     * @param {String} key
     * @param {=*} defaultValue
     * @return {*}
     */
    store.get = function(key, defaultValue) {
        var value    = oldStoreGet.call(store, key, sentinelValue);
        var isAbsent = (value === sentinelValue);

        if (!isAbsent) {
            preserveValue(key, value);
        } else if (isHTTPS && preserveKey.hasOwnProperty(key) && cookie.all().hasOwnProperty(preserveKey[key])) {
            // when the store is missing a particular key, and we are on an
            // HTTPS page, and the cookie happens to have the given key, attempt
            // to retrieve the value from the cookie; this is the case where
            // a user's last pageview was in HTTP and their current pageview is
            // HTTPS.
            try {
                value = JSON.parse(cookie.get(preserveKey[key]));
                // if we did this successfully, then we should ensure the value
                // is correctly stored in `localStorage` instead of in the
                // cookie
                store.set(key, value);
            } catch (e) {
                // if this blows up, give back the default value
                value = defaultValue;
            }
        } else {
            value = defaultValue;
        }

        return value;
    };

    /**
     * Monkey patch over store.js's default `store.set` method, which augments
     * the setter to also attempt to preserve the set value across the
     * http -> https transition.
     *
     * @param {String} key
     * @param {*} value
     * @return {*}
     */
    store.set = function(key, value) {
        var retVal = oldStoreSet.call(store, key, value);

        preserveValue(key, value);

        return retVal;
    };

    /**
     * Preserve the value of the key across the http -> https transition
     * boundary, if the key is marked as needed to preserve.
     *
     * @param {String} key
     * @param {*} value
     */
    function preserveValue(key, value) {
        // until this feature is deprecated, continue to set the cookie; at some
        // point, the expiry will be in the past, and the cookie will simply
        // cease to exist.
        if (preserveKey.hasOwnProperty(key)) {
            cookie.set(preserveKey[key], JSON.stringify(value), {
                expires: Settings.httpsQualityCarryover,
                domain:  `.${Settings.domain}`,
                path:    '/',
                secure:  false,
            });
        }
    }
}
