import EventEmitter from 'event-emitter';
import { BackendFlash } from './backend/flash';
import { BackendMse } from './backend/mse';
import { BackendHls } from './backend/hls';
import { BackendChromecast } from './backend/chromecast';
import { BackendMulti } from './backend/multi';
import { BackendIMA } from './backend/ima';
import { BackendBlank } from './backend/blank';
import { videoInfo, channelInfo, offlinePlaylistInfo } from './api';
import store from 'store';
import * as Settings from './settings';
import BigScreen from 'bigscreen/bigscreen';

// TODO VP-365 Remove this monkey patch after the cutoff date
require('./util/https').patch(store);

export function Video(root, analytics, experiments, pubSub, options) {
    var self = this;

    // Names of the available backends, ordered by priority.
    // TODO Add mse as an available backend.
    const BACKENDS = ['flash', 'hls'];

    // Map of backend names to the class reference.
    const BACKEND_MAP = {
        flash: BackendFlash,
        hls: BackendHls,
        mse: BackendMse,
    };

    const events = new EventEmitter();

    // Private variables.
    let backend;
    let chromecast;
    let ads;
    let mainBackend;
    let theatre = false;
    let hasPlayed = false;
    let reloadStreamOnPlay = false;
    let videoURL;

    var init = function() {
        // Create a div for the player so we can start loading ASAP.
        var videoContainer = document.createElement('div');
        videoContainer.className = 'player-video';
        root.appendChild(videoContainer);

        // Use the provided backend, or pick a good default.
        var backendName = options.backend || _.first(self.getSupportedBackends());
        var backendType = BACKEND_MAP[backendName] || BackendBlank;

        // Add in the lastAdDisplay to the options.
        // TODO Refactor this into an advertisements class.
        var backendOpts = _.extend({
            lastAdDisplay: store.get('lastAdDisplay') || 0,
            experiments: experiments,
        }, options);

        // Create the main backend.
        mainBackend = new backendType(analytics, backendOpts);
        mainBackend.attach(videoContainer);

        // Backend used to connect to Chromecast.
        chromecast = BackendChromecast;
        chromecast.init();

        // Use a multi backend to switch between chromecast and main.
        backend = new BackendMulti(chromecast, mainBackend);

        // BackendFlash supports ads itself.
        if (backendType !== BackendFlash && BackendIMA.canPlay() && options.html5Ads) {
            // Only append if the main backend doesn't support ads.
            var adsContainer = document.createElement('div');
            adsContainer.className = 'player-advertisement';
            root.appendChild(adsContainer);

            // Create a backend used to display advertisements.
            ads = new BackendIMA(adsContainer, analytics, backendOpts);

            // Use another multi backend to switch between ads.
            backend = new BackendMulti(ads, backend);

            // Run a preroll only if the channel is online.
            mainBackend.addEventListener('loadedmetadata', function() {
                ads.requestAds('preroll');
            });
        }

        initEvents();
    };

    var initEvents = function() {
        backend.addEventListener('adend', function() {
            store.set('lastAdDisplay', (new Date()).getTime());
        });

        // Monitor if any video has played.
        backend.addEventListener('loadeddata', function() {
            hasPlayed = true;
        });

        // On content stream end, check for Spectre playlists.
        // Attempt to load the offline content pending Spectre is enabled.
        backend.addEventListener('ended', function() {
            channelInfo(backend.getChannel()).then(function(channelData) {
                return offlinePlaylistInfo(channelData._id);
            }).then(function(spectreData) {
                if (spectreData.enabled) {
                    backend.load();
                }
            });
        });

        // Chromecast events that cause its state to change.
        var chromecastEvents = ['emptied', 'suspend', 'loadstart', 'loadedmetadata'];

        _.each(chromecastEvents, function(name) {
            chromecast.addEventListener(name, function() {
                events.emit('castingchange');

                if (options.debug) {
                    console.log('chromecast event: ', name);
                }
            });
        });
    };

    var onTheatreToggle = function() {
        events.emit('theatrechange');
    };

    var onFullscreenStart = function() {
        events.emit('fullscreenchange');
    };

    var onFullscreenEnd = function() {
        events.emit('fullscreenchange');
    };

    self.destroy = function() {
        pubSub.destroy();
        backend.destroy();

        // TODO Clean up any remaining listeners we missed.
        events.removeAllListeners();
    };

    self.addEventListener = function(name, callback) {
        if (!_.contains(Settings.allEvents, name)) {
            console.error('subscribing to unknown event: ', name);
        }

        backend.addEventListener(name, callback);
        events.on(name, callback);
    };

    self.removeEventListener = function(name, callback) {
        backend.removeEventListener(name, callback);
        events.off(name, callback);
    };

    /**
     * Called when a channel online message has been
     * received from the PubSub client.
     */
    self.onChannelOnline = function() {
        // If the player has played content and isn't paused,
        // Attempt to begin playback of the new live stream.
        if (!backend.getPaused() || backend.getPaused() && !hasPlayed || backend.getEnded()) {
            // Player is currently playing content or has not been loaded, reload.
            backend.load();
        } else {
            // Player is currently paused. Mark for stream
            // reload on resume.
            reloadStreamOnPlay = true;
        }
    };

    self.getNetworkProfile = function() {
        return mainBackend.getNetworkProfile();
    };

    // IE8 does not support getters/setters so there are two APIs.
    // Use the function prefixed with get/set for IE8 support, or use
    // standard getters/setters to be compatible with the media tag.

    self.getError = function() {
        return backend.getError();
    };

    self.getSrc = function() {
        // unimplemented
    };

    self.setSrc = function() {
        // unimplemented
    };

    self.getCurrentSrc = function() {
        // unimplemented
    };

    self.NETWORK_EMPTY = 0;
    self.NETWORK_IDLE = 1;
    self.NETWORK_LOADING = 2;
    self.NETWORK_NO_SOURCE = 3;

    self.getNetworkState = function() {
        return backend.getNetworkState();
    };

    self.getPreload = function() {
        return backend.getPreload();
    };

    self.setPreload = function(value) {
        return backend.setPreload(value);
    };

    self.getBuffered = function() {
        return backend.getBuffered();
    };

    self.load = function() {
        backend.load();
    };

    self.HAVE_NOTHING = 0;
    self.HAVE_METADATA = 1;
    self.HAVE_CURRENT_DATA = 2;
    self.HAVE_FUTURE_DATA = 3;
    self.HAVE_ENOUGH_DATA = 4;

    self.getReadyState = function() {
        return backend.getReadyState();
    };

    self.getSeeking = function() {
        return backend.getSeeking();
    };

    self.getCurrentTime = function() {
        return backend.getCurrentTime();
    };

    self.setCurrentTime = function(value) {
        backend.setCurrentTime(value);
    };

    self.getInitialTime = function() {
        return backend.getInitialTime();
    };

    self.getDuration = function() {
        return backend.getDuration();
    };

    self.getStartOffsetTime = function() {
        return backend.getStartOffsetTime();
    };

    self.getPaused = function() {
        return backend.getPaused();
    };

    self.getDefaultPlaybackRate = function() {
        return backend.getDefaultPlaybackRate();
    };

    self.setDefaultPlaybackRate = function(value) {
        backend.setDefaultPlaybackRate(value);
    };

    self.getPlaybackRate = function() {
        return backend.getPlaybackRate();
    };

    self.setPlaybackRate = function(value) {
        backend.setPlaybackRate(value);
    };

    self.getPlayed = function() {
        return backend.getPlayed();
    };

    self.getSeekable = function() {
        return backend.getSeekable();
    };

    self.getEnded = function() {
        return backend.getEnded();
    };

    self.getAutoplay = function() {
        return backend.getAutoplay();
    };

    self.setAutoplay = function(value) {
        backend.setAutoplay(value);
    };

    self.getLoop = function() {
        return backend.getLoop();
    };

    self.setLoop = function(value) {
        backend.setLoop(value);
    };

    self.play = function() {
        if (reloadStreamOnPlay) {
            reloadStreamOnPlay = false;
            backend.load();
        } else {
            backend.play();
        }
    };

    self.pause = function() {
        backend.pause();
    };

    self.getControls = function() {
        // TODO Support disabling controls.
        return true;
    };

    self.setControls = function() {
        // TODO Support disabling controls.
        // unimplemented
    };

    self.getVolume = function() {
        return backend.getVolume();
    };

    self.setVolume = function(value) {
        // Clamp the value between 0 and 1.
        value = Math.max(0, Math.min(1, value));

        // Tell the backend the volume to use.
        backend.setVolume(value);

        // Save the volume in local storage for next load.
        store.set('volume', value);
    };

    self.getMuted = function() {
        return backend.getMuted();
    };

    self.setMuted = function(value) {
        backend.setMuted(value);

        // Save the muted state in local storage for next load.
        store.set('muted', value);
    };

    self.getTheatre = function() {
        return theatre;
    };

    self.setTheatre = function(value) {
        onTheatreToggle();
        theatre = value;
    };

    self.getDefaultMuted = function() {
        return backend.getDefaultMuted();
    };

    self.setDefaultMuted = function(value) {
        backend.setDefaultMuted(value);
    };

    self.getQuality = function() {
        return backend.getQuality();
    };

    self.setQuality = function(value) {
        backend.setQuality(value);

        // Save the quality in local storage for next load.
        store.set('quality', value);
    };

    self.getQualities = function() {
        return backend.getQualities();
    };

    self.getChannel = function() {
        return backend.getChannel();
    };

    self.setChannel = function(value) {
        // Force the channel lowercase; Usher requires it.
        value = value.toLowerCase();

        videoURL = null;
        backend.setChannel(value);
    };

    self.getVideo = function() {
        return backend.getVideo();
    };

    self.setVideo = function(value) {
        // TODO Remove videoURL entirely from this file.
        videoURL = null;
        videoInfo(value).then(function(videoInfo) {
            videoURL = videoInfo.url;
        });

        backend.setVideo(value);
    };

    /**
     * Returns the Video Twitch URL source from the videos API endpoint.
     *
     * @Returns {String} The URL string.
     */
    self.getVideoURL = function() {
        return videoURL;
    };

    // TODO Delete this unused method.
    self.getCurrentSegment = function() {
        return backend.getCurrentSegment();
    };

    self.getStats = function() {
        return backend.getStats();
    };

    self.getStatsEnabled = function() {
        return backend.getStatsEnabled();
    };

    self.setStatsEnabled = function(value) {
        backend.setStatsEnabled(value);
    };

    self.startCast = function() {
        chromecast.load();
    };

    self.stopCast = function() {
        chromecast.stop();
    };

    self.getCasting = function() {
        var readyState = chromecast.getReadyState();
        var networkState = chromecast.getNetworkState();

        if (readyState === chromecast.HAVE_NOTHING) {
            if (networkState === chromecast.NETWORK_EMPTY) {
                return 'unavailable';
            } else if (networkState === chromecast.NETWORK_IDLE) {
                return 'available';
            } else {
                return 'connecting';
            }
        } else {
            return 'connected';
        }
    };

    self.getCastDevice = function() {
        return chromecast.getDevice();
    };

    self.getFullscreen = function() {
        return BigScreen.element === root;
    };

    self.setFullscreen = function(value) {
        // BigScreen only supports toggling.
        var fullscreen = self.getFullscreen();
        if (fullscreen !== value) {
            BigScreen.toggle(root, onFullscreenStart, onFullscreenEnd);
        }
    };

    self.getFullscreenEnabled = function() {
        return BigScreen.enabled;
    };

    self.midroll = function(duration) {
        if (ads) {
            ads.requestAds('midroll', duration);
        }
    };

    self.getVideoInfo = function() {
        return backend.getVideoInfo();
    };

    self.getBackend = function() {
        return backend.getBackend();
    };

    self.setBackend = function(value) {
        // Save the backend in local storage for next load.
        store.set('backend', value);

        // TODO TODO Update the backend inline instead of reloading.
        location.reload();
    };

    // Return an array of supported backends.
    self.getSupportedBackends = function() {
        return _.filter(BACKENDS, function(name) {
            var backendType = BACKEND_MAP[name];
            return backendType.canPlay();
        });
    };

    self.getVersion = function() {
        return mainBackend.getVersion();
    };

    self.isSpectre = function() {
        return backend.isSpectre();
    };

    self.getViewerCount = function () {
        return this._viewers;
    };

    self.onViewersChange = function(data) {
        this._viewers = data.viewers;
        // the viewerCount event is used by the player in the highlighter and the mobile channel page
        events.emit('viewerCount', { count: data.viewers });
        events.emit('viewerschange');
    };

    self.getCaption = function() {
        return backend.getCaption();
    };

    self.getEventEmitter = function() {
        return events;
    };

    // Define the getters/setters to be compatible with the <video> tag.
    // TODO Imagine a world with no IE8...
    var defineProperty = function(name, args) {
        try {
            // enumerable is required for extending/forwarding properties
            args = _.defaults(args, {
                enumerable: true,
            });
            Object.defineProperty(self, name, args);
        } catch (defPropException) {
            // IE8 only, have to use get/set methods.
        }
    };

    defineProperty('error', {
        get: self.getError,
    });

    defineProperty('src', {
        get: self.getSrc,
        set: self.setSrc,
    });

    defineProperty('currentSrc', {
        get: self.getCurrentSrc,
    });

    defineProperty('networkState', {
        get: self.getNetworkState,
    });

    defineProperty('preload', {
        get: self.getPreload,
        set: self.setPreload,
    });

    defineProperty('buffered', {
        get: self.getBuffered,
    });

    defineProperty('readyState', {
        get: self.getReadyState,
    });

    defineProperty('seeking', {
        get: self.getSeeking,
    });

    defineProperty('currentTime', {
        get: self.getCurrentTime,
        set: self.setCurrentTime,
    });

    defineProperty('initialTime', {
        get: self.getInitialTime,
    });

    defineProperty('duration', {
        get: self.getDuration,
    });

    defineProperty('startOffsetTime', {
        get: self.getStartOffsetTime,
    });

    defineProperty('paused', {
        get: self.getPaused,
    });

    defineProperty('defaultPlaybackRate', {
        get: self.getDefaultPlaybackRate,
        set: self.setDefaultPlaybackRate,
    });

    defineProperty('playbackRate', {
        get: self.getPlaybackRate,
        set: self.setPlaybackRate,
    });

    defineProperty('played', {
        get: self.getPlayed,
    });

    defineProperty('seekable', {
        get: self.getSeekable,
    });

    defineProperty('ended', {
        get: self.getEnded,
    });

    defineProperty('autoplay', {
        get: self.getAutoplay,
        set: self.setAutoplay,
    });

    defineProperty('loop', {
        get: self.getLoop,
        set: self.setLoop,
    });

    defineProperty('controls', {
        get: self.getControls,
        set: self.setControls,
    });

    defineProperty('volume', {
        get: self.getVolume,
        set: self.setVolume,
    });

    defineProperty('muted', {
        get: self.getMuted,
        set: self.setMuted,
    });

    defineProperty('defaultMuted', {
        get: self.getDefaultMuted,
        set: self.setDefaultMuted,
    });

    defineProperty('quality', {
        get: self.getQuality,
        set: self.setQuality,
    });

    defineProperty('qualities', {
        get: self.getQualities,
    });

    defineProperty('channel', {
        get: self.getChannel,
        set: self.setChannel,
    });

    defineProperty('video', {
        get: self.getVideo,
        set: self.setVideo,
    });

    defineProperty('stats', {
        get: self.getStats,
    });

    defineProperty('statsEnabled', {
        get: self.getStatsEnabled,
        set: self.setStatsEnabled,
    });

    defineProperty('casting', {
        get: self.getCasting,
    });

    defineProperty('castDevice', {
        get: self.getCastDevice,
    });

    defineProperty('fullscreen', {
        get: self.getFullscreen,
        set: self.setFullscreen,
    });

    defineProperty('fullscreenEnabled', {
        get: self.getFullscreenEnabled,
    });

    // TODO Delete this property.
    defineProperty('currentSegment', {
        get: self.getCurrentSegment,
    });

    defineProperty('theatre', {
        get: self.getTheatre,
        set: self.setTheatre,
    });

    defineProperty('viewers', {
        get: self.getViewerCount,
    });

    init();
}
