"use strict";

$(window).on("beforeunload", function() {
  $(window).scrollTop(0);
});

$(document).ready(function () {
  // Get current reports
  getReports();

  // Report time auto update
  window.timeTimer = setInterval(updateTime, 1000);

});

// relative time from unix timestamp
function relativeDate(from, to) {
  // takes a unix timestamp, spits out relative time

  var then = new Date(from * 1000).getTime(),
  now = to || new Date().getTime();

  var switched = false;
  if(then > now) {
    var new_then = now;
    var new_now = then;
    switched = true;

    then = new_then;
    now = new_now;
  }

  var seconds = Math.floor(((now - then) / 1000) % 60), //every 60 seconds, reset to 0
  minutes = Math.floor(((now - then) / 1000) / 60 % 60), //every 60 minutes, reset to 0
  hours = Math.floor(((now - then) / 1000) / 60 / 60 % 24), //every 24 hours, reset to 0
  days = Math.floor(((now - then) / 1000) / 60 / 60 / 24 % 7), //every 7 days, reset to 0
  weeks = Math.floor(((now - then) / 1000) / 60 / 60 / 24 / 7 % 4), //every 4 weeks, reset to 0
  months = Math.floor(((now - then) / 1000) / 60 / 60 / 24 / 7 / 4 % 12), // every 12 months, reset to 0
  years = Math.floor(((now - then) / 1000) / 60 / 60 / 24 / 7 / 52); // every 52 weeks, reset to 0. More accurate years than using months.

  // string building through ternary abuse so we don't have to use an if latter or another function
  // returns 0m if there is 1 hour because 1h 0m 24s is prefferable to 1h 24s
  return (!!years || false ? years + "yr " : "") +
  (!!months || false ? months + "mo " : "") +
  (!!weeks || false ? weeks + "wk " : "") +
  (!!days || false ? days + "d " : "") +
  (!!hours || false ? hours + "h " : "") +
  (!!hours || false ? (!!minutes || false ? minutes + "m " : "0m ") : (!!minutes || false ? minutes + "m " : "")) +
  seconds + "s" +
  (switched === false ? " ago" : "");
}

// time function
function updateTime() {
  // make a single date object for comparison, get the unix milisecond time from it
  // this is far more performant than making a new date object for every pass
  var now = new Date().getTime();

  // get all the reports, then get their friendly time elements
  $(".panel")
    .find(".time-friendly")
    .each(function(i, e) {
      switch (e.tagName) {
        case "DIV":
          // get the unixtime data attribute from their sibling exact element
          // then get the relative conversion from that and set it as text content
          if(typeof $(e).parent().attr("class") !== "undefined" && ($(e).parent().attr("class") == "list-group-item auto-resolve" || $(e).parent().attr("class") == "list-group-item hold")) {
            if($(e).siblings(".time-exact").data("unixtime") * 1000 < now) {
              // Remove the entry from the auto resolve or hold list (just visibly) as it's run out
              $(e).parent().remove();
              break;
            }
          }
          $(e)
            .text("(" + relativeDate($(e)
              .siblings(".time-exact")
              .data("unixtime"), now) + ")");
          break;
        case "SPAN":
          // don't instantiate new jQuery objects a bunch
          var jQe = $(e);

          // hack(?) to not show garbage on Offline channels
          // sorting an array by default sorts elements in lexical order
          // numbers come first, then Capital Letters in alphabetical order, then lowercase letters
          // so actual durations, then NaN, then Offline
          // ["3h 23m 59s ago", "Offline"]
          // ["NaNs ago", "Offline"]
          //
          // Solution: Lowercase duration result before sorting
          // ["Offline", "nans ago"]
          var selection = [
            relativeDate(
            // get the createdat data attribute from their preview parent element
            // then get the relative conversion from that
            jQe
              .parent() //uptime div
              .parent() //preview info div
              .parent() //stream preview div
              .parent() //entire report panel, yeah this is stupid
              .find(".stream-preview-image")
              .children("img")
              .data("createdat"),
              now
            ).toLowerCase(),
            "Offline"
          ].sort()[0];

          // set the relative time or Offline result as text content
          jQe.text(selection);

            // remove the timer class if it's offline, no sense checking this every second
            if ( selection === "Offline" ) { jQe.removeClass("time-friendly") }
          break;
      }
    })
}

function getReports() {
  // Update the last get reports time
  last_get_reports_time = current_unixtime();

  // Make the request to the "api"
  $.ajax({
    url: "/reports.json" + show_all_reports_param(),
    method: "GET",
    dataType: "json",
    cache: false,
    timeout: 10000, // sets timeout to 10 seconds
    success: function(response_body, textStatus, jqXHR) {
      // console.log(response_body);
      // Put the response into the local "cache"
      reports = response_body;

      // Check if another admin joined and we redistributed reports or if the site changed (after a push or something)
      check_report_report_version_difference(jqXHR.getResponseHeader("X-Report-Version"));

      // Handle aka show auto resolves
      handle_autoresolves(response_body.auto_resolves);

      // Handle aka show holds
      handle_holds(response_body.holds);

      // Handle aka show active admins
      handle_active_admins(response_body.active_admins);

      // Update overall unresolved reports count
      update_overall_unresolved_report_count(response_body._overall_total);

      // Remove the loading info
      if($("#report_loading_info").html().length > 0) {
        clear_waiting_for_reports_text();

        // There are currently no reports on the page so handle the array directly
        handle_reports();

        if(mode == "normal") {
          // Update the page title with the current report count
          update_report_number_in_title($(".report").length);
        }
      }

      // setTimeout to load the new reports in x seconds again
      if(mode == "fade") {
        setTimeout(function () { getReports(); }, 10000);

        if($(".report").length <= 0) {
          set_waiting_for_reports_text();
        }

        // Update the page title with the current report count
        update_report_number_in_title($(".report").length);
      } else if (mode == "normal" && get_report_number_from_title() <= 0) {
        set_waiting_for_reports_text();
        setTimeout(function () { getReports(); }, 10000);
      }
    },
    error: function(jqXHR, textStatus, errorThrown) {
      var err = jqXHR.status + ", " + textStatus + ", " + errorThrown;
      console.log("Reports request Failed: " + err);
      // Just try again
      setTimeout(function () { getReports(); }, 10000);
    }
  });
}

function remove_old_entries_from_handled_reports() {
  Object.keys(handled_reports).forEach(function (key) {
    if(handled_reports[key] < (last_get_reports_time - 10)) {
      delete handled_reports[key];
    }
  });
}

function remove_old_entries_from_handled_emails() {
  Object.keys(handled_emails).forEach(function (key) {
    if(handled_emails[key] < (last_get_reports_time - 30)) {
      delete handled_emails[key];
    }
  });
}

function handle_autoresolves(autoresolve_array) {
  // THis should never happen so better don't do anything
  if(autoresolve_array === undefined) {
    return;
  }

  if(autoresolve_array.length > 0) {
    // Remove autoresolves from the page which don't exist in the autoresolve_array
    $(".auto-resolve")
      .each(function (i, e) {
        var autoresolve_id = e.id.split("_")[1];
        if(in_autoresolve_array(autoresolve_id, autoresolve_array) === false) {
          $('#' + e.id).remove();
          console.log("Removing autoresolve: " + autoresolve_id);
        }
     });

    // Add autoresolves to the page
    for(var i = 0; i < autoresolve_array.length; i++) {
      // Check if it already exists
      if($("#autoresolve_" + autoresolve_array[i]["id"]).length === 0) {
        $("#auto_resolves").append(generate_autoresolve_list_HTML(autoresolve_array[i]));
        console.log("Added autoresolve: " + autoresolve_array[i]["id"]);
      }
    }
  } else {
    // Clear all currently displaying on the page
    $("#auto_resolves").empty();
  }
}

function handle_holds(hold_array) {
  // This should never happen so better don't do anything
  if(hold_array === undefined) {
    return;
  }

  if(hold_array.length > 0) {
    // Remove holds from the page which don't exist in the hold_array
    $(".hold")
      .each(function (i, e) {
        var hold_id = e.id.split("_")[1];
        if(in_hold_array(hold_id, hold_array) === false) {
          $('#' + e.id).remove();
          console.log("Removing hold: " + hold_id);
        }
     });

    // Add holds to the page
    for(var i = 0; i < hold_array.length; i++) {
      // Check if it already exists
      if($("#hold_" + hold_array[i]["id"]).length === 0) {
        $("#holds").append(generate_hold_list_HTML(hold_array[i]));
        console.log("Added hold: " + hold_array[i]["id"]);
      }
    }
  } else {
    // Clear all currently displaying on the page
    $("#holds").empty();
  }
}

function handle_active_admins(active_admins_array) {
  // This should never happen so better don't do anything
  if(active_admins_array === undefined) {
    return;
  }

  if(active_admins_array.length > 0) {
    // Remove active admins from the page which don't exist in the active_admins_array
    $(".active-admin")
      .each(function (i, e) {
        var active_admin_id = e.id.split("_")[2];
        if(in_active_admins_array(active_admin_id, active_admins_array) === false) {
          $('#' + e.id).remove();
          console.log("Removing active admin: " + active_admin_id);
        }
     });

    // Add active admins to the page
    for(var i = 0; i < active_admins_array.length; i++) {
      // Check if it already exists
      if($("#active_admin_" + active_admins_array[i]["id"]).length === 0) {
        $("#active_admins").append(generate_active_admin_list_HTML(active_admins_array[i]));
        console.log("Added active admin: " + active_admins_array[i]["id"]);
      }
    }
  } else {
    // Clear all currently displaying on the page
    $("#active_admins").empty();
  }

  // Check if the current user is within the active list, else show the join button
  if($(".active-admin[data-admin-id='" + current_user_id + "']").length === 0) {
    $("#active_admins_join").html('<button class="btn btn-success btn-xs" onclick="join_active_admins();">Join as active admin</button>');
  } else {
    $("#active_admins_join").empty();
  }
}

function generate_active_admin_list_HTML(active_admin) {
  var html = '<li class="list-group-item active-admin" id="active_admin_' + active_admin['id'] + '" data-admin-id="' + active_admin['admin_id'] + '" data-admin-weight="' + active_admin['admin_weight'] + '">';
  html += active_admin['admin_username'] + ' (';
  html += '<a href="javascript:show_active_admin_weight_change_bootbox(' + active_admin['id'] + ', ' + active_admin['admin_weight'] + ')">'
  html += active_admin['admin_weight'];
  html += '</a>';
  html += ')';
  // Remove button
  html += '<button class="btn btn-default btn-sm pull-right" onclick="disable_active_admin(' + active_admin['id'] + ')"><span class="glyphicon glyphicon-remove"></span></button>';
  // ---
  html += '</li>';
  return html;
}

function generate_hold_list_HTML(hold) {
  var html = '<li class="list-group-item hold" id="hold_' + hold['id'] + '" data-report-id="' + hold['report_id'] + '">';
  if(current_user_id != hold['created_by']) {
    html += '<abbr title="Created by ' + hold['created_by_username'] + '">*</abbr> ';
  }
  switch(hold["content"]) {
    case "user_report":
      html += '<abbr title="User report">[U]</abbr> ';
      break;
    case "whisper_report":
      html += '<abbr title="Whisper report">[W]</abbr> ';
      break;
    case "channel_feed_post_report":
      html += '<abbr title="Channel feed post report">[FP]</abbr> ';
      break;
    case "channel_feed_comment_report":
      html += '<abbr title="Channel feed comment report">[FC]</abbr> ';
      break;
    case "clip_report":
      html += '<abbr title="Clip report">[C]</abbr> ';
      break;
    case "vod_report":
      html += '<abbr title="VOD report">[V]</abbr> ';
      break;
    case "vod_comment_report":
      html += '<abbr title="VOD comment report">[VC]</abbr> ';
      break;
  }
  html += '<a href="/reports/' + hold['report_id'] + '" target="_blank">' + hold['target_user_username'] + '</a>';
  html += '<div class="time-exact" data-unixtime="' + Math.floor(Date.parse(hold['hold_until']) / 1000) + '" style="display:inline;"></div> <div class="time-friendly" style="display:inline;">(' + relativeDate(Math.floor(Date.parse(hold['hold_until']) / 1000)) + ')</div>';
  // Remove button
  html += '<button class="btn btn-default btn-sm pull-right" onclick="disable_hold(' + hold['id'] + ')"><span class="glyphicon glyphicon-remove"></span></button>';
  // ---
  html += '</li>';
  return html;
}

function generate_autoresolve_list_HTML(autoresolve) {
  var html = '<li class="list-group-item auto-resolve" id="autoresolve_' + autoresolve['id'] + '">';
  if(current_user_id != autoresolve['created_by']) {
    html += '<abbr title="Added by ' + autoresolve['created_by_username'] + '">*</abbr> ';
  }
  switch(autoresolve["content"]) {
    case "user_report":
      html += '<abbr title="User report">[U]</abbr> ';
      break;
    case "whisper_report":
      html += '<abbr title="Whisper report">[W]</abbr> ';
      break;
    case "channel_feed_post_report":
      html += '<abbr title="Channel feed post report">[FP]</abbr> ';
      break;
    case "channel_feed_comment_report":
      html += '<abbr title="Channel feed comment report">[FC]</abbr> ';
      break;
    case "clip_report":
      html += '<abbr title="Clip report">[C]</abbr> ';
      break;
    case "vod_report":
      html += '<abbr title="VOD report">[V]</abbr> ';
      break;
    case "vod_comment_report":
      html += '<abbr title="VOD comment report">[VC]</abbr> ';
      break;
  }

  if(autoresolve["target_user_username"] !== null && autoresolve["from_user_username"] !== null) {
    var is_not_placeholder_target = '';
    if(autoresolve["target_user_id"] < 0) {
      is_not_placeholder_target = ' IS NOT';
    }
    var is_not_placeholder_from = '';
    if(autoresolve["from_user_id"] < 0) {
      is_not_placeholder_from = ' IS NOT';
    }
    html += '<abbr title="Target">T</abbr>' + is_not_placeholder_target + ": " + link_to_twitch_user(autoresolve["target_user_username"]) + ', <abbr title="Reporter">R</abbr>' + is_not_placeholder_from + ": " + link_to_twitch_user(autoresolve["from_user_username"]);
  } else if(autoresolve["target_user_username"] !== null && autoresolve["from_user_username"] === null) {
    var is_not_placeholder = '';
    if(autoresolve["target_user_id"] < 0) {
      is_not_placeholder = ' IS NOT';
    }
    html += '<abbr title="Target">T</abbr>' + is_not_placeholder + ": " + link_to_twitch_user(autoresolve["target_user_username"]);
  } else if(autoresolve["target_user_username"] === null && autoresolve["from_user_username"] !== null) {
    var is_not_placeholder = '';
    if(autoresolve["from_user_id"] < 0) {
      is_not_placeholder = ' IS NOT';
    }
    html += '<abbr title="Reporter">R</abbr>' + is_not_placeholder + ": " + link_to_twitch_user(autoresolve["from_user_username"]);
  }

  html += '<div class="time-exact" data-unixtime="' + Math.floor(Date.parse(autoresolve['resolve_until']) / 1000) + '" style="display:inline;"></div> <div class="time-friendly" style="display:inline;">(' + relativeDate(Math.floor(Date.parse(autoresolve['resolve_until']) / 1000)) + ')</div>';
  // Remove button
  html += '<button class="btn btn-default btn-sm pull-right" onclick="disable_autoresolve(' + autoresolve['id'] + ')"><span class="glyphicon glyphicon-remove"></span></button>';
  // ---
  html += '</li>';
  return html;
}

function link_to_twitch_user(username) {
  return '<a href="' + twitch_base_url + username + '" target="_blank" rel="noreferrer">' + username + '</a>';
}

function handle_reports() {
  // Handle emails first
  handle_emails();

  // Get the local reports
  var report_array = reports["reports"];
  var start_count = $(".report").length;
  var added_count = 0;


  // Can't handle reports if there are none in the local var
  if(report_array === undefined) {
    return;
  }

  // Remove reports from the page which don't exist in the report_array
  $(".report")
    .each(function (i, e) {
      var report_id = e.id.split("_")[1];
      if(in_report_array(report_id, report_array) === false) {
        $('#' + e.id).remove();
        $('#' + e.id + '_hr').remove();
        console.log("Removing: " + e.id);
      }
   });

  // Handled reports cache control
  remove_old_entries_from_handled_reports();

  // Add reports to the page
  for(var i = 0; i < report_array.length; i++) {
    // Check if it already exists or has already been handled or is on the hold list
    if(handled_reports[report_array[i]["id"]] === undefined && report_id_is_on_page(report_array[i]["id"]) === false && $('.hold[data-report-id="' + report_array[i]["id"] + '"]').length === 0) {
      // Check if we have a report against this user on our page already
      if($('.report[data-target-id="' + report_array[i]["target_user_id"] + '"]').filter('[data-content="' + report_array[i]["content"] + '"]').length > 0) {
        // First try to match by target and content
        $('.report[data-target-id="' + report_array[i]["target_user_id"] + '"]')
          .filter('[data-content="' + report_array[i]["content"] + '"]')
          .last()
            .next('hr')
              .after(generate_report_HTML(report_array[i]));
        console.log("Added by target and content match: " + report_array[i]["id"]);
        added_count = added_count + 1;
      } else if ($('.report[data-target-id="' + report_array[i]["target_user_id"] + '"]').length > 0) {
        // Match by target
        $('.report[data-target-id="' + report_array[i]["target_user_id"] + '"]')
          .last()
            .next('hr')
              .after(generate_report_HTML(report_array[i]));
        console.log("Added by target match: " + report_array[i]["id"]);
        added_count = added_count + 1;
      } else {
        // Just add it at the end
        $("#reports_area").append(generate_report_HTML(report_array[i]));
        console.log("Added to end: " + report_array[i]["id"]);
        added_count = added_count + 1;
      }
    }
  }

  if(added_count > 0) {
    if(start_count == 0) {
      play_sound_notification();
    }
    // Remove a possible waiting for new reports text
    clear_waiting_for_reports_text();

    // Make sure the descriptions aren't too big
    check_all_reports_description_height();

    // Request stream preview images
    update_all_stream_preview_images();

    // Load suspension history for all reports
    load_all_suspension_histories();

    // Load channel content auto previews
    load_all_channel_content_auto_previews();
  }

  // Clear the reports object
  reports = {};
}

function handle_emails() {
  // Get the local emails
  var email_array = reports["emails"];
  var added_count = 0;


  // Can't handle emails if there are none in the local var
  if(email_array === undefined) {
    return;
  }

  // Remove emails from the page which don't exist in the email_array
  $(".email")
    .each(function (i, e) {
      var email_id = e.id.split("_")[1];
      if(in_email_array(email_id, email_array) === false) {
        $('#' + e.id).remove();
        $('#' + e.id + '_hr').remove();
        console.log("Removing: " + e.id);
      }
   });

  // Handled emails cache control
  remove_old_entries_from_handled_emails();

  // Add emails to the page
  for(var i = 0; i < email_array.length; i++) {
    // Check if it already exists or has already been handled
    if(handled_emails[email_array[i]["id"]] === undefined && email_id_is_on_page(email_array[i]["id"]) === false) {
      // Just add it at the end
      $("#notifications").after(generate_email_HTML(email_array[i]));
      console.log("Added to top: " + email_array[i]["id"]);
      added_count = added_count + 1;
    }
  }

  if(added_count > 0) {
    play_sound_notification();
  }
}

function in_report_array(id, report_array) {
  if(report_array.length == 0) {
    return false;
  }

  for(var i = 0; i < report_array.length; i++) {
    if(report_array[i]["id"] == id) {
      return true;
    }
  }
  return false;
}

function in_email_array(id, email_array) {
  if(email_array.length == 0) {
    return false;
  }

  for(var i = 0; i < email_array.length; i++) {
    if(email_array[i]["id"] == id) {
      return true;
    }
  }
  return false;
}

function in_autoresolve_array(id, autoresolve_array) {
  if(autoresolve_array.length == 0) {
    return false;
  }

  for(var i = 0; i < autoresolve_array.length; i++) {
    if(autoresolve_array[i]["id"] == id) {
      return true;
    }
  }
  return false;
}

function in_hold_array(id, hold_array) {
  if(hold_array.length == 0) {
    return false;
  }

  for(var i = 0; i < hold_array.length; i++) {
    if(hold_array[i]["id"] == id) {
      return true;
    }
  }
  return false;
}

function in_active_admins_array(id, active_admins_array) {
  if(active_admins_array.length == 0) {
    return false;
  }

  for(var i = 0; i < active_admins_array.length; i++) {
    if(active_admins_array[i]["id"] == id) {
      return true;
    }
  }
  return false;
}

function report_id_is_on_page(id) {
  if($("#report_" + id).length > 0) {
    return true;
  } else {
    return false;
  }
}

function email_id_is_on_page(id) {
  if($("#email_" + id).length > 0) {
    return true;
  } else {
    return false;
  }
}

function generate_report_HTML(report) {
  var html = "";
  // Report header --------------------------
  html += '<div class="panel panel-default report" id="report_' + report["id"] + '"' +
            ' data-report-id="' + report['id'] + '"' +
            ' data-target-name="' + report['target_user_username'] + '"' +
            ' data-target-id="' + report['target_user_id'] + '"' +
            ' data-target-type="' + report['target_user_type'] + '"' +
            ' data-target-partner="' + boolean_to_string(report['target_user_partner']) + '"' +
            ' data-from-name="' + report['from_user_username'] + '"' +
            ' data-from-id="' + report['from_user_id'] + '"' +
            ' data-from-type="' + report['from_user_type'] + '"' +
            ' data-from-partner="' + boolean_to_string(report['from_user_partner']) + '"' +
            ' data-content="' + report['content'] + '"' +
            ' data-content-id="' + escapeHtml(report['content_id']) + '"' +
            ' data-reason="' + report['reason'] + '"' +
            ' data-created-at="' + (Math.floor(Date.parse(report['created_at']) / 1000)) + '"' +
          '>';
  // "Subject" line --------------------------
  html += '<div class="panel-heading">';
  // Display if it's specifically assigned to this user
  if(report["status"] == "open" && report["assigned_to"] == current_user_id) {
    html += '<span class="label label-info" style="margin-right: 5px;">Assigned to you</span> ';
  }
  // Channel status (TOS'd, deleted ...)
  html += '<span id="channel_status_' + report["id"] + '"></span>';
  // Content type
  switch(report["content"]) {
    case "user_report":
      html += "User report";
      break;
    case "whisper_report":
      html += "Whisper report";
      break;
    case "channel_feed_post_report":
      html += "Channel feed post report";
      break;
    case "channel_feed_comment_report":
      html += "Channel feed comment report";
      break;
    case "clip_report":
      html += "Clip report";
      break;
    case "vod_report":
      html += "VOD report";
      break;
    case "vod_comment_report":
      html += "VOD comment report";
      break;
    default:
      html += "Report";
  }
  html += " concerning "
  switch(report["target_user_type"]) {
    case "staff":
      html += '<span class="btn btn-xs btn-danger">Staff</span> ';
      break;
    case "admin":
      html += '<span class="btn btn-xs btn-warning">Admin</span> ';
      break;
    case "global_mod":
      html += '<span class="btn btn-xs btn-success">Global Mod</span> ';
      break;
    default:
  }
  if(report["target_user_partner"] == true) {
    html += "[P] ";
  }
  html += '<b><a href="' + twitch_base_url + report["target_user_username"] + '?bttvAnonChat=true&amp;tmi_host=false" target="_blank" rel="noreferrer">' + report["target_user_username"] + '</a></b>';
  html += ' <a href="https://translate.google.com/#auto/en/' + report["target_user_username"] + '" target="_blank" rel="noreferrer" onclick="return translate_popup(this.href, ' + "'" + 'target_user_username_' + report["id"] + "'" + ');"><img src="'+ translate_image_url + '" title="Translate" alt=""></a>';
  // Display name
  if(report["target_user_display_name"] !== null) {
    html += ' (' + htmlEncode(report["target_user_display_name"].trim()) + ')';
    html += ' <a href="https://translate.google.com/#auto/en/' + encodeURI(report["target_user_display_name"].trim()) + '" target="_blank" rel="noreferrer" onclick="return translate_popup(this.href, ' + "'" + 'target_user_display_name_' + report["id"] + "'" + ');"><img src="' + translate_image_url + '" title="Translate" alt=""></a>';
  }
  html += ' | ';
  html += '<div class="btn-group">';
  // Auto resolve ---
  html += generate_autoresolve_HTML('target', report);
  // ---
  // Hold ------
  html += generate_hold_HTML(report);
  // ---
  html += '</div>';
  // --- end of btn-group
  // Ban exception ---
  if(typeof report["ban_exception"] !== "undefined" && report["ban_exception"].length > 0) {
    html += ' | <b>DO NOT BAN!</b> for "' + htmlEncode(report["ban_exception"]) + '"';
  }
  // ---
  html += '</div>';
  // Subject end ------------------------------
  // Body start -------------------------------
  html += '<div class="panel-body" style="word-wrap: normal;">';
  // Quick links start ------------------------
  html += '<div class="btn-group">';
  html += '<a class="btn btn-default btn-sm btn-link" href="/chatlogs/?table=' + chatlog_date(report['created_at']) + '&amp;username=' + report["target_user_username"] + '" target="_blank" rel="noreferrer">Chatlogs</a>';
  html += '<a class="btn btn-default btn-sm btn-link" href="https://player.twitch.tv/?channel=' + report["target_user_username"] + '" target="_blank" rel="noreferrer">Popout</a>';
  if(report["content"] == 'clip_report') {
    html += '<a class="btn btn-default btn-sm btn-link" href="javascript:show_clip_preview(\'' + report["id"] + "', '" + escapeHtml(report["content_id"]) + '\');">Clip Preview</a>';
  } else if(report["content"] == 'vod_report') {
    html += '<a class="btn btn-default btn-sm btn-link" href="javascript:show_vod_preview(\'' + report["id"] + "', '" + escapeHtml(report["content_id"]) + "', '" + escapeHtml(report["extra1"]) + '\');">VOD Preview</a>';
  } else {
    html += '<a class="btn btn-default btn-sm btn-link" href="javascript:create_stream_preview(\'' + report["id"] + "', '" + report["target_user_username"] + '\');">Stream Preview</a>';
  }
  html += '<a class="btn btn-default btn-sm btn-link" href="https://translate.google.com/#auto/en/' + generate_translation_text(report["content"], report["description"]) + '" target="_blank" rel="noreferrer" onclick="return translate_popup(this.href, ' + report["id"] + ');">Translate</a>';
  html += '<a class="btn btn-default btn-sm btn-link" href="' + twitch_base_url + report["target_user_username"] + '/manager/past_broadcasts" target="_blank" rel="noreferrer">VOD&#039;s</a>';
  html += '<a class="btn btn-default btn-sm btn-link" href="/reports/' + report["id"] + '" target="_blank" rel="noreferrer">Report Link</a>';
  // Channel content preview buttons
  html += '<div class="btn-group">';
  html += '<button type="button" data-toggle="dropdown" class="btn btn-default btn-sm dropdown-toggle">Channel content <span class="caret"></span></button>';
  html += '<ul class="dropdown-menu bullet pull-right">';
  html += '<li><a class="btn btn-default btn-sm" href="javascript:create_profile_image_preview(' + report["id"] + ', ' + "'" + report["target_user_username"] + "'" + ');">Profile Image</a></li>';
  html += '<li><a class="btn btn-default btn-sm" href="javascript:create_profile_banner_preview(' + report["id"] + ', ' + "'" + report["target_user_username"] + "'" + ');">Profile Banner</a></li>';
  html += '<li><a class="btn btn-default btn-sm" href="javascript:create_offline_image_preview(' + report["id"] + ', ' + "'" + report["target_user_username"] + "'" + ');">Offline Image</a></li>';
  html += '<li><a class="btn btn-default btn-sm" href="javascript:create_profile_image_preview(' + report["id"] + ', ' + "'" + report["target_user_username"] + "'" + ');create_profile_banner_preview(' + report["id"] + ', ' + "'" + report["target_user_username"] + "'" + ');create_offline_image_preview(' + report["id"] + ', ' + "'" + report["target_user_username"] + "'" + ');">Show all</a></li>';
  html += '<li class="divider"></li>';
  html += '<li><a class="btn btn-default btn-sm" href="javascript:create_user_bio_preview(' + report["id"] + ', ' + "'" + report["target_user_username"] + "'" + ');">User bio</a></li>';
  html += '</ul>';
  html += '</div>';
  // ---
  html += '<a class="btn btn-default btn-sm btn-link" href="/suspensions/search?suspensions[target_user_id]=' + report["target_user_id"] + '" target="_blank" rel="noreferrer">TOS History</a>';
  // Whisper report history
  if(report["content"] == 'whisper_report' || report["content"] == 'user_report') {
    html += '<a class="btn btn-default btn-sm btn-link" href="/reports/search?reports[content]=whisper_report&amp;reports[target_user_id]=' + report["target_user_id"] + '" target="_blank" rel="noreferrer">Whisper report history</a>';
  }
  html += '<a class="btn btn-default btn-sm btn-link" href="/alt_lookup?username=' + report["target_user_username"] + '" target="_blank" rel="noreferrer">Alt lookup</a>';
  html += '<a class="btn btn-default btn-sm btn-link" href="/files/new?channel=' + report["target_user_username"] + '" target="_blank" rel="noreferrer">Add Screenshot</a>';
  if(report["target_user_partner"] === false) {
    html += '<a class="btn btn-default btn-link btn-sm" href="' + twitch_base_url + report["target_user_username"] + '/report_form?tos_ban=true&amp;' + generate_report_link_content(report) + '" target="_blank" rel="noreferrer">Strike Link</a>';
  } else if (report["target_user_partner"] === true) {
    html += '<a class="btn btn-default btn-link btn-sm" href="/reports/partnerconduct_report?' + generate_report_link_content(report) + '" target="_blank" rel="noreferrer">Partnerconduct Link</a>';
  }
  html += '</div><hr>';
  // Quick links end --------------------------
  // Divs for expendable content
  html += '<div class="target_user_suspension_history" id="report_history_' + report["id"] + '"></div>';
  html += '<div class="stream-preview-image" id="stream_preview_image_' + report["id"] + '" data-channel="' + report["target_user_username"] + '"></div>';
  html += '<div id="profile_image_preview_' + report["id"] + '"></div>';
  html += '<div id="profile_banner_preview_' + report["id"] + '"></div>';
  html += '<div id="offline_image_preview_' + report["id"] + '"></div>';
  html += '<div id="user_bio_preview_' + report["id"] + '"></div>';
  html += '<div id="clip_preview_' + report["id"] + '"></div>';
  html += '<div id="vod_preview_' + report["id"] + '"></div>';
  html += '<div id="stream_preview_' + report["id"] + '"></div>';
  // Report body ------------------------------
  html += '<div class="well">';
  switch(report["content"]) {
    case "user_report":
      if(report["extra1"] !== null) {
        html += '<b>VOD:</b> <span id="vod_link_' + report["id"] + '"><a target="_blank" rel="noreferrer" href="' + twitch_base_url + encodeURI(report["target_user_username"]) + '/v/' + encodeURI(report["extra1"]) + '">' + htmlEncode(twitch_base_url + report["target_user_username"] + '/v/' + report["extra1"]) + '</a></span><br>';
      }
      html += '<b>Clips:</b> <span id="clip_link_' + report["id"] + '"><a class="btn btn-xs btn-default btn-link" href="javascript:show_clips_timestamp_history(\'' + report["target_user_username"] + "', '" + Math.floor(Date.parse(report['created_at']) / 1000) + '\');">Check Clips around the report time</a></span><br>';
      html += '<b>Reason:</b> <span class="reason-' + report["reason"] + '">' + report["reason"] + '</span><br>';
      html += '<b>Description:</b><br>';
      html += '<div id="report_description_' + report["id"] + '" class="report-description" style="word-wrap: break-word;">';
      html += htmlEncode(report['description']).replace(/\r\n/g, "\n").replace(/\r/g, "\n").replace(/\n/g, "<br>");
      html += '</div>';
      break;
    case "whisper_report":
      html += '<b>Reports against in the past 24h:</b> ' + report["target_user_received_whisper_reports"] + '<br>';
      html += '<b>Report ID:</b> ' + report["id"] + '<br>';
      html += '<b>Reason:</b> <span class="reason-' + report["reason"] + '">' + report["reason"] + '</span><br>';
      html += '<b>Recent Messages:</b><br>';
      html += '<div id="report_description_' + report["id"] + '" class="report-description" style="word-wrap: break-word;">';
      html += parse_whisper_description(report["description"], report["target_user_username"]);
      html += '</div>';
      break;
    case "channel_feed_post_report":
      html += '<b>Report ID:</b> ' + report["id"] + '<br>';
      html += '<b>Content ID:</b> <a href="' + twitch_base_url + report["target_user_username"] + '/p/' + encodeURI(report["content_id"]) + '" target="_blank" rel="noreferrer">' + htmlEncode(report["content_id"]) + '</a><br>';
      html += '<b>Reason:</b> <span class="reason-' + report["reason"] + '">' + report["reason"] + '</span><br>';
      html += '<b>Feed post content:</b><br>';
      html += '<div id="report_description_' + report["id"] + '" class="report-description" style="word-wrap: break-word;">';
      html += htmlEncode(report['description']).replace(/\r\n/g, "\n").replace(/\r/g, "\n").replace(/\n/g, "<br>");
      html += '</div>';
      break;
    case "channel_feed_comment_report":
      html += '<b>Report ID:</b> ' + report["id"] + '<br>';
      html += '<b>Content ID:</b> ' + htmlEncode(report["content_id"]) + '<br>';
      html += '<b>Link to post:</b> <span id="channel_feed_comment_link_' + report["id"] + '"><a class="btn btn-xs btn-default btn-link" href="javascript:get_channel_feed_comment_url(\'' + report["id"] + "', '" + escapeHtml(report["extra1"]) + "', '" + escapeHtml(report["content_id"]) + '\');">Get channel feed post URL</a></span><br>';
      html += '<b>Reason:</b> <span class="reason-' + report["reason"] + '">' + report["reason"] + '</span><br>';
      html += '<b>Feed comment content:</b><br>';
      html += '<div id="report_description_' + report["id"] + '" class="report-description" style="word-wrap: break-word;">';
      html += htmlEncode(report['description']).replace(/\r\n/g, "\n").replace(/\r/g, "\n").replace(/\n/g, "<br>");
      html += '</div>';
      break;
    case "clip_report":
      html += '<b>Target clip:</b> <a href="https://clips.twitch.tv/' + encodeURI(report["content_id"]) + '" target="_blank" rel="noreferrer">https://clips.twitch.tv/' + htmlEncode(report["content_id"]) + '</a> <span id="clip_status_' + report["id"] + '"><a class="violating_content_preview_link" href="javascript:check_clip_status(' + report["id"] + ',' + "'" + escapeHtml(report["content_id"]) + "'" + ');"></a></span><br>';
      html += '<b>Clips:</b> <span id="clip_link_' + report["id"] + '"><a class="btn btn-xs btn-default btn-link" href="javascript:show_clips_similar_history(\'' + report["target_user_username"] + "', '" + escapeHtml(report["content_id"]) + '\');">Check for other clips around the reported one</a></span><br>';
      html += '<b>Reason:</b> <span class="reason-' + report["reason"] + '">' + report["reason"] + '</span><br>';
      html += '<b>Description:</b><br>';
      html += '<div id="report_description_' + report["id"] + '" class="report-description" style="word-wrap: break-word;">';
      html += htmlEncode(report['description']).replace(/\r\n/g, "\n").replace(/\r/g, "\n").replace(/\n/g, "<br>");
      html += '</div>';
      break;
    case "vod_report":
      html += '<b>Target VOD:</b> <a href="' + gen_vod_url_with_timestamp(report) + '" target="_blank" rel="noreferrer">' + htmlEncode(gen_vod_url_with_timestamp(report)) + '</a><br>';
      html += '<b>Reason:</b> <span class="reason-' + report["reason"] + '">' + report["reason"] + '</span><br>';
      html += '<b>Description:</b><br>';
      html += '<div id="report_description_' + report["id"] + '" class="report-description" style="word-wrap: break-word;">';
      html += htmlEncode(report['description']).replace(/\r\n/g, "\n").replace(/\r/g, "\n").replace(/\n/g, "<br>");
      html += '</div>';
      break;
   case "vod_comment_report":
      html += '<b>Report ID:</b> ' + report["id"] + '<br>';
      html += '<b>Content ID:</b> ' + htmlEncode(report["content_id"]) + '<br>';
      html += '<b>Link to VOD:</b> <span id="vod_comment_link_' + report["id"] + '"><a class="btn btn-xs btn-default btn-link" href="javascript:get_vod_comment_url(\'' + report["id"] + "', '" + escapeHtml(report["extra1"]) + "', '" + escapeHtml(report["content_id"]) + '\');">Get VOD comment URL</a></span><br>';
      html += '<b>Reason:</b> <span class="reason-' + report["reason"] + '">' + report["reason"] + '</span><br>';
      html += '<b>VOD comment content:</b><br>';
      html += '<div id="report_description_' + report["id"] + '" class="report-description" style="word-wrap: break-word;">';
      html += htmlEncode(report['description']).replace(/\r\n/g, "\n").replace(/\r/g, "\n").replace(/\n/g, "<br>");
      html += '</div>';
      break;
    default:
      html += "Unknown content";
  }
  // Violating content auto load preview links
  html += gen_violating_content_preview_links(report);
  html += '</div>';
  html += '<hr>';
  // Report comments
  html += '<div class="row">';
  html += '<span class="pull-right"><button class="btn btn-xs btn-default" onclick="show_report_comment_create_window(' + report["id"] + ');">Create new comment</button></span>';
  html += '</div>';
  html += '<div class="row" id="report_comments_' + report["id"] + '">';
  if(typeof report["comments"] !== "undefined" && report["comments"].length > 0) {
    html += gen_report_comments_html(report["comments"]);
  }
  html += '</div>';
  html += '<hr>';
  // Reporter
  html += '<div class="row">';
  html += '<div class="col-md-7">';
  if(report["from_user_id"] !== null && report["from_user_id"] > 0 && report["from_user_username"] !== null && report["from_user_username"].length > 0) {
    html += 'Reporter: ';
    switch(report["from_user_type"]) {
      case "staff":
        html += '<span class="btn btn-xs btn-danger">Staff</span> ';
        break;
      case "admin":
        html += '<span class="btn btn-xs btn-warning">Admin</span> ';
        break;
      case "global_mod":
        html += '<span class="btn btn-xs btn-success">Global Mod</span> ';
        break;
      default:
    }
    if(report["from_user_partner"] == true) {
      html += "[P] ";
    }
    html += '<a href="' + twitch_base_url + report["from_user_username"] + '?bttvAnonChat=true&amp;tmi_host=false" target="_blank" rel="noreferrer">' + report["from_user_username"] + '</a>';
    html += ' <a href="https://translate.google.com/#auto/en/' + report["from_user_username"] + '" target="_blank" rel="noreferrer" onclick="return translate_popup(this.href, ' + "'" + 'from_user_username' + report["id"] + "'" + ');"><img src="' + translate_image_url + '" title="Translate" alt=""></a>';
    // Display name
    if(report["from_user_display_name"] !== null) {
      html += ' (' + htmlEncode(report["from_user_display_name"].trim()) + ')';
      html += ' <a href="https://translate.google.com/#auto/en/' + encodeURI(report["from_user_display_name"].trim()) + '" target="_blank" rel="noreferrer" onclick="return translate_popup(this.href, ' + "'" + 'from_user_display_name' + report["id"] + "'" + ');"><img src="' + translate_image_url + '" title="Translate" alt=""></a>';
    }
    // Report reporter link
    html += ' <a href="' + twitch_base_url + report["from_user_username"] + '/report_form?tos_ban=true" title="Report" target="_blank" rel="noreferrer"><span class="glyphicon glyphicon-remove-circle"></span></a>';
    // Auto resolve ---
    html += ' | ';
    html += generate_autoresolve_HTML('from', report);
  }
  // Report time
  html += '<div class="time-exact" data-unixtime="' + Math.floor(Date.parse(report['created_at']) / 1000) + '">' + convert_time_to_normal_format(report['created_at']) + '</div>';
  html += '<div class="time-friendly">(' + relativeDate(Math.floor(Date.parse(report['created_at']) / 1000)) + ')</div>';
  if(typeof report["holds"] !== "undefined" && report["holds"].length > 0) {
    html += '<div class="report-holds"><p class="bg-warning">Report was held for: ' + calculate_overall_hold_time(report["holds"]) + '</p></div>';
  }
  html += '</div>';
  // Resolve buttons
  html += '<div class="col-md-5">';
  html += '<div class="pull-right">';
  html += '<div class="btn-group">';
  html += '<button type="button" class="btn btn-success btn-sm" id="assign_and_resolve_' + report["id"] + '" onclick="resolve_report(' + report["id"] + ');">Resolve</button>';
  // Scroll to next unresolved (on normal mode only)
  if(mode == "normal") {
    html += '<button type="button" class="btn btn-default btn-sm" onclick="scroll_to_next_report(\'' + report["id"] + '\');" title="Scroll to next report"><span class="glyphicon glyphicon-arrow-down"></span></button>';
  }
  // Resolve other whisper reports button
  if(report["content"] == "whisper_report") {
    html += '<button type="button" class="btn btn-default btn-sm" id="resolve_others_' + report["id"] + '" onclick="resolve_other_whisper_reports_against_target(\'' + report["target_user_id"] + "', '" + report["id"] + '\');">Resolve others</button>';
  }
  html += '<button type="button" data-toggle="dropdown" class="btn btn-default btn-sm dropdown-toggle" id="assign_to_staff_' + report["id"] + '">Assign to Queue <span class="caret"></span></button>';
  html += '<ul class="dropdown-menu bullet">';
  html += '<li><a class="btn btn-default btn-sm" onclick="bootbox.confirm(' + "'" + 'Are you sure you want to move this report to the check later queue?' + "'" + ', function(result) { if(result === true) { assign_to_check_later(' + report["id"] + '); } });">Check later</a></li>';
  html += '<li><a class="btn btn-default btn-sm" onclick="bootbox.confirm(' + "'" + 'Are you sure you want to escalate this report to Staff?' + "'" + ', function(result) { if(result === true) { assign_to_staff_for_escalation(' + report["id"] + '); } });">Escalation</a></li>';
  // Only need PM check for user reports
  if(report["content"] == "user_report") {
    html += '<li><a class="btn btn-default btn-sm" onclick="bootbox.confirm(' + "'" + 'Are you sure you want to assign this report for a PM Check?' + "'" + ', function(result) { if(result === true) { assign_to_staff_for_pm_check(' + report["id"] + '); } });">PM Check</a></li>';
  }
  html += '</ul>';
  html += '</div>'
  html += '</div>';
  html += '</div>';
  html += '</div>'; // Row
  // Body end ---------------------------------
  html += '</div>';
  // Report end -------------------------------
  html += '</div>';
  html += '<hr class="report-hr" id="report_' + report["id"] + '_hr">';
  return html;
}

function generate_email_HTML(email) {
  var html = "";
  // Report header --------------------------
  html += '<div class="panel panel-default email" id="email_' + email["id"] + '">';
  // "Subject" line --------------------------
  html += '<div class="panel-heading"><b>Email:</b> ' + htmlEncode(email["subject"]) +  '</div>';
  // Body start -------------------------------
  html += '<div class="panel-body" style="word-wrap: normal;">';
  // Report body ------------------------------
  html += '<div class="well">';
  if(email["from"].length > 0) {
    html += '<b>From:</b> ' + htmlEncode(jQuery.parseJSON(email["from"]).join(", ")) + '<br>';
  }
  if(email["to"].length > 0) {
    html += '<b>To:</b> ' + htmlEncode(jQuery.parseJSON(email["to"]).join(", ")) + '<br>';
  }
  if(email["cc"].length > 0) {
    html += '<b>Cc:</b> ' + htmlEncode(jQuery.parseJSON(email["cc"]).join(", ")) + '<br>';
  }
  html += '<hr>';
  html += '<a href="/emails/' + email["id"] + '" target="_blank" rel="noreferrer">Check email</a>';
  html += '</div>';
  html += '<hr>';
  // Resolve buttons
  html += '<div class="row">';
  html += '<div class="col-md-6 pull-right">';
  html += '<div class="pull-right">';
  html += '<button type="button" class="btn btn-success btn-sm" onclick="remove_from_view(' + email["id"] + ');">Remove from view</button>';
  html += '</div>';
  html += '</div>';
  html += '</div>'; // Row
  // Body end ---------------------------------
  html += '</div>';
  // Report end -------------------------------
  html += '</div>';
  html += '<hr class="email-hr" id="email_' + email["id"] + '_hr">';
  return html;
}

function calculate_overall_hold_time(holds) {
  var overall_time = 0;
  for(var i = 0; i < holds.length; i++) {
    var created_at = Date.parse(holds[i]["created_at"]);
    if(holds[i]["disabled_at"] !== null) {
      // If it was disabled before it ended we use the difference between created and disabled
      overall_time = overall_time + (Date.parse(holds[i]["disabled_at"]) - created_at);
    } else {
      // In this case it ran the full duration so we take difference between hold_until and created_at
      overall_time = overall_time + (Date.parse(holds[i]["hold_until"]) - created_at);
    }
  }

  // Do it like this so it's displayed correctly
  return relativeDate(Math.floor(overall_time / 1000) * 2, overall_time);
}

function chatlog_date(timestamp) {
  return moment(timestamp).tz("America/Los_Angeles").format("YYYY_MM_DD");
}

function generate_translation_text(content, description) {
  var translation_text = "";
  switch(content) {
    case "user_report":
      translation_text = description;
      break;
    case "whisper_report":
      var description_lines = description.split("\n");
      $(description_lines).each(function(i, line) {
        // Skip the first line as it has "header" info
        if(i !== 0) {
          var line_content_pos = line.indexOf("\t");
          if(line_content_pos !== -1) {
            translation_text += line.substr(line_content_pos + 1) + "\n\n";
          } else {
            translation_text += line + "\n\n";
          }
        }
      });
      break;
    case "channel_feed_post_report":
      translation_text = description;
      break;
    case "channel_feed_comment_report":
      translation_text = description;
      break;
    case "clip_report":
      translation_text = description;
      break;
    case "vod_report":
      translation_text = description;
      break;
    case "vod_comment_report":
      translation_text = description;
      break;
    default:
      // As default just return the full description
      translation_text = description;
  }

  // URL-encode the content before returning it
  return encodeURIComponent(translation_text);
}

function parse_whisper_description(description, target_user_username) {
  var html = "";
  // Make sure we only have one type of new line indicator
  description = description.replace(/\r\n/g, "\n").replace(/\r/g, "\n");
  var description_lines = description.split("\n");
  // console.log(description_lines);
  html += '<table class="table table-condensed" style="table-layout: fixed;">';
  html += '<colgroup>';
  html += '<col class="col-md-3 text-nowrap">';
  html += '<col class="col-md-2 text-nowrap">';
  html += '<col class="col-md-7" style="word-wrap: break-word;">';
  html += '</colgroup>';
  html += '<thead>';
  html += '<tr>';
  html += '<th>Time</th>';
  html += '<th>From</th>';
  html += '<th>Body</th>';
  html += '</tr>';
  html += '</thead>';
  html += '<tbody>';
  for(var i = 1; i < description_lines.length; i++) {
    var line_split = description_lines[i].split("\t");
    html += '<tr>';
    // Time
    html += '<td><abbr title="' + relativeDate(Math.floor(moment.tz(line_split[0], "America/Los_Angeles").format("X"))) + '">' + htmlEncode(line_split[0]) + '</abbr></td>';
    // Sender
    html += '<td>';
    if(line_split[1] == target_user_username) {
      html += '[T] ';
    }
    html += '<a href="' + twitch_base_url + encodeURIComponent(line_split[1]) + '?bttvAnonChat=true&amp;tmi_host=false" target="_blank" rel="noreferrer">' + htmlEncode(line_split[1]) + '</a>';
    html += '</td>';
    // Message
    html += '<td>' + htmlEncode(line_split.splice(2).join("\t")) + '</td>';
  }
  html += '</tbody>';
  html += '</table>';
  return html;
}

function convert_time_to_normal_format(timestamp) {
  return moment(timestamp).tz("America/Los_Angeles").format("YYYY-MM-DD HH:mm:ss z");
}

function boolean_to_string(boolean) {
  if(boolean === true) {
    return "true";
  } else if (boolean === false){
    return "false";
  }
}

function check_report_report_version_difference(current) {
  if(current !== null && current != "null" && local_report_version !== null && local_report_version != "null") {
    if(current > local_report_version) {
      // check if there isn't already a box and if there isn't display one
      if($("#report_version_change_bootbox").length == 0) {
        bootbox.confirm('<span id="report_version_change_bootbox">Another admin has joined or the site has been updated, are you ready to reload the page?<br>If not please click &quot;Cancel&quot; and reload it manually after you&#039;re done with your current case.</span>', function(result) { if(result === true) { window.location.reload(); } });
        // Set the page value to the current value so this pop-up doesn't come up again unless the value changes again
        local_report_version = current;
      }
      console.log("Report version has changed");
    }
  }
}

function update_report_number_in_title(number) {
  var current = $(document).attr("title");
  if(current.match(/^\(\-?[0-9]+\) .*$/i) == null) {
    // Now number exists yet so just prepend it to the current title
    $(document).attr("title", "(" + number + ") " + current);
  } else {
    // Replace the current number
    $(document).attr("title", current.replace(/(\-?[0-9]+)/i, number));
  }

  check_for_reload();
}

function get_report_number_from_title() {
  if($(document).attr("title").match(/(\-?[0-9]+)/i) !== null) {
    return $(document).attr("title").match(/(\-?[0-9]+)/i)[0];
  } else {
    return $(".report").length;
  }
}

function htmlEncode(value) {
  return $("<div/>").text(value).html();
}

function htmlDecode(value) {
  return $("<div/>").html(value).text();
}

function escapeHtml(string) {
  var entityMap = {
    "&": "&amp;",
    "<": "&lt;",
    ">": "&gt;",
    '"': '&quot;',
    "'": '&#39;',
    "/": '&#x2F;'
  };

  return String(string).replace(/[&<>"'\/]/g, function (s) {
    return entityMap[s];
  });
}

function resolve_report(report_id, add_new_after_done) {
  var add_new_after_done = typeof add_new_after_done !== 'undefined' ? add_new_after_done : true;

  $.ajax({
    url: "/reports/" + report_id,
    type: "PATCH",
    timeout: 10000, // sets timeout to 10 seconds
    data: { do_action: "resolve", authenticity_token: $("meta[name=csrf-token]").attr("content") },
    success: function(result, textStatus, jqXHR) {
      // Check if another admin joined and we redistributed reports or if the site changed (after a push or something)
      check_report_report_version_difference(jqXHR.getResponseHeader("X-Report-Version"));

      // Update the last action time
      last_action_time = current_unixtime();

      if(mode == "normal") {
        $("#assign_and_resolve_" + report_id).html("Report resolved.");
        // Reduce the report counter in the title
        update_report_number_in_title(get_report_number_from_title() - 1);
      } else if (mode == "fade") {
        remove_report(report_id);
      }

    },
    error: function(jqXHR, textStatus, errorThrown) {
      if(mode == "normal") {
        // Try again
        $("#assign_and_resolve_" + report_id).html(errorThrown + '... Try again');
        $("#assign_and_resolve_" + report_id).removeAttr('disabled');
      } else if (mode == "fade") {
        remove_report(report_id, false);
      }
    }
  });

  $("#assign_and_resolve_" + report_id).html('Resolving ...');
  $("#assign_and_resolve_" + report_id).attr("disabled", "disabled");
  // Disable other buttons
  $("#assign_to_staff_" + report_id).attr("disabled", "disabled");
  // Remove the stream preview on report resolve
  if($("#stream_preview_" + report_id).length > 0 && $("#stream_preview_" + report_id).html().length > 0)
  {
     $("#stream_preview_" + report_id).html('');
  }

  fadeOut_report(report_id, add_new_after_done);
}

function assign_to_staff_for_escalation(report_id, add_new_after_done) {
  var add_new_after_done = typeof add_new_after_done !== 'undefined' ? add_new_after_done : true;

  $.ajax({
    url: "/reports/" + report_id,
    type: "PATCH",
    timeout: 10000, // sets timeout to 10 seconds
    data: { do_action: "staff_escalation", authenticity_token: $("meta[name=csrf-token]").attr("content") },
    success: function(result, textStatus, jqXHR) {
      // Check if another admin joined and we redistributed reports or if the site changed (after a push or something)
      check_report_report_version_difference(jqXHR.getResponseHeader("X-Report-Version"));

      // Update the last action time
      last_action_time = current_unixtime();

      if(mode == "normal") {
        $("#assign_to_staff_" + report_id).html("Report assigned to staff.");
        // Reduce the report counter in the title
        update_report_number_in_title(get_report_number_from_title() - 1);
      } else if (mode == "fade") {
        remove_report(report_id);
      }

    },
    error: function(jqXHR, textStatus, errorThrown) {
      if(mode == "normal") {
        // Try again
        $("#assign_to_staff_" + report_id).html(errorThrown + '... Try again');
        $("#assign_to_staff_" + report_id).removeAttr('disabled');
      } else if (mode == "fade") {
        remove_report(report_id, false);
      }
    }
  });

  $("#assign_to_staff_" + report_id).html('Assigning ...');
  $("#assign_to_staff_" + report_id).attr("disabled", "disabled");
  // Disable other buttons
  $("#assign_and_resolve_" + report_id).attr("disabled", "disabled");
  // Remove the stream preview on report resolve
  if($("#stream_preview_" + report_id).length > 0 && $("#stream_preview_" + report_id).html().length > 0)
  {
     $("#stream_preview_" + report_id).html('');
  }

  fadeOut_report(report_id, add_new_after_done);
}

function assign_to_staff_for_pm_check(report_id, add_new_after_done) {
  var add_new_after_done = typeof add_new_after_done !== 'undefined' ? add_new_after_done : true;

  $.ajax({
    url: "/reports/" + report_id,
    type: "PATCH",
    timeout: 10000, // sets timeout to 10 seconds
    data: { do_action: "pm_check", authenticity_token: $("meta[name=csrf-token]").attr("content") },
    success: function(result, textStatus, jqXHR) {
      // Check if another admin joined and we redistributed reports or if the site changed (after a push or something)
      check_report_report_version_difference(jqXHR.getResponseHeader("X-Report-Version"));

      // Update the last action time
      last_action_time = current_unixtime();

      if(mode == "normal") {
        $("#assign_to_staff_" + report_id).html("Report assigned to staff.");
        // Reduce the report counter in the title
        update_report_number_in_title(get_report_number_from_title() - 1);
      } else if (mode == "fade") {
        remove_report(report_id);
      }

    },
    error: function(jqXHR, textStatus, errorThrown) {
      if(mode == "normal") {
        // Try again
        $("#assign_to_staff_" + report_id).html(errorThrown + '... Try again');
        $("#assign_to_staff_" + report_id).removeAttr('disabled');
      } else if (mode == "fade") {
        remove_report(report_id, false);
      }
    }
  });

  $("#assign_to_staff_" + report_id).html('Assigning ...');
  $("#assign_to_staff_" + report_id).attr("disabled", "disabled");
  // Disable other buttons
  $("#assign_and_resolve_" + report_id).attr("disabled", "disabled");
  // Remove the stream preview on report resolve
  if($("#stream_preview_" + report_id).length > 0 && $("#stream_preview_" + report_id).html().length > 0)
  {
     $("#stream_preview_" + report_id).html('');
  }

  fadeOut_report(report_id, add_new_after_done);
}

function assign_to_check_later(report_id, add_new_after_done) {
  var add_new_after_done = typeof add_new_after_done !== 'undefined' ? add_new_after_done : true;

  $.ajax({
    url: "/reports/" + report_id,
    type: "PATCH",
    timeout: 10000, // sets timeout to 10 seconds
    data: { do_action: "check_later", authenticity_token: $("meta[name=csrf-token]").attr("content") },
    success: function(result, textStatus, jqXHR) {
      // Check if another admin joined and we redistributed reports or if the site changed (after a push or something)
      check_report_report_version_difference(jqXHR.getResponseHeader("X-Report-Version"));

      // Update the last action time
      last_action_time = current_unixtime();

      if(mode == "normal") {
        $("#assign_to_staff_" + report_id).html("Report moved to check later.");
        // Reduce the report counter in the title
        update_report_number_in_title(get_report_number_from_title() - 1);
      } else if (mode == "fade") {
        remove_report(report_id);
      }

    },
    error: function(jqXHR, textStatus, errorThrown) {
      if(mode == "normal") {
        // Try again
        $("#assign_to_staff_" + report_id).html(errorThrown + '... Try again');
        $("#assign_to_staff_" + report_id).removeAttr('disabled');
      } else if (mode == "fade") {
        remove_report(report_id, false);
      }
    }
  });

  $("#assign_to_staff_" + report_id).html('Assigning ...');
  $("#assign_to_staff_" + report_id).attr("disabled", "disabled");
  // Disable other buttons
  $("#assign_and_resolve_" + report_id).attr("disabled", "disabled");
  // Remove the stream preview on report resolve
  if($("#stream_preview_" + report_id).length > 0 && $("#stream_preview_" + report_id).html().length > 0)
  {
     $("#stream_preview_" + report_id).html('');
  }

  fadeOut_report(report_id, add_new_after_done);
}

function check_for_reload() {
  if(mode == "normal") {
    if(parseInt($(document).attr("title").match(/(\-?[0-9]+)/i)[0]) <= 0) {
      // Only do this if we have reports on the page, if not the getReports() function deals with this
      if($(".report").length > 0) {
        // Clear all current reports from the page (as they are resolved in this case)
        setTimeout(function () {
            $(".report").remove();
            $(".report-hr").remove();
            set_waiting_for_reports_text();
           }, 1000);

        // Request new reports
        setTimeout(function () { getReports(); }, 2500);
      }
    }
  }
}

function fadeOut_report(report_id, add_new_after_done) {
  var add_new_after_done = typeof add_new_after_done !== 'undefined' ? add_new_after_done : true;

  if(mode == "fade") {
     $.when(
        $("#report_" + report_id).fadeOut({ duration: 200, queue: false }) &&
        $("#report_" + report_id + "_hr").fadeOut({ duration: 200, queue: false })
      )
      .then(
        function() {
          if(add_new_after_done === true) {
            handle_reports();

            // Update the page title with the current report count
            update_report_number_in_title($(".report").length);
          }
        }
    );
  }
}

function remove_report(report_id, count_as_handled) {
  var count_as_handled = typeof count_as_handled !== 'undefined' ? count_as_handled : true;

  // Remove the report
  $("#report_" + report_id).remove();

  // Remove the hr below it
  $("#report_" + report_id + "_hr").remove();

  if(count_as_handled === true) {
    // Add the report_id with the current time to the array so we know that we worked it and when
    handled_reports[report_id] = current_unixtime();
  }

  // Update report number in title
  update_report_number_in_title($(".report").length);

  if(mode == "fade" && $(".report").length <= 0) {
    set_waiting_for_reports_text();
  }
}

function remove_from_view(email_id) {
  // Remove the email
  $("#email_" + email_id).remove();

  // Remove the hr below it
  $("#email_" + email_id + "_hr").remove();

  // Add the email_id with the current time to the array so we know that we worked it and when
  handled_emails[email_id] = current_unixtime();
}

function current_unixtime() {
  return Math.floor(Date.now() / 1000);
}

function set_waiting_for_reports_text() {
  $("#report_loading_info").text("Waiting for new reports to arrive ...");
}

function clear_waiting_for_reports_text() {
  $("#report_loading_info").empty();
}

function play_sound_notification() {
  if(sound_notification_url.length > 0) {
    var audio = new Audio(sound_notification_url);
    audio.play();
  }
}

function generate_autoresolve_HTML(direction, data_object) {
  var html = '<div class="btn-group">';
  html += '<button type="button" data-toggle="dropdown" class="btn btn-default btn-sm dropdown-toggle"><span class="glyphicon glyphicon-ok"></span> <span class="caret"></span></button>';
  html += '<ul class="dropdown-menu bullet">';
  html += '<li><p class="text-center"><b>Autoresolve duration:</b></p></li>';
  if(direction == "target") {
    html += '<li><button class="btn btn-default btn-sm btn-block" onclick="show_autoresolve_create_window(' + "'" + data_object['target_user_username'] + "'" + ', null, ' + "'" + data_object['content'] + "'" + ', 5)">5 sec</button></li>';
    html += '<li><button class="btn btn-default btn-sm btn-block" onclick="show_autoresolve_create_window(' + "'" + data_object['target_user_username'] + "'" + ', null, ' + "'" + data_object['content'] + "'" + ', 300)">5 min</button></li>';
    html += '<li><button class="btn btn-default btn-sm btn-block" onclick="show_autoresolve_create_window(' + "'" + data_object['target_user_username'] + "'" + ', null, ' + "'" + data_object['content'] + "'" + ', 600)">10 min</button></li>';
    html += '<li><button class="btn btn-default btn-sm btn-block" onclick="show_autoresolve_create_window(' + "'" + data_object['target_user_username'] + "'" + ', null, ' + "'" + data_object['content'] + "'" + ', 900)">15 min</button></li>';
    html += '<li><button class="btn btn-default btn-sm btn-block" onclick="show_autoresolve_create_window(' + "'" + data_object['target_user_username'] + "'" + ', null, ' + "'" + data_object['content'] + "'" + ', 1800)">30 min</button></li>';
  } else if (direction == "from") {
    html += '<li><button class="btn btn-default btn-sm btn-block" onclick="show_autoresolve_create_window(null, ' + "'" + data_object['from_user_username'] + "'" + ', ' + "'" + data_object['content'] + "'" + ', 5)">5 sec</button></li>';
    html += '<li><button class="btn btn-default btn-sm btn-block" onclick="show_autoresolve_create_window(null, ' + "'" + data_object['from_user_username'] + "'" + ', ' + "'" + data_object['content'] + "'" + ', 300)">5 min</button></li>';
    html += '<li><button class="btn btn-default btn-sm btn-block" onclick="show_autoresolve_create_window(null, ' + "'" + data_object['from_user_username'] + "'" + ', ' + "'" + data_object['content'] + "'" + ', 600)">10 min</button></li>';
    html += '<li><button class="btn btn-default btn-sm btn-block" onclick="show_autoresolve_create_window(null, ' + "'" + data_object['from_user_username'] + "'" + ', ' + "'" + data_object['content'] + "'" + ', 900)">15 min</button></li>';
    html += '<li><button class="btn btn-default btn-sm btn-block" onclick="show_autoresolve_create_window(null, ' + "'" + data_object['from_user_username'] + "'" + ', ' + "'" + data_object['content'] + "'" + ', 1800)">30 min</button></li>';
  }
  html += '</ul>';
  html += '</div>';
  return html;
}

function generate_hold_HTML(data_object) {
  var html = '<div class="btn-group">';
  html += '<button type="button" data-toggle="dropdown" class="btn btn-default btn-sm dropdown-toggle"><span class="glyphicon glyphicon-tag"></span> <span class="caret"></span></button>';
  html += '<ul class="dropdown-menu bullet">';
  html += '<li><p class="text-center"><b>Hold duration:</b></p></li>';
  html += '<li><button class="btn btn-default btn-sm btn-block" onclick="create_hold(' + data_object['id'] + ', 300)">5 min</button></li>';
  html += '<li><button class="btn btn-default btn-sm btn-block" onclick="create_hold(' + data_object['id'] + ', 600)">10 min</button></li>';
  html += '<li><button class="btn btn-default btn-sm btn-block" onclick="create_hold(' + data_object['id'] + ', 900)">15 min</button></li>';
  html += '<li><button class="btn btn-default btn-sm btn-block" onclick="create_hold(' + data_object['id'] + ', 1800)">30 min</button></li>';
  html += '<li><button class="btn btn-default btn-sm btn-block" onclick="create_hold(' + data_object['id'] + ', 3600)">60 min</button></li>';
  html += '</ul>';
  html += '</div>';
  return html;
}

function create_hold(report_id, duration) {
  $.ajax({
    url: "/report_holds",
    type: "POST",
    timeout: 10000,
    data: {
      "report_hold[report_id]": report_id,
      "duration": duration,
      "authenticity_token": $("meta[name=csrf-token]").attr("content")
    },
    success: function(result, textStatus, jqXHR) {
      // Check if another admin joined and we redistributed reports or if the site changed (after a push or something)
      check_report_report_version_difference(jqXHR.getResponseHeader("X-Report-Version"));

      if(result.status == 200) {
        show_notification(result.message, "success");
        handle_holds(result.holds);
        $("#report_" + report_id).remove();
        $("#report_" + report_id + "_hr").remove();

        // Recalculate current report number
        if(mode == "fade") {
          update_report_number_in_title($(".report").length);
        } else if (mode == "normal") {
          update_report_number_in_title(get_report_number_from_title() - 1);
        }

        if(get_report_number_from_title() <= 0) {
          set_waiting_for_reports_text();
          if(mode == "normal") {
            getReports();
          }
        }
      } else {
        show_notification(result.message);
      }
    },
    error: function(jqXHR, textStatus, errorThrown) {
      // Error
      show_notification("There was some kind of error creating the hold");
    },
  });

  console.log("Creating hold request send");
}

function show_autoresolve_create_window(target_user_username, from_user_username, content, duration) {
  var target_user_username = target_user_username !== null ? target_user_username : "";
  var from_user_username = from_user_username !== null ? from_user_username : "";

  console.log("Showing autoresolve create window");

  bootbox.dialog({
    title: "Create autoresolve",
    message: '<div class="row">' +
              '<div class="col-md-12">' +
                '<form class="form-horizontal" id="autoresolve_create_form" onsubmit="return submit_autoresolve_create_form();">' +
                  '<div id="autoresolve_create_form_notifications"></div>' +
                  '<div class="form-group">' +
                    '<label class="col-md-3 control-label">Target user</label>' +
                    '<div class="col-md-3">' +
                      '<select class="form-control input-md" name="target_user_username_option">' +
                        '<option value="is" selected="selected">IS</option>' +
                        '<option value="is_not">IS NOT</option>' +
                      '</select>' +
                    '</div>' +
                    '<div class="col-md-5">' +
                      '<input type="text" name="target_user_username" value="' + target_user_username + '" class="form-control input-md">' +
                    '</div>' +
                  '</div>' +
                  '<div class="form-group">' +
                    '<label class="col-md-3 control-label">From user</label>' +
                    '<div class="col-md-3">' +
                      '<select class="form-control input-md" name="from_user_username_option">' +
                        '<option value="is" selected="selected">IS</option>' +
                        '<option value="is_not">IS NOT</option>' +
                      '</select>' +
                    '</div>' +
                    '<div class="col-md-5">' +
                      '<input type="text" name="from_user_username" value="' + from_user_username + '" class="form-control input-md">' +
                    '</div>' +
                  '</div>' +
                  '<div class="form-group">' +
                    '<label class="col-md-3 control-label">Content</label>' +
                    '<div class="col-md-5">' +
                      '<select class="form-control input-md" name="content">' +
                        '<option value="channel_feed_comment_report"' + select_selected_helper('channel_feed_comment_report', content) + '>Channel feed comment reports</option>' +
                        '<option value="channel_feed_post_report"' + select_selected_helper('channel_feed_post_report', content) + '>Channel feed post reports</option>' +
                        '<option value="clip_report"' + select_selected_helper('clip_report', content) + '>Clip reports</option>' +
                        '<option value="user_report"' + select_selected_helper('user_report', content) + '>User reports</option>' +
                        '<option value="vod_comment_report"' + select_selected_helper('vod_comment_report', content) + '>VOD comment reports</option>' +
                        '<option value="vod_report"' + select_selected_helper('vod_report', content) + '>VOD reports</option>' +
                        '<option value="whisper_report"' + select_selected_helper('whisper_report', content) + '>Whisper reports</option>' +
                      '</select>' +
                    '</div>' +
                  '</div>' +
                  '<div class="form-group">' +
                    '<label class="col-md-3 control-label">Duration</label>' +
                    '<div class="col-md-5">' +
                      '<select class="form-control input-md" name="duration">' +
                        '<option value="5"' + select_selected_helper(5, duration) + '>5 sec</option>' +
                        '<option value="300"' + select_selected_helper(300, duration) + '>5 min</option>' +
                        '<option value="600"' + select_selected_helper(600, duration) + '>10 min</option>' +
                        '<option value="900"' + select_selected_helper(900, duration) + '>15 min</option>' +
                        '<option value="1800"' + select_selected_helper(1800, duration) + '>30 min</option>' +
                      '</select>' +
                    '</div>' +
                  '</div>' +
                '</form>' +
              '</div>' +
            '</div>',
    buttons: {
      success: {
        label: "Create AutoResolve",
          className: "btn-success auto-resolve-create-button auto-resolve-create-button-success",
          callback: function () {
            var data = {};
            data['target_user_username'] = $('#autoresolve_create_form').find('input[name="target_user_username"]').val();
            data['from_user_username'] = $('#autoresolve_create_form').find('input[name="from_user_username"]').val();
            data['content'] = $('#autoresolve_create_form').find('select[name="content"]').val();
            data['duration'] = $('#autoresolve_create_form').find('select[name="duration"]').val();
            data['target_user_username_option'] = $('#autoresolve_create_form').find('select[name="target_user_username_option"]').val();
            data['from_user_username_option'] = $('#autoresolve_create_form').find('select[name="from_user_username_option"]').val();
            if(data['target_user_username_option'] == "is_not" && data['from_user_username_option'] == "is_not") {
              // Display error that this is not allowed (would resolve everything except this combo)
              show_notification("It's not possible to have both set to \"IS NOT\". This would resolve all reports except for this combo, that's bad.", undefined, "autoresolve_create_form_notifications");
              return false;
            }
            create_autoresolve(data['target_user_username'], data['target_user_username_option'], data['from_user_username'], data['from_user_username_option'], data['content'], data['duration']);
            // console.log(data);

            // Disable the buttons and change the text to waiting
            $(".auto-resolve-create-button").attr("disabled", "disabled");
            $(".auto-resolve-create-button-success").text("Creating ...");

            // Return false so the window doesn't get closed
            return false;
          }
      },
      danger: {
        label: "Cancel",
        className: "btn-danger auto-resolve-create-button",
        callback: function() {
          return true;
        }
      }
    }
  });
}

function submit_autoresolve_create_form() {
  // Click the create button on submission (mostly just so we can catch ENTER)
  $(".auto-resolve-create-button-success").click();

  return false;
}

function select_selected_helper(option_content, real_content) {
  if(option_content == real_content) {
    return ' selected="selected"';
  }
  return '';
}

function create_autoresolve(target_user_username, target_user_username_option, from_user_username, from_user_username_option, content, duration) {
  $.ajax({
    url: "/auto_resolves",
    type: "POST",
    timeout: 10000,
    data: {
      "auto_resolve[content]": content,
      "target_user_username": target_user_username,
      "target_user_username_option": target_user_username_option,
      "from_user_username": from_user_username,
      "from_user_username_option": from_user_username_option,
      "duration": duration,
      "authenticity_token": $("meta[name=csrf-token]").attr("content")
    },
    success: function(result, textStatus, jqXHR) {
      // Remove the autoresolve window
      $(".auto-resolve-create-button-success").closest(".modal-content").find(".bootbox-close-button").click();

      // Check if another admin joined and we redistributed reports or if the site changed (after a push or something)
      check_report_report_version_difference(jqXHR.getResponseHeader("X-Report-Version"));

      if(result.status == 200) {
        show_notification(result.message, "success");
        handle_autoresolves(result.autoresolves);
        // Because autoresolves could affect the currently visible reports too (Probably more often than not)
        // we should remove those cases. Removing only them would be best but that would require us having the
        // autoresolve logic in JS too. So for now we just remove all reports and set the page so it loads them fresh
        $(".report").remove();
        $(".report-hr").remove();
        set_waiting_for_reports_text();
        getReports();
      } else {
        show_notification(result.message);
      }
    },
    error: function(jqXHR, textStatus, errorThrown) {
      // Remove the autoresolve window
      $(".auto-resolve-create-button-success").closest(".modal-content").find(".bootbox-close-button").click();

      // Error
      show_notification("There was some kind of error creating the autoresolve");
    },
  });

  console.log("Creating autoresolve request send");
}

function disable_autoresolve(autoresolve_id) {
  $.ajax({
    url: "/auto_resolves/" + autoresolve_id,
    type: "PATCH",
    data: {
      "do_action": "disable",
      "authenticity_token": $("meta[name=csrf-token]").attr("content")
    },
    success: function(result, textStatus, jqXHR) {
      // Check if another admin joined and we redistributed reports or if the site changed (after a push or something)
      check_report_report_version_difference(jqXHR.getResponseHeader("X-Report-Version"));

      if(result.status == 200) {
        show_notification(result.message, "success");
        handle_autoresolves(result.autoresolves);
      } else {
        show_notification(result.message);
      }
    },
    error: function(jqXHR, textStatus, errorThrown) {
      // Error
      show_notification("There was some kind of error disabling the autoresolve");
    },
  });
  console.log("Disabled autoresolve " + autoresolve_id);
}

function disable_hold(hold_id) {
  $.ajax({
    url: "/report_holds/" + hold_id,
    type: "PATCH",
    data: {
      "do_action": "disable",
      "authenticity_token": $("meta[name=csrf-token]").attr("content")
    },
    success: function(result, textStatus, jqXHR) {
      // Check if another admin joined and we redistributed reports or if the site changed (after a push or something)
      check_report_report_version_difference(jqXHR.getResponseHeader("X-Report-Version"));

      if(result.status == 200) {
        show_notification(result.message, "success");
        handle_holds(result.holds);
      } else {
        show_notification(result.message);
      }
    },
    error: function(jqXHR, textStatus, errorThrown) {
      // Error
      show_notification("There was some kind of error disabling the hold");
    },
  });
  console.log("Disabled hold " + hold_id);
}

function join_active_admins() {
  $.ajax({
    url: "/report_admins",
    type: "POST",
    data: {
      "authenticity_token": $("meta[name=csrf-token]").attr("content")
    },
    success: function(result, textStatus, jqXHR) {
      // Check if another admin joined and we redistributed reports or if the site changed (after a push or something)
      check_report_report_version_difference(jqXHR.getResponseHeader("X-Report-Version"));

      if(result.status == 200) {
        $("#active_admins_join").empty();
        show_notification(result.message, "success");
        handle_active_admins(result.active_admins);
      } else {
        show_notification(result.message);
      }
    },
    error: function(jqXHR, textStatus, errorThrown) {
      // Error
      show_notification("There was some kind of error joining the active admins");
    },
  });
  console.log("Joined active admins");
}


function disable_active_admin(active_admin_id) {
  $.ajax({
    url: "/report_admins/" + active_admin_id,
    type: "PATCH",
    data: {
      "do_action": "disable",
      "authenticity_token": $("meta[name=csrf-token]").attr("content")
    },
    success: function(result, textStatus, jqXHR) {
      // Check if another admin joined and we redistributed reports or if the site changed (after a push or something)
      check_report_report_version_difference(jqXHR.getResponseHeader("X-Report-Version"));

      if(result.status == 200) {
        show_notification(result.message, "success");
        handle_active_admins(result.active_admins);
      } else {
        show_notification(result.message);
      }
    },
    error: function(jqXHR, textStatus, errorThrown) {
      // Error
      show_notification("There was some kind of error disabling the active admin");
    },
  });
  console.log("Disabled active_admin " + active_admin_id);
}

function modify_active_admin_weight(active_admin_id, weight) {
  $.ajax({
    url: "/report_admins/" + active_admin_id,
    type: "PATCH",
    data: {
      "do_action": "modify_weight",
      "weight": weight,
      "authenticity_token": $("meta[name=csrf-token]").attr("content")
    },
    success: function(result, textStatus, jqXHR) {
      // Check if another admin joined and we redistributed reports or if the site changed (after a push or something)
      check_report_report_version_difference(jqXHR.getResponseHeader("X-Report-Version"));

      if(result.status == 200) {
        show_notification(result.message, "success");
        handle_active_admins(result.active_admins);
      } else {
        show_notification(result.message);
      }
    },
    error: function(jqXHR, textStatus, errorThrown) {
      // Error
      show_notification("There was some kind of error changing the weight of the admin");
    },
  });
  console.log("Changing active_admin " + active_admin_id + " weight to " + weight);
}

function show_active_admin_weight_change_bootbox(id, current_weight) {
  bootbox.prompt(
    {
      title: 'Set the distribution weight (10 is default). Allowed values are 1 - 100.',
      value: current_weight,
      callback: function(result) {
        if(result !== null) {
          modify_active_admin_weight(id, parseInt(result));
        }
      }
    });
}

function show_notification(content, alert_type, target_div, escape_content) {
  var target_div = typeof target_div !== 'undefined' ? target_div : "notifications";
  var alert_type = typeof alert_type !== 'undefined' ? alert_type : "warning";
  var escape_content = typeof escape_content !== 'undefined' ? escape_content : true;
  var random_id = 'random_' + Date.now();

  if(escape_content === true) {
    var content = htmlEncode(content);
  }
  var html = '<div class="alert alert-' + alert_type + ' alert-dismissible" id="' + random_id + '" role="alert">' +
             '<button type="button" class="close" data-dismiss="alert"><span aria-hidden="true">&times;</span><span class="sr-only">Close</span></button>' +
              content +
              '</div>';

  // Add to the DIV
  $('#' + target_div).append(html);

  // Set time for it to get purged from the site
  setTimeout(function () { $('#' + random_id).remove(); }, 5000);

  // Let it fade out after 4 seconds
  $('#' + random_id).delay(4000).fadeOut();
}

function create_profile_image_preview(report_id, channel, via_autoload) {
  var via_autoload = typeof via_autoload !== 'undefined' ? via_autoload : false;

  if($("#profile_image_preview_" + report_id).html().length === 0) {
    $("#profile_image_preview_" + report_id).html("Loading ...<hr>");
    $.getJSON("https://api.twitch.tv/kraken/users/" + channel + "?client_id=" + twitch_kraken_client_id + "&callback=?", function() {})
      .done(function(json) {
        if(typeof json.logo !== "undefined" && json.logo !== null && json.logo.length > 0) {
          // Clear it first
          $("#profile_image_preview_" + report_id).html("");
          // Add the content
          $("#profile_image_preview_" + report_id).append([
            // Upload and get report link button
            $("<span/>", { class: "pull-right", id: "profile_image_upload_" + report_id, html: '<a class="btn btn-xs btn-default btn-link" href="javascript:upload_channel_image(' + "'" + report_id + "', '" + channel  + "', 'profile_image', '" + json.logo.replace("-300x300.", "-600x600.") + "'" + ');">Upload and gen report link</a>' } ),
            // Normal Data output
              $("<div/>").append([
              $("<strong/>", { text: "User: " }),
              $("<span/>", { text: json.name })
            ]),
              $("<div/>").append([
              $("<strong/>", { text: "Profile image: " })
            ]),
              $("<div/>").append([
              $("<img/>", { src: json.logo.replace("-300x300.", "-600x600."), width: "600", height: "600" }),
            ]),
              $("<div/>").append([
              $("<hr/>")
            ])
          ]);
        } else if(typeof json.error !== "undefined" && json.error !== null & typeof json.status !== "undefined" && json.status !== null && json.status == 422 && json.message.indexOf(" is unavailable") != -1) {
          // Clear it first
          $("#profile_image_preview_" + report_id).html("");
          // Add the content
          $("#profile_image_preview_" + report_id).append([
              $("<div/>", { "class": report_id + "_user" }).append([
              $("<strong/>", { text: "User: " }),
              $("<span/>", { text: channel })
            ]),
              $("<div/>").append([
              $("<span/>", { text: "This user is TOS'd." })
            ]),
              $("<div/>").append([
              $("<hr/>")
            ])
          ]);
        } else if(typeof json.error !== "undefined" && json.error !== null & typeof json.status !== "undefined" && json.status !== null && json.status == 422 && json.message.indexOf(" is not available on Twitch") != -1) {
          // Clear it first
          $("#profile_image_preview_" + report_id).html("");
          // Add the content
          $("#profile_image_preview_" + report_id).append([
              $("<div/>").append([
              $("<strong/>", { text: "User: " }),
              $("<span/>", { text: channel })
            ]),
              $("<div/>").append([
              $("<span/>", { text: "This user is a JTV account." })
            ]),
              $("<div/>").append([
              $("<hr/>")
            ])
          ]);
        } else {
          // Clear it first
          $("#profile_image_preview_" + report_id).html("");
          // Add the content
          $("#profile_image_preview_" + report_id).append([
              $("<div/>").append([
              $("<strong/>", { text: "User: " }),
              $("<span/>", { text: channel })
            ]),
              $("<div/>").append([
              $("<strong/>", { text: "Profile image: " })
            ]),
              $("<div/>").append([
              $("<span/>", { text: "This user doesn't have a profile image." })
            ]),
              $("<div/>").append([
              $("<hr/>")
            ])
          ]);
        }
      })
      .fail(function(jqXHR, textStatus, error) {
        var err = jqXHR.status + ", " + textStatus + ", " + error;
        console.log("Request Failed: " + err);
        // Try again
        create_profile_image_preview(report_id, channel);
      });
  } else {
    if(via_autoload === false) {
      $("#profile_image_preview_" + report_id).html("");
    }
  }
}

function create_profile_banner_preview(report_id, channel, via_autoload) {
  var via_autoload = typeof via_autoload !== 'undefined' ? via_autoload : false;

  if($("#profile_banner_preview_" + report_id).html().length === 0) {
    $("#profile_banner_preview_" + report_id).html("Loading ...<hr>");
    $.getJSON("https://api.twitch.tv/kraken/channels/" + channel + "?client_id=" + twitch_kraken_client_id + "&callback=?", function() {})
      .done(function(json) {
        if(typeof json.profile_banner !== "undefined" && json.profile_banner !== null && json.profile_banner.length > 0) {
          // Clear it first
          $("#profile_banner_preview_" + report_id).html("");
          // Add the content
          $("#profile_banner_preview_" + report_id).append([
            // Upload and get report link button
            $("<span/>", { class: "pull-right", id: "profile_banner_upload_" + report_id, html: '<a class="btn btn-xs btn-default btn-link" href="javascript:upload_channel_image(' + "'" + report_id + "', '" + channel + "', 'profile_banner', '" + json.profile_banner + "'" + ');">Upload and gen report link</a>' } ),
            // Normal Data output
              $("<div/>").append([
              $("<strong/>", { text: "User: " }),
              $("<span/>", { text: json.name })
            ]),
              $("<div/>").append([
              $("<strong/>", { text: "Profile banner: " })
            ]),
              $("<div/>").append([
              $("<img/>", { src: json.profile_banner, style: "max-width:550px;" }),
            ]),
              $("<div/>").append([
              $("<hr/>")
            ])
          ]);
        } else if(typeof json.error !== "undefined" && json.error !== null & typeof json.status !== "undefined" && json.status !== null && json.status == 422 && json.message.indexOf(" is unavailable") != -1) {
          // Clear it first
          $("#profile_banner_preview_" + report_id).html("");
          // Add the content
          $("#profile_banner_preview_" + report_id).append([
              $("<div/>", { "class": report_id + "_user" }).append([
              $("<strong/>", { text: "User: " }),
              $("<span/>", { text: channel })
            ]),
              $("<div/>").append([
              $("<span/>", { text: "This user is TOS'd." })
            ]),
              $("<div/>").append([
              $("<hr/>")
            ])
          ]);
        } else if(typeof json.error !== "undefined" && json.error !== null & typeof json.status !== "undefined" && json.status !== null && json.status == 422 && json.message.indexOf(" is not available on Twitch") != -1) {
          // Clear it first
          $("#profile_banner_preview_" + report_id).html("");
          // Add the content
          $("#profile_banner_preview_" + report_id).append([
              $("<div/>").append([
              $("<strong/>", { text: "User: " }),
              $("<span/>", { text: channel })
            ]),
              $("<div/>").append([
              $("<span/>", { text: "This user is a JTV account." })
            ]),
              $("<div/>").append([
              $("<hr/>")
            ])
          ]);
        } else {
          // Clear it first
          $("#profile_banner_preview_" + report_id).html("");
          // Add the content
          $("#profile_banner_preview_" + report_id).append([
              $("<div/>").append([
              $("<strong/>", { text: "User: " }),
              $("<span/>", { text: channel })
            ]),
              $("<div/>").append([
              $("<strong/>", { text: "Profile banner: " })
            ]),
              $("<div/>").append([
              $("<span/>", { text: "This user doesn't have a profile banner." })
            ]),
              $("<div/>").append([
              $("<hr/>")
            ])
          ]);
        }
      })
      .fail(function(jqXHR, textStatus, error) {
        var err = jqXHR.status + ", " + textStatus + ", " + error;
        console.log("Request Failed: " + err);
        // Try again
        create_profile_banner_preview(report_id, channel);
      });
  } else {
    if(via_autoload === false) {
      $("#profile_banner_preview_" + report_id).html("");
    }
  }
}

function create_offline_image_preview(report_id, channel, via_autoload) {
  var via_autoload = typeof via_autoload !== 'undefined' ? via_autoload : false;

  if($("#offline_image_preview_" + report_id).html().length === 0) {
    $("#offline_image_preview_" + report_id).html("Loading ...<hr>");
    $.getJSON("https://api.twitch.tv/kraken/channels/" + channel + "?client_id=" + twitch_kraken_client_id + "&callback=?", function() {})
      .done(function(json) {
        if(typeof json.video_banner !== "undefined" && json.video_banner !== null && json.video_banner.length > 0) {
          // Clear it first
          $("#offline_image_preview_" + report_id).html("");
          // Add the content
          $("#offline_image_preview_" + report_id).append([
            // Upload and get report link button
            $("<span/>", { class: "pull-right", id: "offline_image_upload_" + report_id, html: '<a class="btn btn-xs btn-default btn-link" href="javascript:upload_channel_image(' + "'" + report_id + "', '" + channel + "', 'offline_image', '" + json.video_banner + "'" + ');">Upload and gen report link</a>' } ),
            // Normal Data output
              $("<div/>").append([
              $("<strong/>", { text: "User: " }),
              $("<span/>", { text: json.name })
            ]),
              $("<div/>").append([
              $("<strong/>", { text: "Offline image: " })
            ]),
              $("<div/>").append([
              $("<img/>", { src: json.video_banner, width: "650", height: "366" }),
            ]),
              $("<div/>").append([
              $("<hr/>")
            ])
          ]);
        } else if(typeof json.error !== "undefined" && json.error !== null & typeof json.status !== "undefined" && json.status !== null && json.status == 422 && json.message.indexOf(" is unavailable") != -1) {
          // Clear it first
          $("#offline_image_preview_" + report_id).html("");
          // Add the content
          $("#offline_image_preview_" + report_id).append([
              $("<div/>", { "class": report_id + "_user" }).append([
              $("<strong/>", { text: "User: " }),
              $("<span/>", { text: channel })
            ]),
              $("<div/>").append([
              $("<span/>", { text: "This user is TOS'd." })
            ]),
              $("<div/>").append([
              $("<hr/>")
            ])
          ]);
        } else if(typeof json.error !== "undefined" && json.error !== null & typeof json.status !== "undefined" && json.status !== null && json.status == 422 && json.message.indexOf(" is not available on Twitch") != -1) {
          // Clear it first
          $("#offline_image_preview_" + report_id).html("");
          // Add the content
          $("#offline_image_preview_" + report_id).append([
              $("<div/>").append([
              $("<strong/>", { text: "User: " }),
              $("<span/>", { text: channel })
            ]),
              $("<div/>").append([
              $("<span/>", { text: "This user is a JTV account." })
            ]),
              $("<div/>").append([
              $("<hr/>")
            ])
          ]);
        } else {
          // Clear it first
          $("#offline_image_preview_" + report_id).html("");
          // Add the content
          $("#offline_image_preview_" + report_id).append([
              $("<div/>").append([
              $("<strong/>", { text: "User: " }),
              $("<span/>", { text: channel })
            ]),
              $("<div/>").append([
              $("<strong/>", { text: "Offline image: " })
            ]),
              $("<div/>").append([
              $("<span/>", { text: "This user doesn't have an offline image." })
            ]),
              $("<div/>").append([
              $("<hr/>")
            ])
          ]);
        }
      })
      .fail(function(jqXHR, textStatus, error) {
        var err = jqXHR.status + ", " + textStatus + ", " + error;
        console.log("Request Failed: " + err);
        // Try again
        create_offline_image_preview(report_id, channel);
      });
  } else {
    if(via_autoload === false) {
      $("#offline_image_preview_" + report_id).html("");
    }
  }
}

function create_user_bio_preview(report_id, channel, via_autoload) {
  var via_autoload = typeof via_autoload !== 'undefined' ? via_autoload : false;

  if($("#user_bio_preview_" + report_id).html().length === 0) {
    $("#user_bio_preview_" + report_id).html("Loading ...<hr>");
    $.getJSON("https://api.twitch.tv/kraken/users/" + channel + "?client_id=" + twitch_kraken_client_id + "&callback=?", function() {})
      .done(function(json) {
        if(typeof json.bio !== "undefined" && json.bio !== null && json.bio.length > 0) {
          // Clear it first
          $("#user_bio_preview_" + report_id).html("");
          // Add the content
          $("#user_bio_preview_" + report_id).append([
            // Upload and get report link button
            $("<span/>", { class: "pull-right", id: "user_bio_report_" + report_id, html: '<a class="btn btn-default btn-xs btn-link" href="'+ twitch_base_url + channel + '/report_form?tos_ban=true&description=' + encodeURIComponent(json.bio) + '" target="_blank" rel="noreferrer">Report the bio</a>' } ),
              // Normal Data output
              $("<div/>").append([
              $("<strong/>", { text: "User: " }),
              $("<span/>", { text: json.name })
            ]),
              $("<div/>").append([
              $("<strong/>", { text: "Bio: " })
            ]),
              $("<div/>").append([
              $("<span/>", { text: json.bio }),
            ]),
              $("<div/>").append([
              $("<hr/>")
            ])
          ]);
        } else if(typeof json.error !== "undefined" && json.error !== null & typeof json.status !== "undefined" && json.status !== null && json.status == 422 && json.message.indexOf(" is unavailable") != -1) {
          // Clear it first
          $("#user_bio_preview_" + report_id).html("");
          // Add the content
          $("#user_bio_preview_" + report_id).append([
              $("<div/>", { "class": report_id + "_user" }).append([
              $("<strong/>", { text: "User: " }),
              $("<span/>", { text: channel })
            ]),
              $("<div/>").append([
              $("<span/>", { text: "This user is TOS'd." })
            ]),
              $("<div/>").append([
              $("<hr/>")
            ])
          ]);
        } else if(typeof json.error !== "undefined" && json.error !== null & typeof json.status !== "undefined" && json.status !== null && json.status == 422 && json.message.indexOf(" is not available on Twitch") != -1) {
          // Clear it first
          $("#user_bio_preview_" + report_id).html("");
         // Add the content
          $("#user_bio_preview_" + report_id).append([
              $("<div/>").append([
              $("<strong/>", { text: "User: " }),
              $("<span/>", { text: channel })
            ]),
              $("<div/>").append([
              $("<span/>", { text: "This user is a JTV account." })
            ]),
              $("<div/>").append([
              $("<hr/>")
            ])
          ]);
        } else {
          // Clear it first
          $("#user_bio_preview_" + report_id).html("");
          // Add the content
          $("#user_bio_preview_" + report_id).append([
              $("<div/>").append([
              $("<strong/>", { text: "User: " }),
              $("<span/>", { text: channel })
            ]),
              $("<div/>").append([
              $("<strong/>", { text: "Bio: " })
            ]),
              $("<div/>").append([
              $("<span/>", { text: "This user doesn't have a bio." })
            ]),
              $("<div/>").append([
              $("<hr/>")
            ])
          ]);
        }
      })
      .fail(function(jqXHR, textStatus, error) {
        var err = jqXHR.status + ", " + textStatus + ", " + error;
        console.log("Request Failed: " + err);
        // Try again
        create_user_bio_preview(report_id, channel);
      });
  } else {
    if(via_autoload === false) {
      $("#user_bio_preview_" + report_id).html("");
    }
  }
}

function upload_channel_image(report_id, channel, image_type, image_url) {
  $.ajax({
    url: "/files.json",
    type: "POST",
    data: {
      "file_upload[channel]": channel,
      "url": image_url,
      "authenticity_token": $("meta[name=csrf-token]").attr("content")
    },
    success: function(result, textStatus, jqXHR) {
      var html = '<a class="btn btn-default btn-sm btn-link" href="' + result.report_link + '" target="_blank" rel="noreferrer">Report Link</a>';
      $("#" + image_type + "_upload_" + report_id).html(html);
    },
    error: function(jqXHR, textStatus, errorThrown) {
      // Error
      $("#" + image_type + "_upload_" + report_id).html("There was some kind of error");
    },
  });
  $("#" + image_type + "_upload_" + report_id).html("Uploading ...");
}

function create_stream_preview(report_id, channel) {
  if($("#stream_preview_" + report_id).html().length === 0) {
    $("#stream_preview_" + report_id).html('<div id="video_preview_title_' + report_id + '"></div><br><iframe src="https://player.twitch.tv/?channel=' + channel + '" height="360" width="640" frameborder="0" scrolling="no" allowfullscreen="true"></iframe> <a href="javascript:remove_stream_preview(' + "'" + report_id + "', '" + channel + "'" + ');"><span class="glyphicon glyphicon-remove pull-right"></span></a><hr>');
    get_stream_title(report_id, channel);
  } else {
    $("#stream_preview_" + report_id).html("");
  }
}

function get_stream_title(report_id, channel) {
  $.getJSON("https://api.twitch.tv/kraken/channels/" + channel + "?client_id=" + twitch_kraken_client_id + "&callback=?", function() {})
    .done(function(json) {
      $("#video_preview_title_" + report_id).append([
          $("<div/>", { "class": report_id + "_user" }).append([
          $("<strong/>", { text: "User: " }),
          $("<span/>", { text: json.name })
        ]),
          $("<div/>", { "class": report_id + "_title" }).append([
          $("<strong/>", { text: "Title: " }),
          $("<span/>", { text: json.status }),
          $("<span/>", { html: ' | <a href="https://translate.google.com/#auto/en/' + encodeURIComponent(json.status) + '" target="_blank" rel="noreferrer" onclick="return translate_popup(this.href, ' + "'" + 'stream_title_' + report_id + "'" + ');"><img src="'+ translate_image_url + '" title="Translate" alt=""></a>' })
        ]),
          $("<div/>", { "class": report_id + "_game" }).append([
          $("<strong/>", { text: "Game: " }),
          $("<span/>", { text: json.game })
        ]),
          $("<div/>", { "class": report_id + "_uptime" }).append([
          $("<strong/>", { text: "Stream started: " }),
          $("<span/>", { text: "Update in less than 1 second", "class": "time-friendly" })
        ])
      ]);
    })
    .fail(function(jqXHR, textStatus, error) {
      var err = jqXHR.status + ", " + textStatus + ", " + error;
      console.log("Request Failed: " + err);
      // Try again
      get_stream_title(report_id, channel);
    });
}

function remove_stream_preview(report_id, channel) {
  $("#stream_preview_" + report_id).html("");
}

function randomIntFromInterval(min, max) {
  return Math.floor(Math.random()*(max-min+1)+min);
}

function check_all_reports_description_height() {
  // Check each report description
  $(".report-description").each(function(i, obj) {
    if($("#" + obj.id).data("heightchecked") !== "done") {
      var div_height = $("#" + obj.id).height();
      // Only do stuff if it's too large
      if(div_height > 180) {
        // Limit the height of the report description div
        $("#" + obj.id).css({"overflow": "hidden", "text-overflow": "ellipsis", "max-height": "140px"});

        // Add the Show more button
        $("#" + obj.id).parent(".well").append('<span class="pull-left report-description-show-more"><a href="javascript:show_full_report_description(' + "'" + obj.id + "'" + ');">Show more</a></span>');
      }

      // Mark as checked, but only if it's above 0 height
      if(div_height > 0) {
        $("#" + obj.id).data("heightchecked", "done");
      }
    }
  });
}

function show_full_report_description(ID) {
  // Remove attributes which hide it
  $("#" + ID).css({"overflow": "", "max-height": "", "text-overflow": ""});

  // Remove Show more "button"
  $("#" + ID).closest(".well").children(".report-description-show-more").remove();
}

function show_strike_history_description(obj) {
  // Display the description
  $(obj).closest(".well").children(".strike-history-description").css({"display": "block"});

  // Remove the show more button
  $(obj).closest(".strike-history-description-more").remove();
}

function translate_popup(url, report_id) {
  var popup_window = window.open(url, "Translate_popup_" + report_id, "width=865,height=475,resizable=yes");
  popup_window.focus();
  return false;
}

function get_channel_status_after_stream_request_response(report_id, channel) {
  $.getJSON("https://api.twitch.tv/kraken/channels/" + channel + "?client_id=" + twitch_kraken_client_id + "&callback=?", function() {
    })
    .done(function(json) {
      // Check if there is an existing cache entry which we update
      var delete_other = false;
      for (var i = 0; i < stream_preview_image_request_cache.length; i++) {
        if(typeof stream_preview_image_request_cache[i] !== "undefined" && stream_preview_image_request_cache[i]["channel"] == channel && delete_other === false) {
          stream_preview_image_request_cache[i]["requestPending"] = false;
          stream_preview_image_request_cache[i]["requestTime"] = Math.floor(Date.now() / 1000);
          stream_preview_image_request_cache[i]["json"] = json;
          // console.log("Updated old cache entry for " + channel);
          delete_other = true;
        } else if(typeof stream_preview_image_request_cache[i] !== "undefined" && stream_preview_image_request_cache[i]["channel"] == channel && delete_other === true) {
          // Delete double entry, how ever it even existed in the first place
          stream_preview_image_request_cache[i].delete;
          // console.log("Deleting double cache object for channel: " + channel);
        }
      }

      if(typeof json.error !== "undefined" && json.error !== null & typeof json.status !== "undefined" && json.status !== null && json.status == 422 && json.message.indexOf(" is unavailable") != -1) {
        // Get all reports for this channel
        $('.report[data-target-name="' + channel + '"]')
          .each(function(i, e) {
            var e_reportID = e.id.split("_")[1];
            // Add a space so it doesn't get resolved multiple times
            $("#channel_status_" + e_reportID).html('<span class="label label-danger" style="margin-right: 5px;">TOS&#039;d</span> ');
            // Add a space so it's marked as done if it's empty
            if($("#stream_preview_image_" + e_reportID).html() == "") {
              $("#stream_preview_image_" + e_reportID).html(" ");
            }

            /*
            if(auto_resolve_TOSd_channels === true) {
              resolve_report(e_reportID);
            }
            */
          });
      } else if(typeof json.error !== "undefined" && json.error !== null & typeof json.status !== "undefined" && json.status !== null && json.status == 422 && json.message.indexOf(" is not available on Twitch") != -1) {
        // Get all reports for this channel
        $('.report[data-target-name="' + channel + '"]')
          .each(function(i, e) {
            var e_reportID = e.id.split("_")[1];
            // Add a space so it doesn't get resolved multiple times
            $("#channel_status_" + e_reportID).html('<span class="label label-warning" style="margin-right: 5px;">JTV ACC</span> ');
            // Add a space so it's marked as done if it's empty
            if($("#stream_preview_image_" + e_reportID).html() == "") {
              $("#stream_preview_image_" + e_reportID).html(" ");
            }
          });
      } else {
        // Add a space so it doesn't get resolved multiple times
        $("#stream_preview_image_" + report_id).html(" ");
      }
    })
    .fail(function(jqXHR, textStatus, error) {
      var err = jqXHR.status + ", " + textStatus + ", " + error;
      console.log("Request Failed: " + err);
      // Try again
      // get_channel_status_after_stream_request_response(report_id, channel);
    });

}

function get_stream_preview_image(report_id, channel) {
  // Only do stuff if it's not already set
  if($("#stream_preview_image_" + report_id).html().length == 0) {
    var do_new_request = true;
    var now = Math.floor(Date.now() / 1000);
    // Check if we have a cached entry for it already
    for (var i = 0; i < stream_preview_image_request_cache.length; i++) {
      if(typeof stream_preview_image_request_cache[i] !== "undefined" && stream_preview_image_request_cache[i]["channel"] == channel) {
        if(stream_preview_image_request_cache[i]["requestPending"] === false) {
          if(stream_preview_image_request_cache[i]["requestTime"] <= (now - randomIntFromInterval(30, 40))) { // 30 - 40 seconds to span it out and possible stop lots of requests at once
            // Cache is too old so we delete it and request a new one
            stream_preview_image_request_cache[i].delete;
            do_new_request = true;
            // console.log("Preview cache for " + channel + " was too old (" + (now - stream_preview_image_request_cache[i]["requestTime"]) +  ")");
          } else {
            // console.log("Preview cache for " + channel + " found (" + (now - stream_preview_image_request_cache[i]["requestTime"]) +  " seconds old)");
            // It's not too old so we can use it
            var json = stream_preview_image_request_cache[i]["json"];
            if(typeof json.stream !== "undefined" && json.stream !== null && typeof json.stream.preview.large !== "undefined" && json.stream.preview.large.length > 0) {
              $("#stream_preview_image_" + report_id).html('<img src="' + json.stream.preview.large + '?_=' + now + '" width="320" height="180" data-createdAt="' + moment(json.stream.created_at).unix() + '"><hr>');
              do_new_request = false;
            } else if(typeof json.error !== "undefined" && json.error !== null & typeof json.status !== "undefined" && json.status !== null && json.status == 422) {
              // Thanks to the api we can't know if it's a JTV account or if it's really TOS'd here
              $("#stream_preview_image_" + report_id).html(" ");
              stream_preview_image_request_cache[i]["requestPending"] = true;
              get_channel_status_after_stream_request_response(report_id, channel);
              do_new_request = false;
            } else if(typeof json.error !== "undefined" && json.error !== null & typeof json.status !== "undefined" && json.status !== null && json.status == 404) {
              $("#channel_status_" + report_id).html('<span class="label label-danger" style="margin-right: 5px;">Deleted</span> ');
              do_new_request = false;
            } else if(typeof json.error !== "undefined" && json.error !== null & typeof json.status !== "undefined" && json.status !== null && (json.status == 503 || json.status == 502 || json.status == 500)) {
              // Server error so try again
              stream_preview_image_request_cache[i].delete;
              do_new_request = true;
            } else {
              // Not online or different unknown error
              do_new_request = false;
            }
          }
        } else {
          // console.log("Already a request pending for channel: " + channel);
          // Request pending, check how long it's pending
          if(stream_preview_image_request_cache[i]["requestTime"] < (now - 20)) { // If the request didn't return within 20 seconds then we do it again
            stream_preview_image_request_cache[i].delete;
            do_new_request = true;
            // console.log("The request is pending for too long for channel: " + channel);
          } else {
            do_new_request = false;
          }
        }
        break;
      }
    }

    if(do_new_request === true) {
      var new_index = stream_preview_image_request_cache.length;
      stream_preview_image_request_cache[new_index] = new Object();
      stream_preview_image_request_cache[new_index]["channel"] = channel;
      stream_preview_image_request_cache[new_index]["requestPending"] = true;
      get_stream_preview_image_request(channel);
      // console.log("No preview cache for " + channel + " found");
    }
  }
}

function get_stream_preview_image_request(channel) {
  $.getJSON("https://api.twitch.tv/kraken/streams/" + channel + "?client_id=" + twitch_kraken_client_id + "&callback=?", function() {})
    .done(function(json) {
      var cache_updated = false;
      // Save the json with the current time etc in the cache --
      // Check if there is an existing cache entry which we update
      var delete_other = false;
      for (var i = 0; i < stream_preview_image_request_cache.length; i++) {
        if(typeof stream_preview_image_request_cache[i] !== "undefined" && stream_preview_image_request_cache[i]["channel"] == channel && delete_other === false) {
          stream_preview_image_request_cache[i]["requestPending"] = false;
          stream_preview_image_request_cache[i]["requestTime"] = Math.floor(Date.now() / 1000);
          stream_preview_image_request_cache[i]["json"] = json;
          cache_updated = true;
          // console.log("Updated old cache entry for " + channel);
          delete_other = true;
        } else if(typeof stream_preview_image_request_cache[i] !== "undefined" && stream_preview_image_request_cache[i]["channel"] == channel && delete_other === true) {
          // Delete double entry, how ever it even existed in the first place
          stream_preview_image_request_cache[i].delete;
          // console.log("Deleting double cache object for channel: " + channel);
        }
      }

      // Create a new entry
      if(cache_updated === false) {
        var new_index = stream_preview_image_request_cache.length;
        stream_preview_image_request_cache[new_index] = new Object();
        stream_preview_image_request_cache[new_index]["channel"] = channel;
        stream_preview_image_request_cache[new_index]["requestPending"] = false;
        stream_preview_image_request_cache[new_index]["requestTime"] = Math.floor(Date.now() / 1000);
        stream_preview_image_request_cache[new_index]["json"] = json;
        // console.log("Created new cache entry for " + channel);
      }


      // Call the get_stream_preview_image() for every report against this channel
      $(".stream-preview-image").each(function(i, obj) {
        var reportID = obj.id.split("_")[3];
        var reportChannel = $("#" + obj.id).data("channel");
        if(reportChannel == channel) {
          get_stream_preview_image(reportID, channel);
        }
      });
    })
    .fail(function(jqXHR, textStatus, error) {
      var err = jqXHR.status + ", " + textStatus + ", " + error;
      console.log("Request Failed: " + err);
      // Try again
      // get_stream_preview_image_request(channel);
    });
}

function update_all_stream_preview_images() {
  $(".stream-preview-image").each(function(i, obj) {
    var reportID = obj.id.split("_")[3];
    var channel = $("#" + obj.id).data("channel");
    get_stream_preview_image(reportID, channel);
  });
}


function load_suspension_history(report_id, user_id) {
  if($("#report_history_" + report_id).length > 0 && $("#report_history_" + report_id).html().length === 0) {
    $.ajax({
      url: "/suspensions/history/" + user_id,
      type: "GET",
      cache: true,
      success: function(result){
        $("#report_history_" + report_id).html(result);
      },
      error: function(jqXHR, textStatus, errorThrown){
        $("#report_history_" + report_id).html(errorThrown + '... Try again');
      },
    });
    $("#report_history_" + report_id).html('Loading ...<br><hr>');
  }
}

function load_all_suspension_histories() {
  $(".target_user_suspension_history").each(function(i, obj) {
    var reportID = obj.id.split("_")[2];
    var userID = $("#" + obj.id).closest(".report").data("target-id");
    load_suspension_history(reportID, userID);
  });
}

function load_all_channel_content_auto_previews() {
  $(".violating_content_preview_link").each(function(i){ this.click(); });
}

function get_vod_timestamp_url(report_id, channel_id, time) {
  $.ajax({
    url: "/reports/vod_timestamp?channel_id=" + channel_id + "&time=" + time,
    type: "GET",
    cache: true,
    success: function(result, textStatus, jqXHR){
      $("#vod_link_" + report_id).html(result);
    },
    error: function(jqXHR, textStatus, errorThrown){
      // Error
      $("#vod_link_" + report_id).html("Error getting info from API.");
    },
  });
  $("#vod_link_" + report_id).html("Requesting ...");
}

function get_channel_feed_comment_url(report_id, channel_id, comment_id) {
  $.ajax({
    url: "/reports/channel_feed_comment_link?channel_id=" + channel_id + "&comment_id=" + comment_id,
    type: "GET",
    cache: true,
    success: function(result, textStatus, jqXHR){
      $("#channel_feed_comment_link_" + report_id).html(result);
    },
    error: function(jqXHR, textStatus, errorThrown){
      // Error
      $("#channel_feed_comment_link_" + report_id).html("Error getting info from API.");
    },
  });
  $("#channel_feed_comment_link_" + report_id).html("Requesting ...");
}

function get_vod_comment_url(report_id, vod_id, comment_id) {
  $.ajax({
    url: "/reports/vod_comment_link?vod_id=" + vod_id + "&comment_id=" + comment_id,
    type: "GET",
    cache: true,
    success: function(result, textStatus, jqXHR){
      $("#vod_comment_link_" + report_id).html(result);
    },
    error: function(jqXHR, textStatus, errorThrown){
      // Error
      $("#vod_comment_link_" + report_id).html("Error getting info from API.");
    },
  });
  $("#vod_comment_link_" + report_id).html("Requesting ...");
}

function generate_report_link_content(report) {
  var url_content = "";
  if(report["content"] == "whisper_report") {
    url_content += 'content=whisper&amp;reason=spam&amp;detailed_reason=spam_bots&amp;description=' + encodeURIComponent("Original Report ID: " + report["id"] + "\n");
    var first_line_pos = report["description"].indexOf("\n");
    if(first_line_pos !== -1 && encodeURIComponent(report["description"].substr(first_line_pos).trim()).length <= 4500) {
      url_content += encodeURIComponent(report["description"].substr(first_line_pos).trim()); // To skip the first line
    }
  } else if(report["content"] == "channel_feed_post_report") {
    url_content += 'content=feed&amp;description=' + encodeURIComponent("Original Report ID: " + report["id"] + "\n");
    if(encodeURIComponent(report["description"]).length <= 4500) {
      url_content += encodeURIComponent(report["description"]);
    }
  } else if(report["content"] == "channel_feed_comment_report") {
    url_content += 'content=feed&amp;description=' + encodeURIComponent("Original Report ID: " + report["id"] + "\n");
    if(encodeURIComponent(report["description"]).length <= 4500) {
      url_content += encodeURIComponent(report["description"]);
    }
  } else if(report["content"] == "user_report" && report["from_user_id"] == 31942535 /* CommanderBot */) {
    if(report["description"].indexOf("Hello human,") !== -1) {
      // Automated name violation report
      url_content = 'content=general&amp;reason=offensive_username&amp;detailed_reason=inappropriate_username';
    } else if(report["description"].indexOf("Hello you,") !== -1) {
      // Automated game violation report
      url_content = 'content=live&amp;reason=other';
    }
  } else if(report["content"] == "vod_report") {
    url_content = 'content=vod&amp;vod_id=' + encodeURIComponent(report["content_id"]);
  } else if(report["content"] == "user_report" && report["extra1"] !== null) {
    url_content = 'vod_id=' + encodeURIComponent(report["extra1"].split('?')[0]);
  } else if(report["content"] == "vod_comment_report") {
    url_content += 'description=' + encodeURIComponent("Original Report ID: " + report["id"] + "\n");
    if(encodeURIComponent(report["description"]).length <= 4500) {
      url_content += encodeURIComponent(report["description"]);
    }
  }

  return url_content;
}

function gen_violating_content_preview_links(report) {
  var html = '';
  if(report["content"] == "user_report" && report["from_user_id"] == 31942535 /* CommanderBot */ && report["description"].indexOf('Be advised human,') !== -1) {
    // Auto load channel content if reported from the bot
    if(report["description"].indexOf('rules offending profile image:') !== -1) {
      html += '<a class="violating_content_preview_link" href="javascript:create_profile_image_preview(' + "'" + report['id'] + "', '" + report['target_user_username'] + "'" + ', true);"></a>';
    } else if(report["description"].indexOf('rules offending profile banner:') !== -1) {
      html += '<a class="violating_content_preview_link" href="javascript:create_profile_banner_preview(' + "'" + report['id'] + "', '" + report['target_user_username'] + "'" + ', true);"></a>';
    } else if(report["description"].indexOf('rules offending offline image:') !== -1) {
      html += '<a class="violating_content_preview_link" href="javascript:create_offline_image_preview(' + "'" + report['id'] + "', '" + report['target_user_username'] + "'" + ', true);"></a>';
    } else if(report["description"].indexOf('rules offending profile bio:') !== -1) {
      html += '<a class="violating_content_preview_link" href="javascript:create_user_bio_preview(' + "'" + report['id'] + "', '" + report['target_user_username'] + "'" + ', true);"></a>';
    }
  } else if(report["content"] == "user_report" && report["from_user_id"] == 25681094 /* CommanderRoot */ && report["description"].indexOf('possible porn link in bio') !== -1) {
      html += '<a class="violating_content_preview_link" href="javascript:create_user_bio_preview(' + "'" + report['id'] + "', '" + report['target_user_username'] + "'" + ', true);"></a>';
  }

  // If we haven't matched a bot above yet we look for user reports which have image urls in them
  if(html.length == 0 && report["content"] == "user_report") {
    // User reports which have a profile image url
    if(report["description"].match(/http[s]?:\/\/static\-cdn\.jtvnw\.net\/jtv_user_pictures\/[a-z0-9_]+\-profile_image\-[a-z0-9]+\-[0-9]+x[0-9]+\.[a-z]{3,4}/i) !== null) {
      html += '<a class="violating_content_preview_link" href="javascript:create_profile_image_preview(' + "'" + report['id'] + "', '" + report['target_user_username'] + "'" + ', true);"></a>';
    }

    // User reports which have a profile banner url
    if(report["description"].match(/http[s]?:\/\/static\-cdn\.jtvnw\.net\/jtv_user_pictures\/[a-z0-9_]+\-profile_banner\-[a-z0-9]+\-[0-9]+\.[a-z]{3,4}/i) !== null) {
      html += '<a class="violating_content_preview_link" href="javascript:create_profile_banner_preview(' + "'" + report['id'] + "', '" + report['target_user_username'] + "'" + ', true);"></a>';
    }

    // User reports which have a channel offline image url
    if(report["description"].match(/http[s]?:\/\/static\-cdn\.jtvnw\.net\/jtv_user_pictures\/[a-z0-9_]+\-channel_offline_image\-[a-z0-9]+\-[0-9]+x[0-9]+\.[a-z]{3,4}/i) !== null) {
      html += '<a class="violating_content_preview_link" href="javascript:create_offline_image_preview(' + "'" + report['id'] + "', '" + report['target_user_username'] + "'" + ', true);"></a>';
    }
  }

  return html;
}

function show_clips_timestamp_history(channel, timestamp) {
  bootbox.alert({
    size: 'large',
    message: '<iframe src="clips_overview?mode=timestamp&amp;channel=' + channel + '&amp;timestamp=' + timestamp + '" height="500px" width="850px" frameborder="0"></iframe>'
  });
}

function show_clips_similar_history(channel, slug) {
  bootbox.alert({
    size: 'large',
    message: '<iframe src="clips_overview?mode=similar&amp;channel=' + channel + '&amp;slug=' + encodeURIComponent(slug) + '" height="500px" width="850px" frameborder="0"></iframe>'
  });
}

function check_clip_status(report_id, slug) {
  // Clear the span so we don't do double requests
  $("#clip_status_" + report_id).html('');

  // Make the Ajax request to check the clip status
  $.ajax({
    url: "/clips-api/" + slug,
    type: "HEAD",
    cache: false,
    success: function(result, textStatus, jqXHR) {
      $("#clip_status_" + report_id).html('<span style="color:#000000; background-color:#009900">(OK)</span>');
    },
    error: function(jqXHR, textStatus, errorThrown) {
      if(jqXHR.status == 404) {
        $("#clip_status_" + report_id).html('<span style="color:#000000; background-color:#ff0000">(Disabled)</span>');
      } else {
        $("#clip_status_" + report_id).text("(Unknown error)");
      }
    }
  });
}

function show_clip_preview(report_id, slug) {
  if($("#clip_preview_" + report_id).html().length === 0) {
    $("#clip_preview_" + report_id).html('<b>Clip:</b> ' + htmlEncode(emote_string) + '<br><iframe src="https://clips.twitch.tv/embed?clip=' + encodeURIComponent(slug) + '&_=' + Math.floor(Date.now() / 1000) + '" height="360" width="640" frameborder="0" scrolling="no" allowfullscreen="true"></iframe><hr>');
  } else {
    $("#clip_preview_" + report_id).html('');
  }
}

function show_vod_preview(report_id, vod_id, timestamp) {
  var timestamp = typeof timestamp !== 'undefined' ? timestamp : "";
  var vod_id = vod_id.replace("v", "");

  if($("#vod_preview_" + report_id).html().length === 0) {
    $("#vod_preview_" + report_id).html('<b>VOD:</b> ' + htmlEncode(vod_id) + '<br><iframe src="https://player.twitch.tv/?video=v' + encodeURIComponent(vod_id) + '&time=' + parse_vod_timestamp_to_seconds(timestamp) + '" height="360" width="640" frameborder="0" scrolling="no" allowfullscreen="true"></iframe><hr>');
  } else {
    $("#vod_preview_" + report_id).html('');
  }
}

function scroll_to_next_report(current_report_id) {
  var use_next_ID = false;
  $(".report").each(function(i, obj) {
    var reportID = obj.id.split("_")[1];
    if(use_next_ID === true) {
      // Check if the Resolve button is disable (only scroll to it if it isn't)
      if($("#assign_and_resolve_" + reportID).attr("disabled") === undefined) {
        // console.log("Scrolling to: " + obj.id);
        $("html, body").animate({ scrollTop: ($("#" + obj.id).offset().top - 75) }, 500);
        use_next_ID = false;
        return false;
      }
    }

    if(reportID == current_report_id) {
      use_next_ID = true;
    }
  });
}

function update_overall_unresolved_report_count(count) {
  $("#overall_report_count").text(count);
}

function resolve_other_whisper_reports_against_target(target_user_id, current_report_id) {
  $('.report[data-target-id="' + target_user_id + '"]')
    .filter('[data-content="whisper_report"]')
    .each(function(i, e) {
      var e_report_id = e.id.split("_")[1];
      // Disable resolve other button
      $("#resolve_others_" + e_report_id).attr("disabled", "disabled");

      if(e_report_id != current_report_id) {
        // Assign and resolve if it's not already resolved
        if($("#assign_and_resolve_" + e_report_id).attr("disabled") === undefined) {
          resolve_report(e_report_id, false);
        }
      }
    });
}

function gen_vod_url_with_timestamp(report) {
  // Make sure the content_id is valid
  if(report["content_id"].match(/^v[0-9]+$/) === null) {
    return "";
  }

  var url = twitch_base_url + report["target_user_username"] + "/" + report["content_id"].substr(0, 1) + "/" + report["content_id"].substr(1);
  var timestamp = parse_vod_timestamp_to_seconds(report["extra1"]);
  if(timestamp) {
    url += "?t=" + timestamp;
  }

  return url;
}

function parse_vod_timestamp_to_seconds(timestamp) {
  // Check if the timestamp is valid
  if(timestamp.match(/^[0-9]{2}:[0-9]{2}:[0-9]{2}$/) === null) {
    return "";
  }

  var timestamp_split = timestamp.split(":");
  if(timestamp_split.length == 3) {
    return timestamp_split[0] + "h" + timestamp_split[1] + "m" + timestamp_split[2] + "s";
  }

  return "";
}

function show_all_reports_param() {
  var url_param = "";
  if(window.location.search.match(/show_all/) !== null) {
    url_param = "?show_all=true";
  }

  return url_param;
}

function gen_report_comments_html(comments) {
  var html = '<div class="panel panel-default">';
  html += '<div class="panel-heading">Comments</div>';
  html += '<div class="panel-body">';
  $.each(comments, function(i, val) {
    html += '<div class="well">';
    html += '<b>Created by:</b> ' + htmlEncode(val['created_by_username']) + '<br>';
    html += '<b>Created at:</b> ' + convert_time_to_normal_format(val['created_at']) + '<br>';
    html += '<div class="report-comment">' + htmlEncode(val['comment']) + '</div>';
    html += '</div>';
  });
  html += '</div>';
  html += '</div>';

  return html;
}

function show_report_comment_create_window(report_id) {
  console.log("Showing report comment create window");

  bootbox.dialog({
    title: "Create comment",
    message: '<div class="row">' +
              '<div class="col-md-12">' +
                '<form class="form-horizontal" id="report_comment_create_form" onsubmit="return submit_report_comment_create_form();">' +
                  '<div id="report_comment_create_form_notifications"></div>' +
                  '<div class="form-group">' +
                    '<div class="col-md-12">' +
                      '<textarea rows="4" name="comment" class="form-control input-md"></textarea>' +
                    '</div>' +
                  '</div>' +
                '</form>' +
              '</div>' +
            '</div>',
    buttons: {
      success: {
        label: "Create Comment",
          className: "btn-success report-comment-create-button report-comment-create-button-success",
          callback: function () {
            create_report_comment(report_id, $('#report_comment_create_form').find('textarea[name="comment"]').val());

            // Disable the buttons and change the text to waiting
            $(".report-comment-create-button").attr("disabled", "disabled");
            $(".report-comment-create-button-success").text("Creating ...");

            // Return false so the window doesn't get closed
            return false;
          }
      },
      danger: {
        label: "Cancel",
        className: "btn-danger report-comment-create-button",
        callback: function() {
          return true;
        }
      }
    }
  });
}

function submit_report_comment_create_form() {
  // Click the create button on submission (mostly just so we can catch ENTER)
  $(".report-comment-create-button-success").click();

  return false;
}

function create_report_comment(report_id, comment) {
  $.ajax({
    url: "/report_comments",
    type: "POST",
    timeout: 10000,
    data: {
      "report_comment[report_id]": report_id,
      "report_comment[comment]": comment,
      "authenticity_token": $("meta[name=csrf-token]").attr("content")
    },
    success: function(result, textStatus, jqXHR) {
      // Remove the report comment create window
      $(".report-comment-create-button-success").closest(".modal-content").find(".bootbox-close-button").click();

      if(result.status == 200) {
        // Re-render the comment area with the new data
        $("#report_comments_" + report_id).html(gen_report_comments_html(result.comments));
      } else {
        show_notification(result.message);
      }
    },
    error: function(jqXHR, textStatus, errorThrown) {
      // Remove the report comment create window
      $(".report-comment-create-button-success").closest(".modal-content").find(".bootbox-close-button").click();

      // Error
      show_notification("There was some kind of error creating the comment");
    },
  });

  console.log("Creating report comment request send");
}
