class AutoResolvesController < ApplicationController
  # Let only people with view_reports permission do stuff
  before_action :require_view_reports_permission
  # Always return the current ReportVersion via the header
  after_action :return_report_version_via_header

  def show
    @auto_resolve = AutoResolve.find_by(id: params[:id])
  end

  def create
    auto_resolve = AutoResolve.new(auto_resolve_create_params)
    if auto_resolve.content.nil? || %w(channel_feed_post_report channel_feed_comment_report clip_report user_report vod_comment_report vod_report whisper_report).include?(auto_resolve.content) == false
      # Content is not nil or not valid
      render(json: {status: 400, message: "You need to provide a content"}, :status => 200) and return
    end

    if auto_resolve.from_user_id.blank? && params[:from_user_username] && !params[:from_user_username].blank?
      # Find id by username
      begin
        auto_resolve.from_user_id = ::TwitchUsersService.get_user_by_username(params[:from_user_username])["id"]
      rescue Exception => e
        render(json: { status: 500, message: "Error: #{e.message}"}, :status => 200) and return
      end

      if params[:from_user_username_option] && params[:from_user_username_option] == "is_not"
        auto_resolve.from_user_id = auto_resolve.from_user_id * (-1)
      end
    end

    if auto_resolve.target_user_id.blank? && params[:target_user_username] && !params[:target_user_username].blank?
      # Find id by username
      begin
        auto_resolve.target_user_id = ::TwitchUsersService.get_user_by_username(params[:target_user_username])["id"]
      rescue Exception => e
        render(json: { status: 500, message: "Error: #{e.message}"}, :status => 200) and return
      end

      if params[:target_user_username_option] && params[:target_user_username_option] == "is_not"
        auto_resolve.target_user_id = auto_resolve.target_user_id * (-1)
      end
    end

    if auto_resolve.from_user_id.blank? && auto_resolve.target_user_id.blank?
      # Both are blank, this doesn't work
      render(json: { status: 400, message: "Neither target or reporter provided"}, :status => 200) and return
    end

    if !auto_resolve.from_user_id.blank? && !auto_resolve.target_user_id.blank? && auto_resolve.from_user_id < 0 && auto_resolve.target_user_id < 0
      # Both are negative, not allowed
      render(json: { status: 400, message: "It's not possible to have both target and reporter set to \"IS NOT\""}, :status => 200) and return
    end

    # Check if this autoresolve already exists / is active
    autoresolve_exists = AutoResolve.where(
                                            "resolve_until > ? AND disabled_at IS NULL AND content_id IS NULL",
                                            Time.now
                                          ).where(
                                            from_user_id: auto_resolve.from_user_id,
                                            target_user_id: auto_resolve.target_user_id,
                                            content: auto_resolve.content
                                          )
    if autoresolve_exists.count > 0
      render(json: { status: 422, message: "An autoresolve with these parameters already exists."}, :status => 200) and return
    end

    auto_resolve.created_by = current_user.id

    if params[:duration] && params[:duration].to_i > 0 && params[:duration].to_i <= 1800
      auto_resolve.resolve_until = Time.now().advance(seconds: params[:duration].to_i)

      if auto_resolve.save
        # All went well and autoresolve is created

        # Check all current "new" reports against the new autoresolve
        check_current_reports_against_current_autoresolves()

        # Output info
        render(json: { status: 200, message: "Autoresolve successfully created!", autoresolves: active}, :status => 200) and return
      else
        # Some error
        render(json: { status: 500, message: "You broke it"}, :status => 200) and return
      end
    else
      # No duration or not within limits
      render(json: { status: 400, message: "The submitted duration is not valid"}, :status => 200) and return
    end

  end

  def update
    auto_resolve = AutoResolve.find_by(id: params[:id])
    if auto_resolve
      # Check action type
      if params[:do_action] && params[:do_action] == "disable"
        # Check if it's still active
        if (auto_resolve.resolve_until.nil? || auto_resolve.resolve_until > Time.now) && auto_resolve.disabled_at.nil?
          auto_resolve.disabled_at = Time.now
          if auto_resolve.save
            render(json: { status: 200, message: "Autoresolve successfully disabled!", autoresolves: active}, :status => 200) and return
          else
            render(json: { status: 500, message: "You broke it"}, :status => 200) and return
          end
        else
          render(json: { status: 404, message: "This autoresolve isn't active anyway"}, :status => 200) and return
        end
      else
        render(json: { status: 422, message: "Unknown action"}, :status => 200) and return
      end
    else
      render(json: { status: 404, message: "This autoresolve doesn't exist"}, :status => 200) and return
    end
  end

  def active
    @active_autoresolves = []
    db_active_autoresolves = AutoResolve.where("resolve_until > ? AND disabled_at IS NULL AND content_id IS NULL", Time.now)

    # Get all admin emails for the IDs
    if db_active_autoresolves.length > 0
      user_ids = []
      db_active_autoresolves.each do |a|
        user_ids << a.created_by
      end
      user_ids = user_ids.uniq

      admins = User.where(id: user_ids).pluck(:id, :email)
      admin_emails = {}
      admins.each do |admin|
        admin_emails[admin[0]] = admin[1]
      end
    end

    db_active_autoresolves.each do |a|
      begin
        target_user_username = ::TwitchUsersService.get_user_by_id(a.target_user_id)
        target_user_username = target_user_username["username"] unless target_user_username.nil?
      rescue Exception => e
        target_user_username = "ID: #{a.target_user_id}" # Display ID instead if there was an error
      end

      begin
        from_user_username = ::TwitchUsersService.get_user_by_id(a.from_user_id)
        from_user_username = from_user_username["username"] unless from_user_username.nil?
      rescue Exception => e
        from_user_username = "ID: #{a.from_user_id}" # Display ID instead if there was an error
      end

      @active_autoresolves << {
        id: a.id,
        target_user_id: a.target_user_id,
        target_user_username: target_user_username,
        from_user_id: a.from_user_id,
        from_user_username: from_user_username,
        resolve_until: a.resolve_until,
        content: a.content,
        created_by: a.created_by,
        created_by_username: admin_emails[a.created_by]
      }
      end

    @active_autoresolves
  end

  def search
    conditions = auto_resolve_search_params.delete_if {|k,v| v.blank? }
    if conditions[:auto_resolves]
      conditions[:auto_resolves] = conditions[:auto_resolves].delete_if {|k,v| v.blank? }
    end

    if conditions[:auto_resolves] && conditions[:auto_resolves].any?
      @auto_resolves = AutoResolve.where(conditions).order(id: :desc).paginate(page: params[:page])
    end
  end


  private

    def auto_resolve_create_params
      params.require(:auto_resolve).permit(
        :from_user_id,
        :target_user_id,
        :content
        )
    end

    def auto_resolve_search_params
      params.permit(
        auto_resolves: [
          :id,

          :target_user_id,
          :from_user_id,
          :content,
          :content_id,
          :created_by
        ]
      ).to_h
    end

    def check_current_reports_against_current_autoresolves
      current_reports = Report.where(status: "new")
      current_autoresolves = AutoResolve.where("resolve_until > ? AND disabled_at IS NULL AND content_id IS NULL", Time.now)
      current_holds = ReportHold.where("hold_until > ? AND disabled_at IS NULL", Time.now).pluck(:report_id)

      current_reports.each do |r|
        # Only check reports which are not currently on hold
        unless current_holds.include?(r.id)
          resolve_it = 0
          autoresolve = nil

          current_autoresolves.each do |a|
            current_resolve_it = a.resolves?(r.from_user_id, r.target_user_id, r.content)

            # Directly return if one says - 1
            if current_resolve_it == -1
              resolve_it = -1
              break
            elsif current_resolve_it > resolve_it
              autoresolve = a unless autoresolve
              resolve_it = current_resolve_it
            end
          end

          if resolve_it == 1
            report_before_autoresolve = r.dup
            r.status = "resolved"
            r.assigned_to = autoresolve.created_by
            r.save

            # Audit the autoresolve
            ReportAudit.new(
              report_id: r.id,
              action: "autoresolve",
              action_by: r.assigned_to,
              auto_resolve_id: autoresolve.id,
              diff: report_before_autoresolve.diff(r).to_json
            ).save
          end
        end
      end
    end

    def require_view_reports_permission
      unless current_user && current_user.has_permission_to?(:view_reports)
        render 'shared/_no_permission' and return
      end
    end

    def return_report_version_via_header
      redis_report_version = $redis.get("reports:report_version")
      response.headers["X-Report-Version"] = redis_report_version if redis_report_version
      return true
    end
end
