class EmailsController < ApplicationController
  # Disable login check for creating emails
  skip_before_action :require_login, only: [:create]
  # Disable CSRF token check for creating emails
  skip_before_action :verify_authenticity_token, only: [:create]
  # Let only people with view_emails permission do stuff
  before_action :require_view_emails_permission, except: [:create]
  # Let only people with view_email_audits permission view audits
  before_action :require_view_email_audits_permission, only: [:audit, :audits]

  def index
    if params[:status]
      # If status is given only display emails with the selected status
      @emails = Email.where(status: params[:status]).order(id: :desc).paginate(page: params[:page])
    else
      # Display all emails by default
      @emails = Email.order(id: :desc).paginate(page: params[:page])
    end
  end

  def show
    @db_email = Email.find_by(id: params[:id])

    if @db_email
      # Get body from s3
      s3_mail = s3_obj("emails/#{@db_email.s3_message_id}").get()

      # Parse it
      mail = Mail.read_from_string(s3_mail.body.string)

      @mail_attachments = []
      if mail.multipart?
        @mail_body = mail.text_part.body.decoded

        mail.attachments.each do |attachment|
          if params[:attachment] && params[:attachment] == attachment.filename
            # Just output the attachment as download
            response.headers["Content-Disposition"] = "attachment; filename=\"#{attachment.filename}\""
            render(text: attachment.decoded, content_type: attachment.mime_type, :status => 200) and return
          end

          @mail_attachments << attachment.filename
        end
      else
        @mail_body = mail.body.decoded
      end

      # Try to find the parent email
      unless @db_email.in_reply_to.blank?
        @db_email_parents = Email.where(email_message_id: @db_email.in_reply_to).order(id: :asc)
      end

      # Try to find the "children" emails
      @db_email_children = Email.where(in_reply_to: @db_email.email_message_id).order(id: :asc)

      # To as list for reply
      @to_string = get_receiver_from_headers("To", mail.header.to_a)

      # CC as list for reply
      @cc_string = get_receiver_from_headers("Cc", mail.header.to_a)
    end
  end

  def reply
    db_email = Email.find_by(id: params[:id])

    unless db_email
      flash[:danger] = "This email doesn't exist"
      redirect_to(emails_path) and return
    end

    unless params[:message] && !params[:message].blank?
      flash[:danger] = "You need to add a message"
      redirect_to(email_path(db_email)) and return
    end

    unless params[:to] && !params[:to].blank?
      flash[:danger] = "You need to add an email receiver"
      redirect_to(email_path(db_email)) and return
    end

    # Get body from s3
    s3_mail = s3_obj("emails/#{db_email.s3_message_id}").get()

    # Parse it
    mail = Mail.read_from_string(s3_mail.body.string)

    if mail.multipart?
      mail_body = mail.text_part.body.decoded
    else
      mail_body = mail.body.decoded
    end

    # Change old mail body
    mail_body = mail_body.gsub(/\n/, "\n> ")

    # First from
    unless db_email.from.blank?
      first_from_email = JSON.parse(db_email.from).first
    else
      first_from_email = "you"
    end

    # Add message to the mail body
    mail_body = "#{params[:message]}\r\n\r\nOn #{db_email.created_at.in_time_zone("America/Los_Angeles").strftime("%a, %b %-d, %Y at %-l:%M %p")}, #{first_from_email} wrote:\r\n\r\n> #{mail_body}\r\n"

    if db_email.subject.match(/\ARe: .*\z/)
      # No need to add Re: if the subject already has it
      reply_subject = db_email.subject
    else
      reply_subject = "Re: #{db_email.subject}"
    end

    SesMailer.reply_email(params[:to], params[:cc], reply_subject, mail_body, db_email.email_message_id).deliver_now

    # Audit it
    EmailAudit.new(
      email_id: db_email.id,
      action: "reply",
      action_by: current_user.id,
      content: params[:message]
    ).save

    flash[:success] = "Reply successfully send!"
    redirect_to(emails_path) and return
  end

  def create
    unless params[:authorisation_token]
      render(json: {status: 403, error: true, error_message: "No authorisation_token provided"}, :status => 403) and return
    end

    unless params[:authorisation_token] == ENV['CREATE_EMAIL_TOKEN']
      render(json: {status: 403, error: true, error_message: "Invalid authorisation_token provided"}, :status => 403) and return
    end

    if params[:email].blank?
      render(json: {status: 400, error: true, error_message: "No email content submitted"}, :status => 400) and return
    end

    email = Email.new(email_create_params)
    email.status = "new"
    email.direction = "in"

    # Handle self send emails
    email.direction = "out" if email.from.match(/moderation@moderation\.twitch\.tv/)
    email.status = "resolved" if email.direction == "out"

    # Set replies to partnerconduct emails (matched by their subject) to resolved
    email.status = "resolved" if email.subject.match(/\A(Re: )?(\[Violation\] \[P\] |\[Escalation\] \[P\] |\[FYI\] \[P\] ).*/)

    unless email.save
      # Error message about what went wrong
      render(json: {status: 422, error: true, error_message: "Invalid email content submitted", error_detailed: email.errors.full_messages}, :status => 422) and return
    end

    # Audit it
    EmailAudit.new(
      email_id: email.id,
      action: "create"
    ).save

    # Give the all is well response
    render(json: {status: 200, error: false, error_message: ""}, :status => 200) and return
  end

  def update
    db_email = Email.find_by(id: params[:id])

    unless db_email
      render(json: {status: 404, error: true, error_message: "Email not found"}, :status => 404) and return
    end

    if params[:do_action] && params[:do_action] == "resolve"
      if db_email.status != "resolved"
        db_email.assigned_to = current_user.id
        db_email.status = "resolved"
        if db_email.save

          # Audit it
          EmailAudit.new(
            email_id: db_email.id,
            action: "resolve",
            action_by: current_user.id
          ).save

          render(json: {status: 200, error: false, error_message: ""}, :status => 200) and return
        end
      else
        render(json: {status: 422, error: true, error_message: "The email is already resolved"}, :status => 200) and return
      end
    elsif params[:do_action] && params[:do_action] == "assign_to_staff"
      if db_email.status != "open"
        db_email.assigned_to = nil
        db_email.status = "open"
        if db_email.save

          # Audit it
          EmailAudit.new(
            email_id: db_email.id,
            action: "assign_to_staff",
            action_by: current_user.id
          ).save

          render(json: {status: 200, error: false, error_message: ""}, :status => 200) and return
        end
      else
        render(json: {status: 422, error: true, error_message: "The email is already assigned to staff"}, :status => 200) and return
      end
    end

    render(json: {status: 500, error: true, error_message: "Unknown error"}, :status => 500) and return
  end

  def audits
    @email_audits = EmailAudit.order(id: :desc).paginate(page: params[:page])
  end

  def audit
    @email_audits = EmailAudit.where(email_id: params[:id]).order(id: :asc)
  end

  private

    def email_create_params
      params.require(:email).permit(
        :s3_message_id,
        :subject,
        :from,
        :to,
        :cc,
        :email_message_id,
        :in_reply_to
      )
    end

    def get_receiver_from_headers(type, headers)
      # This is needed because rails is stupid
      headers = headers.to_json
      headers = JSON.parse(headers)

      receiver_string = ""
      addresses = []
      if type == "To"
        # If type is To we get From first
        addresses = get_receiver_from_headers("From", headers)
      end

      headers.each do |header|
        if header["name"] == type
          header["field"]["address_list"]["address_list"]["addresses"].each do |address|
            addresses << "#{address["local"]}@#{address["domain"]}"
          end
        end
      end

      if type == "From"
        # Return the array instead of a string
        return addresses.uniq
      end

      # Generate string from array
      addresses = addresses.uniq
      # Remove our address
      addresses = addresses.delete_if {|x| x == "moderation@moderation.twitch.tv" }
      receiver_string = addresses.join(", ")

      return receiver_string
    end

    def s3_obj(filename)
      s3 = Aws::S3::Resource.new(
        credentials: Aws::Credentials.new(ENV['S3_ACCESS_KEY'], ENV['S3_SECRET_KEY']),
        region: ENV['S3_REGION']
      )
      s3_obj = s3.bucket(ENV['S3_BUCKET']).object(filename)
    end

    def require_view_emails_permission
      unless current_user && current_user.has_permission_to?(:view_emails)
        render 'shared/_no_permission' and return
      end
    end

    def require_view_email_audits_permission
      unless current_user && current_user.has_permission_to?(:view_email_audits)
        render 'shared/_no_permission' and return
      end
    end
end
