class ReportsController < ApplicationController
  # Disable login check for creating reports
  skip_before_action :require_login, only: [:create]
  # Disable CSRF token check for creating reports
  skip_before_action :verify_authenticity_token, only: [:create]
  # Let only people with view_reports permission do stuff
  before_action :require_view_reports_permission, except: [:create]
  # Always return the current ReportVersion via the header
  after_action :return_report_version_via_header, except: [:create]


  def index
    if request.format.symbol.to_s == "json"
      # Return the reports for the user as json object

      # Get the currently active admins
      current_active_admins = active_admins()
      current_active_admin_ids = []
      @distribution = ConsistentHashr.new
      if current_active_admins.any?
        current_active_admins.each do |admin|
          @distribution.add_admin(admin[:admin_id], admin[:admin_weight])
          current_active_admin_ids << admin[:admin_id]
        end
        current_active_admin_ids << current_user.id
        current_active_admin_ids = current_active_admin_ids.uniq
        @distribution.calculate_circles()
      end

      # Get reports from the DB, either new or open AND assigned to one of the active admins
      reports = Report.where("created_at <= ? AND (status = ? OR (status = ? AND assigned_to IN (?)))", Time.now - 2.seconds, "new", "open", current_active_admin_ids)
      reports_for_current_user = []

      # Get the emails for the current user
      emails = Email.where(status: "new").order(id: :asc)
      emails_for_current_user = []

      # Check if any of the reports is currently on hold
      if reports.length > 0
        report_ids = []
        reports.each do |r|
          if report_is_for_current_user(r.target_user_id, r.assigned_to) == true || !params[:show_all].nil?
            report_ids << r.id
          end
        end
        report_ids = report_ids.uniq

        current_holds = {}
        if report_ids.size > 0
          holds = ReportHold.where("hold_until > ? AND disabled_at IS NULL AND report_id IN (?)", Time.now, report_ids).pluck(:report_id, :id)
          holds.each do |hold|
            current_holds[hold[0]] = hold[1]
          end
        end
      end

      # Check for ban exceptions for the current reports
      if reports.length > 0
        target_user_ids = []
        reports.each do |r|
          if report_is_for_current_user(r.target_user_id, r.assigned_to) == true || !params[:show_all].nil?
            target_user_ids << r.target_user_id
          end
        end
        target_user_ids = target_user_ids.uniq

        current_ban_exceptions = {}
        if target_user_ids.size > 0
          ban_exceptions = BanException.where(target_user_id: target_user_ids).pluck(:target_user_id, :reason)
          ban_exceptions.each do |ban_exception|
            current_ban_exceptions[ban_exception[0]] = ban_exception[1]
          end
        end
      end

      # Check target user whisper report count
      if reports.length > 0
        target_user_ids = []
        reports.each do |r|
          if report_is_for_current_user(r.target_user_id, r.assigned_to) == true || !params[:show_all].nil?
            if r.content == "whisper_report"
              target_user_ids << r.target_user_id
            end
          end
        end
        target_user_ids = target_user_ids.uniq

        if target_user_ids.size > 0
          report_counts = Report.where("target_user_id IN (?) AND content = ? AND created_at > ?", target_user_ids, "whisper_report", 24.hours.ago.beginning_of_minute).group(:target_user_id).count
        else
          report_counts = {}
        end
      end

      # Check for past holds on reports
      if reports.length > 0
        report_ids = []
        reports.each do |r|
          if report_is_for_current_user(r.target_user_id, r.assigned_to) == true || !params[:show_all].nil?
            report_ids << r.id
          end
        end
        report_ids = report_ids.uniq

        past_holds = {}
        if report_ids.size > 0
          holds = ReportHold.where(report_id: report_ids)
          holds.each do |hold|
            past_holds[hold.report_id] = [] unless past_holds[hold.report_id]
            past_holds[hold.report_id] << hold
          end
        end
      end

      # Check for report comments
      if reports.length > 0
        report_ids = []
        reports.each do |r|
          if report_is_for_current_user(r.target_user_id, r.assigned_to) == true || !params[:show_all].nil?
            report_ids << r.id
          end
        end
        report_ids = report_ids.uniq

        report_comments = {}
        report_comment_admins = {}
        report_comment_admin_ids = []
        if report_ids.size > 0
          comments = ReportComment.where(report_id: report_ids).order(id: :asc)
          comments.each do |comment|
            report_comments[comment.report_id] = [] unless report_comments[comment.report_id]
            report_comments[comment.report_id] << comment

            report_comment_admin_ids << comment.created_by
          end
          report_comment_admin_ids = report_comment_admin_ids.uniq

          if report_comment_admin_ids.size > 0
            admins = User.where(id: report_comment_admin_ids).pluck(:id, :email)
            admins.each do |admin|
              report_comment_admins[admin[0]] = admin[1]
            end
          end
        end
      end

      reports.each do |r|
        if report_is_for_current_user(r.target_user_id, r.assigned_to) == true || !params[:show_all].nil?
          # Only add it to the array if no active hold for this report_id exists
          if current_holds[r.id].nil?
            report = r.attributes

            # Check if this target is on the ban exception "list"
            unless current_ban_exceptions[r.target_user_id].nil?
              report[:ban_exception] = current_ban_exceptions[r.target_user_id]
            end

            # Check how often the target user was reported in the past 24h (only for whispers)
            if r.content == "whisper_report"
              if report_counts[r.target_user_id].nil?
                report[:target_user_received_whisper_reports] = 0
              else
                report[:target_user_received_whisper_reports] = report_counts[r.target_user_id]
              end
            end

            # Check if this case was ever held
            unless past_holds[r.id].nil?
              report[:holds] = []
              past_holds[r.id].each do |h|
                report[:holds] << {
                  id: h.id,
                  hold_until: h.hold_until,
                  disabled_at: h.disabled_at,
                  created_at: h.created_at
                }
              end
            end

            # Check if there are comments for this report
            unless report_comments[r.id].nil?
              report[:comments] = []
              report_comments[r.id].each do |c|
                report[:comments] << {
                  id: c.id,
                  created_by_username: report_comment_admins[c.created_by],
                  comment: c.comment,
                  created_at: c.created_at
                }
              end
            end

            reports_for_current_user << report
          end
        end
      end

      emails.each do |e|
        if report_is_for_current_user(e.id, e.assigned_to) == true || !params[:show_all].nil?
          # No need to return the full object so just output what we display here
          emails_for_current_user << {
            id: e.id,
            subject: e.subject,
            from: e.from,
            to: e.to,
            cc: e.cc
          }
        end
      end

      # Sort the reports based on calculated priority (from high to low)
      reports_for_current_user.sort! do |x,y|
        x_priority = calculate_report_priority(x)
        y_priority = calculate_report_priority(y)

        # If the priority is not the same then use it to calculate the order
        if x_priority != y_priority
          y_priority <=> x_priority
        else
          # Sort by creation date (by old to new) if the priority is the same
          x["id"] <=> y["id"]
        end
      end

      # "Group" reports against the same channel together (but sort the "Group" from low to high priority (this way the "anchor" high priority doesn't go away))
      reports_for_current_user = group_reports_by_target_id(reports_for_current_user)

      # Limit to 25 + x reports (25 + x of the same target)
      reports_for_current_user = limit_report_amount(reports_for_current_user, 25)

      render(
        json: {
          _total: reports_for_current_user.size,
          _overall_total: reports.size,
          reports: reports_for_current_user,
          emails: emails_for_current_user,
          auto_resolves: active_autoresolves(),
          holds: active_holds(),
          active_admins: active_admins()
        },
        status: 200
      ) and return
    else
      # Return the html base of the report page
    end
  end

  def show
    @report = Report.find_by(id: params[:id])
    @report_audits = ReportAudit.where(report_id: params[:id]).order(id: :asc)
    @report_comments = ReportComment.where(report_id: params[:id]).order(id: :asc)
  end

  def create
    unless params[:authorisation_token] || params[:authorization_token]
      render(json: {status: 403, error: true, error_message: "No authorization_token provided"}, :status => 403) and return
    end

    unless (params[:authorisation_token] && params[:authorisation_token] == ENV['CREATE_REPORT_TOKEN']) || (params[:authorization_token] && params[:authorization_token] == ENV['CREATE_REPORT_TOKEN'])
      render(json: {status: 403, error: true, error_message: "Invalid authorization_token provided"}, :status => 403) and return
    end

    if params[:report].blank?
      render(json: {status: 400, error: true, error_message: "No report content submitted"}, :status => 400) and return
    end

    report = Report.new(report_create_params)
    report.status = "new"

    # Fill out the missing variables (like display name, type, partner status), id > username
    # From user
    if report.from_user_id && report.from_user_id > 0
      begin
          from_user = ::TwitchUsersService.get_user_by_id(report.from_user_id)
        rescue Exception => e
          logger.debug "TwitchUsersService Exception (Thread: #{Thread.current.object_id}): #{e.message}"
          retry_report(params) if Rails.env.production? && report.content != "whisper_report"
          render(json: {status: 500, error: true, error_message: "We broke something over here"}, :status => 500) and return
      end

      report.from_user_username = from_user["username"]
      report.from_user_type = from_user["type"]
      report.from_user_display_name = from_user["display_name"]
    elsif report.from_user_username && !report.from_user_username.blank?
      begin
          from_user = ::TwitchUsersService.get_user_by_username(report.from_user_username)
        rescue Exception => e
          logger.debug "TwitchUsersService Exception (Thread: #{Thread.current.object_id}): #{e.message}"
          retry_report(params) if Rails.env.production? && report.content != "whisper_report"
          render(json: {status: 500, error: true, error_message: "We broke something over here"}, :status => 500) and return
      end

      report.from_user_id = from_user["id"]
      report.from_user_type = from_user["type"]
      report.from_user_display_name = from_user["display_name"]
    else
      # Reporter is empty
    end

    # Check rate limit based on reporter info
    if !report.from_user_id.nil? && report.from_user_type == "user"
      if $redis.get("reports:create_rate_limit:#{report.from_user_id}")
        # Return 429 if rate limit exists
        render(json: {status: 429, error: true, error_message: "You are reporting too fast."}, :status => 429) and return
      else
        # Set the key with a TTL of 2 seconds
        $redis.setex("reports:create_rate_limit:#{report.from_user_id}", 2, "true")
      end
    end

    # Get the partner info for the reporter (if type is filled out we know the user exists)
    if report.from_user_type && !report.from_user_type.blank?
      begin
          report.from_user_partner = ::TwitchPartnershipsService.get_partner_by_id(report.from_user_id)
        rescue Exception => e
          logger.debug "TwitchPartnershipsService Exception (Thread: #{Thread.current.object_id}): #{e.message}"
          retry_report(params) if Rails.env.production? && report.content != "whisper_report"
          render(json: {status: 500, error: true, error_message: "We broke something over here"}, :status => 500) and return
      end
    end

    # Target user
    if report.target_user_id && report.target_user_id > 0
      begin
          target_user = ::TwitchUsersService.get_user_by_id(report.target_user_id)
        rescue Exception => e
          logger.debug "TwitchUsersService Exception (Thread: #{Thread.current.object_id}): #{e.message}"
          retry_report(params) if Rails.env.production? && report.content != "whisper_report"
          render(json: {status: 500, error: true, error_message: "We broke something over here"}, :status => 500) and return
      end

      report.target_user_username = target_user["username"]
      report.target_user_type = target_user["type"]
      report.target_user_display_name = target_user["display_name"]
    elsif report.target_user_username && !report.target_user_username.blank?
      begin
          target_user = ::TwitchUsersService.get_user_by_username(report.target_user_username)
        rescue Exception => e
          logger.debug "TwitchUsersService Exception (Thread: #{Thread.current.object_id}): #{e.message}"
          retry_report(params) if Rails.env.production? && report.content != "whisper_report"
          render(json: {status: 500, error: true, error_message: "We broke something over here"}, :status => 500) and return
      end

      report.target_user_id = target_user["id"]
      report.target_user_type = target_user["type"]
      report.target_user_display_name = target_user["display_name"]
    else
      # Target is empty
      render(json: {status: 422, error: true, error_message: "Invalid report content submitted. Missing target user.", error_detailed: report.errors.full_messages}, :status => 422) and return
    end

    # Get the partner info for the target (if type is filled out we know the user exists)
    if report.target_user_type && !report.target_user_type.blank?
      begin
          report.target_user_partner = ::TwitchPartnershipsService.get_partner_by_id(report.target_user_id)
        rescue Exception => e
          logger.debug "TwitchPartnershipsService Exception (Thread: #{Thread.current.object_id}): #{e.message}"
          retry_report(params) if Rails.env.production? && report.content != "whisper_report"
          render(json: {status: 500, error: true, error_message: "We broke something over here"}, :status => 500) and return
      end
    end

    # Get the currently recording VOD if it's a user_report
    if report.content == "user_report"
      begin
          currently_recorded_vod = ::TwitchVinylService.get_currently_recording_vod(report.target_user_id)
          report.extra1 = currently_recorded_vod unless currently_recorded_vod.nil?
        rescue Exception => e
          logger.debug "TwitchVinylService Exception (Thread: #{Thread.current.object_id}): #{e.message}"
      end
    end

    # Make sure all new lines in the description are only \n and not \r\n or \r (just to have consistency)
    report.description = report.description.gsub(/\r\n/, "\n")
    report.description = report.description.gsub(/\r/, "\n")

    # Set the reason to other if it's currently invalid
    report.valid?
    report.reason = "other" unless (report.errors()[:reason]).empty?

    unless report.save
      # Error message about what went wrong
      render(json: {status: 422, error: true, error_message: "Invalid report content submitted", error_detailed: report.errors.full_messages}, :status => 422) and return
    end

    # Audit the new report
    ReportAudit.new(report_id: report.id, action: "create").save

    # Send report to spade
    send_report_to_spade(report) if Rails.env.production?


    do_autoresolve_check = true

    # Check if the content + content_id was already checked
    if report.from_user_type == "user" && !report.content_id.nil? && ["channel_feed_post_report", "channel_feed_comment_report", "clip_report", "vod_comment_report"].include?(report.content)
      content_id_already_checked = AutoResolve.where(disabled_at: nil, content: report.content, content_id: report.content_id).first
      if !content_id_already_checked.nil?
        report_before_autoresolve = report.dup
        report.status = "resolved"
        report.assigned_to = content_id_already_checked.created_by
        report.save

        # Audit the autoresolve
        ReportAudit.new(
          report_id: report.id,
          action: "autoresolve",
          action_by: report.assigned_to,
          auto_resolve_id: content_id_already_checked.id,
          diff: report_before_autoresolve.diff(report).to_json
        ).save
        do_autoresolve_check = false
      end
    end

    # Check new report against current active autoresolves (only autoresolve if the reporter isn't a global mod or higher)
    if report.from_user_type == "user" && do_autoresolve_check == true
      autoresolve_match = report_matches_active_autoresolves(report)
      if !autoresolve_match.nil?
        report_before_autoresolve = report.dup
        report.status = "resolved"
        report.assigned_to = autoresolve_match.created_by
        report.save

        # Audit the autoresolve
        ReportAudit.new(
          report_id: report.id,
          action: "autoresolve",
          action_by: report.assigned_to,
          auto_resolve_id: autoresolve_match.id,
          diff: report_before_autoresolve.diff(report).to_json
        ).save
      end
    end

    # Give the all is well response
    render(json: {status: 200, error: false, error_message: ""}, :status => 200) and return
  end

  def update
    report = Report.find_by(id: params[:id])

    unless report
      render(json: {status: 404, error: true, error_message: "Report not found"}, :status => 404) and return
    end

    report_before_change = report.dup

    if params[:do_action] && params[:do_action] == "resolve"
      report.assigned_to = current_user.id
      report.status = "resolved"
      if report.save
        # Audit it
        ReportAudit.new(
          report_id: report.id,
          action: "resolve",
          action_by: current_user.id,
          diff: report_before_change.diff(report).to_json
        ).save

        # Add to autoresolve if the content_id is not nil and it's not already on the list
        if !report.content_id.nil? && ["channel_feed_post_report", "channel_feed_comment_report", "clip_report", "vod_comment_report"].include?(report.content)
          autoresolve_exists = AutoResolve.where(disabled_at: nil, content: report.content, content_id: report.content_id)
          if autoresolve_exists.count == 0
            AutoResolve.new(created_by: current_user.id, content: report.content, content_id: report.content_id).save
          end
        end

        render(json: {status: 200, error: false, error_message: ""}, :status => 200) and return
      end
    elsif params[:do_action] && params[:do_action] == "pm_check"
      report.assigned_to = -1
      report.status = "open"
      if report.save
        # Audit it
        ReportAudit.new(
          report_id: report.id,
          action: "pm_check",
          action_by: current_user.id,
          diff: report_before_change.diff(report).to_json
        ).save

        render(json: {status: 200, error: false, error_message: ""}, :status => 200) and return
      end
    elsif params[:do_action] && params[:do_action] == "staff_escalation"
      report.assigned_to = -2
      report.status = "open"
      if report.save
        # Audit it
        ReportAudit.new(
          report_id: report.id,
          action: "staff_escalation",
          action_by: current_user.id,
          diff: report_before_change.diff(report).to_json
        ).save

        render(json: {status: 200, error: false, error_message: ""}, :status => 200) and return
      end
    elsif params[:do_action] && params[:do_action] == "check_later"
      report.assigned_to = -3
      report.status = "open"
      if report.save
        # Audit it
        ReportAudit.new(
          report_id: report.id,
          action: "check_later",
          action_by: current_user.id,
          diff: report_before_change.diff(report).to_json
        ).save

        render(json: {status: 200, error: false, error_message: ""}, :status => 200) and return
      end
    elsif params[:do_action] && params[:do_action] == "assign_to_admin"
      # Check if the parameter for the target exists
      unless params[:assign_to] && !params[:assign_to].blank?
        render(json: {status: 422, error: true, error_message: "No target admin selected"}, :status => 200) and return
      end

      # Check if the admin exists
      user = User.find_by(id: params[:assign_to])
      unless user
        render(json: {status: 422, error: true, error_message: "The selected admin doesn't exist"}, :status => 200) and return
      end

      report.assigned_to = params[:assign_to]
      report.status = "open"
      if report.save
        # Audit it
        ReportAudit.new(
          report_id: report.id,
          action: "assign_to_admin",
          action_by: current_user.id,
          diff: report_before_change.diff(report).to_json
        ).save

        render(json: {status: 200, error: false, error_message: ""}, :status => 200) and return
      end
    end
    render(json: {status: 500, error: true, error_message: "Unknown error"}, :status => 500) and return
  end

  def me
    if params[:status]
      # If status is given only display reports with the selected status
      @reports = Report.where(assigned_to: current_user.id, status: params[:status]).order(id: :desc).paginate(page: params[:page])
    else
      # Display all reports for the current user by default
      @reports = Report.where(assigned_to: current_user.id).order(id: :desc).paginate(page: params[:page])
    end
  end

  def pm_check
    @reports = Report.where(assigned_to: -1).order(id: :asc).paginate(page: params[:page])
  end

  def escalations
    @reports = Report.where(assigned_to: -2).order(id: :asc).paginate(page: params[:page])
  end

  def check_later
    @reports = Report.where(assigned_to: -3).order(id: :asc).paginate(page: params[:page])
  end

  def search
    conditions = report_search_params.delete_if {|k,v| v.blank? }
    if conditions[:reports]
      conditions[:reports] = conditions[:reports].delete_if {|k,v| v.blank? }
    end

    if conditions[:reports] && conditions[:reports].any?
      if !conditions[:reports][:description].blank?
        searched_description = conditions[:reports][:description]

        # Remove description from the normal conditions because we want to do a LIKE search instead
        conditions[:reports].delete(:description)

        if conditions[:reports].any?
          @reports = Report.where(conditions).where("description LIKE ?", "%#{searched_description}%").order(id: :desc).paginate(page: params[:page])
        else
          # Description was the only parameter
          @reports = Report.where("description LIKE ?", "%#{searched_description}%").order(id: :desc).paginate(page: params[:page])
        end
      else
        @reports = Report.where(conditions).order(id: :desc).paginate(page: params[:page])
      end
    end
  end

  def all
    @reports = Report.order(id: :desc).paginate(page: params[:page])
  end

  def vod_timestamp
    # Check if all parts are here
    unless params[:channel_id]
      render html: "No channel id specified" and return
    end

    unless params[:time]
      render html: "No time specified" and return
    end

    # Get the channel login for the channel id (needed for the URL later)
    begin
        channel = ::TwitchUsersService.get_user_by_id(params[:channel_id])
      rescue Exception => e
        render html: "Error while getting user information" and return
    end

    unless channel
      render html: "This user doesn't seem to exist" and return
    end

    api_offset = 0
    begin
        # Do the api request
        url = URI.parse("http://vinyl-internal.production.us-west2.twitch.tv/v1/vods/user/#{URI::escape(params[:channel_id])}?broadcast_type=archive&limit=100&offset=#{api_offset}")
        http = Net::HTTP.new(url.host, url.port)
        http.use_ssl = true if url.scheme == "https"
        http.open_timeout = 3
        http.read_timeout = 10
        req = Net::HTTP::Get.new(url.request_uri)
        res = http.request(req)
        unless res.is_a?(Net::HTTPSuccess)
          render html: "Error while getting VOD information" and return
        end
      rescue Exception => e
        render html: "Error while getting VOD information" and return
    end

    @vod_location = nil
    # Check the returned array the find the first VOD which was recorded before our timestamp
    vinyl_response = JSON.parse(res.body)
    if vinyl_response["vods"] && vinyl_response["vods"].size > 0
      vinyl_response["vods"].each do |vod|
        if @vod_location == nil && vod["recorded_on"].to_time.to_i <= params[:time].to_i
          # Check if the vod is longer than the diff between recorded on and timestamp
          vod_time = params[:time].to_i - vod["recorded_on"].to_time.to_i
          if vod_time.to_i < (vod["total_length"].to_i + 120) # Allow a 120 sec window to record the current stream
            @vod_location = "#{Settings.twitch_base_url}#{URI::escape(channel["username"])}/v/#{URI::escape(vod["id"].to_s)}?t=#{vod_time.to_s}s"
          else
            render html: "No VOD found for this user at this time. :(" and return
          end
        end
      end
    else
      render html: "No VOD found for this user at this time." and return
    end

    unless @vod_location
      render html: "No VOD found for this user at this time." and return
    end
    render layout: false
  end

  def channel_feed_comment_link
    # Check if all parts are here
    unless params[:channel_id]
      render html: "No channel_id specified" and return
    end

    unless params[:comment_id]
      render html: "No comment_id specified" and return
    end

    # Get the channel login for the channel id (needed for the URL later)
    begin
        channel = ::TwitchUsersService.get_user_by_id(params[:channel_id])
      rescue Exception => e
        render html: "Error while getting user information" and return
    end

    unless channel
      render html: "This user doesn't seem to exist" and return
    end

    # Get comment info from the service
    begin
        url = URI.parse("http://audrey.prod.us-west2.justin.tv/v1/feed/#{params[:channel_id]}/comments/#{params[:comment_id]}")
        http = Net::HTTP.new(url.host, url.port)
        http.use_ssl = true if url.scheme == "https"
        http.open_timeout = 3
        http.read_timeout = 10
        req = Net::HTTP::Get.new(url.request_uri)
        res = http.request(req)

        unless res.is_a?(Net::HTTPSuccess)
          if res.is_a?(Net::HTTPNotFound)
            render html: "Comment not found" and return
          else
            render html: "Error getting info from the backend" and return
          end
        end
      rescue Exception => e
        render html: "Error getting info from the backend" and return
    end

    # Parse result
    json_result = JSON.parse(res.body)

    unless json_result["post_id"]
      render html: "Error while parsing comment info" and return
    end

    @feed_comment_link = "#{Settings.twitch_base_url}#{URI::escape(channel["username"])}/p/#{URI::escape(json_result["post_id"])}"

    render layout: false
  end

  def vod_comment_link
    # Check if all parts are here
    unless params[:vod_id]
      render html: "No vod_id specified" and return
    end

    vod_id = params[:vod_id].gsub(/v/, "")

    unless params[:comment_id]
      render html: "No comment_id specified" and return
    end

    # Get vod owner info from the service
    begin
        url = URI.parse("http://vinyl-internal.production.us-west2.twitch.tv/v1/vods?ids=#{vod_id}")
        http = Net::HTTP.new(url.host, url.port)
        http.use_ssl = true if url.scheme == "https"
        http.open_timeout = 3
        http.read_timeout = 10
        req = Net::HTTP::Get.new(url.request_uri)
        res = http.request(req)

        unless res.is_a?(Net::HTTPSuccess)
          render html: "Error getting info from the backend" and return
        end
      rescue Exception => e
        render html: "Error getting info from the backend" and return
    end

    # Parse result
    json_result = JSON.parse(res.body)

    unless json_result["vods"] && json_result["vods"][0]
      render html: "Could not find VOD" and return
    end

    @vod_comment_link = "#{Settings.twitch_base_url}#{URI::escape((json_result["vods"][0]["uri"]).split('/')[0])}/v/#{json_result["vods"][0]["id"]}"

    render layout: false
  end

  def partnerconduct_report
    if request.post?
      # Make sure all necessary parts are here
      unless params[:reported_user] && !params[:reported_user].blank?
        @error_message = "The reported user can't be black"
        render layout: false and return
      end

      unless params[:email_type] && !params[:email_type].blank? && %w(violation escalation fyi).include?(params[:email_type])
        @error_message = "Invalid email type submitted"
        render layout: false and return
      end

      # If it's a violation or escalation check if it has all the required parts
      if %w(violation escalation).include?(params[:email_type])
        unless params[:reason] && !params[:reason].blank?
          @error_message = "No reason submitted"
          render layout: false and return
        end

        unless params[:detailed_reason_title] && !params[:detailed_reason_title].blank?
          @error_message = "You need to provided a detailed_reason_title"
          render layout: false and return
        end
      else
        # It's an FYI so check for subject
        unless params[:subject] && !params[:subject].blank?
          @error_message = "You need to provided a subject"
          render layout: false and return
        end
      end

      # Always needs a description
      unless params[:description] && !params[:description].blank?
        @error_message = "You need to provided a description"
        render layout: false and return
      end

      subject = ""
      case params[:email_type]
        when "violation"
          subject = "[Violation] [P] #{params[:reported_user].downcase} => #{params[:detailed_reason_title]}"
        when "escalation"
          subject = "[Escalation] [P] #{params[:reported_user].downcase} => #{params[:detailed_reason_title]}"
        when "fyi"
          subject = "[FYI] [P] #{params[:reported_user].downcase} => #{params[:subject]}"
      end

      body = ""
      case params[:email_type]
        when "violation"
          body = "Reporting admin: #{current_user.email}\r\n"
          body += "Channel URL: #{Settings.twitch_base_url}#{params[:reported_user].downcase}\r\n"
          body += "Reason: #{params[:reason]}\r\n"
          body += params[:description]
        when "escalation"
          body = "Reporting admin: #{current_user.email}\r\n"
          body += "Channel URL: #{Settings.twitch_base_url}#{params[:reported_user].downcase}\r\n"
          body += "Reason: #{params[:reason]}\r\n"
          body += params[:description]
        when "fyi"
          body = "Reporting admin: #{current_user.email}\r\n"
          body += "Channel URL: #{Settings.twitch_base_url}#{params[:reported_user].downcase}\r\n"
          body += "----------------------------------------\r\n\r\n"
          body += params[:description]
      end

      SesMailer.partnerconduct(subject, body).deliver_now

      render plain: "Thank you for your report. Now have fun autoresolving stuff." and return
    end

    render layout: false and return
  end


  private

    def report_search_params
      params.permit(
        reports: [
          :id,

          :status,
          :assigned_to,

          :from_user_username,
          :from_user_display_name,
          :from_user_id,
          :from_user_type,
          :from_user_partner,

          :target_user_username,
          :target_user_display_name,
          :target_user_id,
          :target_user_type,
          :target_user_partner,

          :content,
          :content_id,
          :reason,
          :description,
          :extra1,

          :created_at
        ]
      ).to_h
    end

    def report_create_params
      params.require(:report).permit(
        :from_user_username,
        :from_user_id,

        :target_user_username,
        :target_user_id,

        :content,
        :content_id,
        :reason,
        :description,
        :extra1,

        :origin
      )
    end

    def active_autoresolves
      active_autoresolves = []
      db_active_autoresolves = AutoResolve.where("resolve_until > ? AND disabled_at IS NULL AND content_id IS NULL", Time.now)

      # Get all admin emails for the IDs
      if db_active_autoresolves.length > 0
        user_ids = []
        db_active_autoresolves.each do |a|
          user_ids << a.created_by
        end
        user_ids = user_ids.uniq

        admins = User.where(id: user_ids).pluck(:id, :email)
        admin_emails = {}
        admins.each do |admin|
          admin_emails[admin[0]] = admin[1]
        end
      end

      db_active_autoresolves.each do |a|
        begin
          target_user_username = ::TwitchUsersService.get_user_by_id(a.target_user_id)
          target_user_username = target_user_username["username"] unless target_user_username.nil?
        rescue Exception => e
          target_user_username = "ID: #{a.target_user_id}" # Display ID instead if there was an error
        end

        begin
          from_user_username = ::TwitchUsersService.get_user_by_id(a.from_user_id)
          from_user_username = from_user_username["username"] unless from_user_username.nil?
        rescue Exception => e
          from_user_username = "ID: #{a.from_user_id}" # Display ID instead if there was an error
        end

        active_autoresolves << {
          id: a.id,
          target_user_id: a.target_user_id,
          target_user_username: target_user_username,
          from_user_id: a.from_user_id,
          from_user_username: from_user_username,
          resolve_until: a.resolve_until,
          content: a.content,
          created_by: a.created_by,
          created_by_username: admin_emails[a.created_by]
        }
      end

      return active_autoresolves
    end

    def active_holds
      active_holds = []
      db_active_holds = ReportHold.where("hold_until > ? AND disabled_at IS NULL", Time.now)

      # Get all admin emails for the IDs
      if db_active_holds.length > 0
        user_ids = []
        db_active_holds.each do |h|
          user_ids << h.created_by
        end
        user_ids = user_ids.uniq

        admins = User.where(id: user_ids).pluck(:id, :email)
        admin_emails = {}
        admins.each do |admin|
          admin_emails[admin[0]] = admin[1]
        end
      end

      # Get report info for the hold reports
      if db_active_holds.length > 0
        report_ids = []
        db_active_holds.each do |h|
          report_ids << h.report_id
        end
        report_ids = report_ids.uniq

        reports = Report.where(id: report_ids).pluck(:id, :target_user_username, :content)
        reports_info = {}
        reports.each do |report|
          reports_info[report[0]] = { target_user_username: report[1], content: report[2] }
        end
      end

      db_active_holds.each do |h|
        active_holds << {
          id: h.id,
          report_id: h.report_id,
          target_user_username: reports_info[h.report_id][:target_user_username],
          hold_until: h.hold_until,
          content: reports_info[h.report_id][:content],
          created_by: h.created_by,
          created_by_username: admin_emails[h.created_by]
        }
      end

      return active_holds
    end

    def report_matches_active_autoresolves(report)
      resolve_it = 0
      autoresolve = nil
      current_autoresolves = AutoResolve.where("resolve_until > ? AND disabled_at IS NULL AND content_id IS NULL", Time.now)

      current_autoresolves.each do |a|
        current_resolve_it = a.resolves?(report.from_user_id, report.target_user_id, report.content)

        # Directly return nil if one says -1
        if current_resolve_it == -1
          return nil
        elsif current_resolve_it > resolve_it
          autoresolve = a unless autoresolve
          resolve_it = current_resolve_it
        end
      end

      if resolve_it == 1
        return autoresolve
      end

      return nil
    end

    def report_is_for_current_user(target_user_id, assigned_to)
      unless assigned_to.nil?
        if assigned_to == current_user.id
          return true
        else
          return false
        end
      end

      unless current_user_is_in_active_admins
        return false
      end

      @mapped_admin_cache = {} unless @mapped_admin_cache
      return @mapped_admin_cache[target_user_id] unless @mapped_admin_cache[target_user_id].nil?

      mapped_admin = @distribution.map(target_user_id)

      if mapped_admin == current_user.id
        @mapped_admin_cache[target_user_id] = true
        return true
      else
        @mapped_admin_cache[target_user_id] = false
        return false
      end
    end

    def current_user_is_in_active_admins
      @active_admin_cache = {} unless @active_admin_cache
      return @active_admin_cache[current_user.id] unless @active_admin_cache[current_user.id].nil?

      active_admin = ReportAdmin.find_by(disabled_at: nil, admin_id: current_user.id)
      if active_admin
        @active_admin_cache[current_user.id] = true
        return true
      else
        @active_admin_cache[current_user.id] = false
        return false
      end
    end

    def active_admins
      @active_admin_cache = {} unless @active_admin_cache
      active_admins_array = []
      db_active_admins = ReportAdmin.where(disabled_at: nil)

      # Get all admin emails for the IDs
      if db_active_admins.length > 0
        user_ids = []
        db_active_admins.each do |admin|
          user_ids << admin.admin_id
        end
        user_ids = user_ids.uniq

        admins = User.where(id: user_ids).pluck(:id, :email)
        admin_emails = {}
        admins.each do |admin|
          admin_emails[admin[0]] = admin[1]
        end
      end

      db_active_admins.each do |admin|
        active_admins_array << {
          id: admin.id,
          admin_id: admin.admin_id,
          admin_username: admin_emails[admin.admin_id],
          admin_weight: admin.admin_weight
        }

        # Add the user to our cache
        @active_admin_cache[admin.admin_id] = true
      end

      return active_admins_array
    end

    def calculate_report_priority(report)
      # Priority is based on
      # content (user reports are higher as whispers),
      # reason (porn is higher as non-gaming) and
      # age (this way non-gaming for example are not un-worked for hours if always new high priority come in)
      # from_user_type so staff, admin and global_mod reports have higher priority because they are very often correct
      priority = (priority_based_on_content(report["content"]) *
                 priority_based_on_reason(report["reason"]) *
                 priority_based_on_age(report["created_at"]) *
                 priority_based_on_from_user_type(report["from_user_type"]))

      return priority
    end

    def priority_based_on_content(content)
      case content
        when "user_report"
          return 1.0
        when "whisper_report"
          return 0.5
        else
          # Default
          return 1.0
      end
    end

    def priority_based_on_reason(reason)
      case reason
        when "porn"
          return 5.0
        when "selfharm"
          return 4.0
        when "harm"
          return 4.0
        when "nongaming"
          return 2.0
        when "spam"
          return 1.0
        else
          # Default
          return 3.0
      end
    end

    def priority_based_on_from_user_type(from_user_type)
      case from_user_type
        when "staff"
          return 3.0
        when "admin"
          return 3.0
        when "global_mod"
          return 3.0
        else
          # Default
          return 1.0
      end
    end

    def priority_based_on_age(created_at)
      time_diff = (Time.now - created_at).to_i

      # Each half hour should double it
      priority = 1.0
      return priority + (time_diff.to_f / 1800.0)
    end

    def group_reports_by_target_id(report_array)
      grouped_report_array = []

      index = 0

      while true
        unless report_array[index]
          # Exit out if we are at the end of the array
          break
        end
        current_target_user_id = report_array[index]["target_user_id"]
        grouped_report_array << report_array[index]

        # Try to find reports against the same target
        index2 = index + 1
        add_pos = 0
        while true
          unless report_array[index2]
            # Exit out if we are at the end of the array
            break
          end

          current2_target_user_id = report_array[index2]["target_user_id"]
          if current_target_user_id == current2_target_user_id
            grouped_report_array.insert(-2 + add_pos, report_array[index2])

            # Delete it so we don't try to add it again
            report_array.delete_at(index2)

            # Move the add pos one down so it's added just before the last one
            add_pos = add_pos - 1
          else
            # Search at the next position
            index2 = index2 + 1
          end
        end

        index = index + 1
      end

      return grouped_report_array
    end

    def limit_report_amount(report_array, limit)
      if report_array.size <= limit
        return report_array
      end

      slice_pos = limit - 1
      end_pos_target_user_id = report_array[slice_pos]["target_user_id"]
      for i in limit..(report_array.size - 1)
        current_pos_target_user_id = report_array[i]["target_user_id"]
        if current_pos_target_user_id == end_pos_target_user_id
          slice_pos = i
        else
          # Exit out as this isn't the same user anymore
          break
        end
      end

      return report_array.slice(0..slice_pos)
    end

    def send_report_to_spade(report)
      # Generate the right array structure
      event = {}
      event[:event] = 'leviathan_report'
      event[:properties] = {}
      event[:properties][:report_id] = report.id

      event[:properties][:from_user_login] = report.from_user_username
      event[:properties][:from_user_display_name] = report.from_user_display_name
      event[:properties][:from_user_id] = report.from_user_id
      event[:properties][:from_user_type] = report.from_user_type
      event[:properties][:from_user_partner] = report.from_user_partner

      event[:properties][:target_user_login] = report.target_user_username
      event[:properties][:target_user_display_name] = report.target_user_display_name
      event[:properties][:target_user_id] = report.target_user_id
      event[:properties][:target_user_type] = report.target_user_type
      event[:properties][:target_user_partner] = report.target_user_partner

      event[:properties][:content] = report.content
      event[:properties][:content_id] = report.content_id
      if report.content == "vod_report"
        event[:properties][:vod_timestamp] = report.extra1
      elsif report.content == "channel_feed_comment_report"
        event[:properties][:channel_feed_owner_id] = report.extra1.to_i if report.extra1
      elsif report.content == "vod_comment_report"
        event[:properties][:vod_id] = report.extra1.gsub(/v/, "").to_i if report.extra1
      end

      event[:properties][:reason] = report.reason

      event[:properties][:created_at] = report.created_at.to_i

      # Remove nil values (Spade fills not submitted ones itself (with nil))
      event[:properties] = event[:properties].delete_if {|k,v| v.blank? }

      # Add event to the ActiveJob queue (this way we don't block on this stats call)
      SpadeRequestJob.perform_later(event.to_json)
    end

    def retry_report(report_params)
      if !report_params[:try_counter] || report_params[:try_counter] < 10
        if report_params[:try_counter]
          report_params[:try_counter] = report_params[:try_counter] + 1
        else
          report_params[:try_counter] = 2
        end

        ReportRetryJob.set(wait: 5.seconds).perform_later(report_params.to_json)
      else
        logger.warn "Report failed after 10 tries: #{report_params.to_json}"
      end
    end

    def require_view_reports_permission
      unless current_user && current_user.has_permission_to?(:view_reports)
        render 'shared/_no_permission' and return
      end
    end

    def return_report_version_via_header
      redis_report_version = $redis.get("reports:report_version")
      response.headers["X-Report-Version"] = redis_report_version if redis_report_version
      return true
    end
end
