class SuspensionGuideDetailedReasonsController < ApplicationController
  # Let only people with manage_suspension_guide permission change stuff
  before_action :require_manage_suspension_guide, except: [:index, :show]

  def index
    @suspension_guide_detailed_reasons = SuspensionGuideDetailedReason.order(detailed_reason: :asc)
  end

  def show
    @suspension_guide_detailed_reason = SuspensionGuideDetailedReason.find_by(id: params[:id])
  end

  def new
    @suspension_guide_detailed_reason = SuspensionGuideDetailedReason.new
  end

  def edit
    @suspension_guide_detailed_reason = SuspensionGuideDetailedReason.find_by(id: params[:id])
  end

  def create
    suspension_guide_detailed_reason = SuspensionGuideDetailedReason.new(suspension_guide_detailed_reason_params)

    if suspension_guide_detailed_reason.save
      # All went well
      flash[:success] = "The suspension guide detailed reason was successfully added."

      # Audit it
      SuspensionGuideAudit.new(
        action: "new_detailed_reason",
        action_by: current_user.id,
        diff: suspension_guide_detailed_reason.attributes.except("created_at", "updated_at").to_json
      ).save

      # Delete our cached version
      $redis.del("SuspensionGuide:addon_json")
      $redis.del("SuspensionGuide:addon_quick_selects_json")
    else
      # Some error
      flash[:danger] = "There was an error adding the suspension guide detailed reason: #{suspension_guide_detailed_reason.errors.full_messages.to_sentence}"
    end

    redirect_to(suspension_guide_detailed_reasons_path)
  end

  def update
    suspension_guide_detailed_reason = SuspensionGuideDetailedReason.find_by(id: params[:id])
    suspension_guide_detailed_reason_before_change = suspension_guide_detailed_reason.dup

    if suspension_guide_detailed_reason.update_attributes(suspension_guide_detailed_reason_params)
      # All went well
      flash[:success] = "The suspension guide detailed reason was successfully updated."

      if suspension_guide_detailed_reason_before_change.diff(suspension_guide_detailed_reason).any?
        # Audit it
        SuspensionGuideAudit.new(
          action: "edit_detailed_reason",
          action_by: current_user.id,
          diff: { id: suspension_guide_detailed_reason.id }.merge(suspension_guide_detailed_reason_before_change.diff(suspension_guide_detailed_reason).to_hash).to_json
        ).save
      end

      # Delete our cached version
      $redis.del("SuspensionGuide:addon_json")
      $redis.del("SuspensionGuide:addon_quick_selects_json")
    else
      # Some error
      flash[:danger] = "There was an error updating the suspension guide detailed reason: #{suspension_guide_detailed_reason.errors.full_messages.to_sentence}"
    end

    redirect_to(edit_suspension_guide_detailed_reason_path(suspension_guide_detailed_reason))
  end

  def destroy
    suspension_guide_detailed_reason = SuspensionGuideDetailedReason.find_by(id: params[:id])
    suspension_guide_detailed_reason_before_change = suspension_guide_detailed_reason.dup

    # Check if the detailed reason is used in the Suspension Guide
    suspension_guide_usage = SuspensionGuide.where(detailed_reason_id: suspension_guide_detailed_reason.id)
    if suspension_guide_usage.any?
      # You can't delete a detailed reason if the Suspension Guide uses it
      flash[:danger] = "You can't delete this detailed reason because it's currently used by the suspension guide."
    else
      if suspension_guide_detailed_reason.destroy
        # All went well
        flash[:success] = "The suspension guide detailed reason was successfully removed."

        # Audit it
        SuspensionGuideAudit.new(
          action: "delete_detailed_reason",
          action_by: current_user.id,
          diff: { id: params[:id].to_i }.reverse_merge(suspension_guide_detailed_reason_before_change.attributes.except("created_at", "updated_at").to_hash).to_json
        ).save

        # Delete our cached version
        $redis.del("SuspensionGuide:addon_json")
        $redis.del("SuspensionGuide:addon_quick_selects_json")
      else
        # Some error
        flash[:danger] = "There was an error removing the suspension guide detailed reason: #{suspension_guide_detailed_reason.errors.full_messages.to_sentence}"
      end
    end

    redirect_to(suspension_guide_detailed_reasons_path)
  end


  private

    def suspension_guide_detailed_reason_params
      params.require(:suspension_guide_detailed_reason).permit(:detailed_reason)
    end

    def require_manage_suspension_guide
      unless current_user && current_user.has_permission_to?(:manage_suspension_guide)
        render 'shared/_no_permission' and return
      end
    end
end
