class SuspensionsController < ApplicationController
  # Disable login check for creating suspensions
  skip_before_action :require_login, only: [:create]
  # Disable CSRF token check for creating suspensions
  skip_before_action :verify_authenticity_token, only: [:create]
  # Let only people with view_reports permission do stuff
  before_action :require_view_reports_permission, except: [:create]
  # Let only people with counter_check_suspensions permission view suspension checks
  before_action :require_counter_check_suspensions_permission, only: [:checks]


  def index
    @suspensions = Suspension.order(id: :desc).paginate(page: params[:page])
  end

  def show
    @suspension = Suspension.find_by(id: params[:id])
    @suspension_comments = SuspensionComment.where(suspension_id: params[:id]).order(id: :asc)
  end

  def create
    unless params[:authorisation_token] || params[:authorization_token]
      render(json: {status: 403, error: true, error_message: "No authorization_token provided"}, :status => 403) and return
    end

    unless (params[:authorisation_token] && params[:authorisation_token] == ENV['CREATE_SUSPENSION_TOKEN']) || (params[:authorization_token] && params[:authorization_token] == ENV['CREATE_SUSPENSION_TOKEN'])
      render(json: {status: 403, error: true, error_message: "Invalid authorization_token provided"}, :status => 403) and return
    end

    if params[:suspension].blank?
      render(json: {status: 400, error: true, error_message: "No suspension content submitted"}, :status => 400) and return
    end

    suspension = Suspension.new(suspension_create_params)

    # Fill out the missing variables (like display name, type, partner status), id > username
    # From user
    # Test if it's an int or a LDAP login
    from_user_id_is_int = Integer(suspension.from_user_id) rescue false
    if suspension.from_user_id && from_user_id_is_int != false && from_user_id_is_int > 0
      begin
          from_user = ::TwitchUsersService.get_user_by_id(suspension.from_user_id)
        rescue Exception => e
          logger.debug "TwitchUsersService Exception (Thread: #{Thread.current.object_id}): #{e.message}"
          retry_suspension(params) if Rails.env.production?
          render(json: {status: 500, error: true, error_message: "We broke something over here"}, :status => 500) and return
      end

      suspension.from_user_type = from_user["type"]
    elsif suspension.from_user_id && !suspension.from_user_id.blank?
      # This is ldap, at least most likely

      suspension.from_user_type = "ldap"
    else
      # Reporter is empty
      render(json: {status: 422, error: true, error_message: "Invalid suspension content submitted. Missing striking user.", error_detailed: suspension.errors.full_messages}, :status => 422) and return
    end

    # Target user
    if suspension.target_user_id && suspension.target_user_id > 0
      begin
          target_user = ::TwitchUsersService.get_user_by_id(suspension.target_user_id)
        rescue Exception => e
          logger.debug "TwitchUsersService Exception (Thread: #{Thread.current.object_id}): #{e.message}"
          retry_suspension(params) if Rails.env.production?
          render(json: {status: 500, error: true, error_message: "We broke something over here"}, :status => 500) and return
      end

      suspension.target_user_username = target_user["username"]
      suspension.target_user_type = target_user["type"]
      suspension.target_user_display_name = target_user["display_name"]
    elsif suspension.target_user_username && !suspension.target_user_username.blank?
      begin
          target_user = ::TwitchUsersService.get_user_by_username(suspension.target_user_username)
        rescue Exception => e
          logger.debug "TwitchUsersService Exception (Thread: #{Thread.current.object_id}): #{e.message}"
          retry_suspension(params) if Rails.env.production?
          render(json: {status: 500, error: true, error_message: "We broke something over here"}, :status => 500) and return
      end

      suspension.target_user_id = target_user["id"]
      suspension.target_user_type = target_user["type"]
      suspension.target_user_display_name = target_user["display_name"]
    else
      # Target is empty
      render(json: {status: 422, error: true, error_message: "Invalid suspension content submitted. Missing target user.", error_detailed: suspension.errors.full_messages}, :status => 422) and return
    end

    # Get the partner info for the target (if type is filled out we know the user exists)
    if suspension.target_user_type && !suspension.target_user_type.blank?
      begin
          suspension.target_user_partner = ::TwitchPartnershipsService.get_partner_by_id(suspension.target_user_id)
        rescue Exception => e
          logger.debug "TwitchPartnershipsService Exception (Thread: #{Thread.current.object_id}): #{e.message}"
          retry_suspension(params) if Rails.env.production?
          render(json: {status: 500, error: true, error_message: "We broke something over here"}, :status => 500) and return
      end
    end

    # Get the currently set game on the target channel
    if suspension.target_user_type && !suspension.target_user_type.blank?
      begin
          target_channel = ::TwitchChannelsService.get_channel_by_username(suspension.target_user_username)
          suspension.target_user_game = target_channel["game"] unless target_channel.nil?
        rescue Exception => e
          logger.debug "TwitchChannelsService Exception (Thread: #{Thread.current.object_id}): #{e.message}"
          suspension.target_user_game = nil
      end
    end

    # Get current recording VOD
    if suspension.content == "live"
      begin
        currently_recorded_vod = ::TwitchVinylService.get_currently_recording_vod(suspension.target_user_id)
        suspension.current_live_vod = (currently_recorded_vod.split("?")[0]).to_i unless currently_recorded_vod.nil?
      rescue Exception => e
        logger.debug "TwitchVinylService Exception (Thread: #{Thread.current.object_id}): #{e.message}"
      end
    end

    # Make sure all new lines in the description are only \n and not \r\n or \r (just to have consistency)
    suspension.description = suspension.description.gsub(/\r\n/, "\n")
    suspension.description = suspension.description.gsub(/\r/, "\n")

    # Set the reason to other if it's currently invalid
    suspension.valid?
    suspension.reason = "other" unless (suspension.errors()[:reason]).empty?

    unless suspension.save
      # Error message about what went wrong
      render(json: {status: 422, error: true, error_message: "Invalid suspension content submitted", error_detailed: suspension.errors.full_messages}, :status => 422) and return
    end

    # Check if we want to add this suspension to the check list
    # Exclude: 41292798 => Kanthes, 20137474 => Tarfu (dschweitzer), 38400823 => Truthful, 30531236 => Enforcer
    admins_excluded_from_check = ["41292798", "20137474", "dschweitzer", "38400823", "30531236"]
    unless admins_excluded_from_check.include?(suspension.from_user_id)
      admin_checked = SuspensionCheck.where("suspension_from_user_id = ? AND created_at > ?", suspension.from_user_id, 30.minutes.ago).count
      if admin_checked == 0
        # Add a new check
        SuspensionCheck.new(
          suspension_id: suspension.id,
          suspension_from_user_id: suspension.from_user_id,
          status: "new"
        ).save
      end
    end

    # Send suspension to spade
    send_suspension_to_spade(suspension) if Rails.env.production?

    # Give the all is well response
    render(json: {status: 200, error: false, error_message: ""}, :status => 200) and return
  end

  def history
    @suspensions = Suspension.where(target_user_id: params[:target_user_id]).order(id: :desc)

    render layout: false
  end

  def search
    conditions = suspension_search_params.delete_if {|k,v| v.blank? }
    if conditions[:suspensions]
      conditions[:suspensions] = conditions[:suspensions].delete_if {|k,v| v.blank? }
    end

    if conditions[:suspensions] && conditions[:suspensions].any?
      if !conditions[:suspensions][:description].blank?
        searched_description = conditions[:suspensions][:description]

        # Remove description from the normal conditions because we want to do a LIKE search instead
        conditions[:suspensions].delete(:description)

        if conditions[:suspensions].any?
          @suspensions = Suspension.where(conditions).where("description LIKE ?", "%#{searched_description}%").order(id: :desc).paginate(page: params[:page])
        else
          # Description was the only parameter
          @suspensions = Suspension.where("description LIKE ?", "%#{searched_description}%").order(id: :desc).paginate(page: params[:page])
        end
      else
        @suspensions = Suspension.where(conditions).order(id: :desc).paginate(page: params[:page])
      end
    end
  end

  def checks
    if request.post?
      # Make sure the post has what we need
      unless params[:suspension_id]
        flash[:danger] = "No suspension id submitted"
        redirect_to(suspension_checks_path) and return
      end

      unless params[:suspension_status] && %w(good wrong).include?(params[:suspension_status])
        flash[:danger] = "No suspension status submitted"
        redirect_to(suspension_checks_path) and return
      end

      # Get suspension check
      db_suspension_check = SuspensionCheck.find_by(suspension_id: params[:suspension_id])
      unless db_suspension_check
        flash[:danger] = "This suspension doesn't exist"
        redirect_to(suspension_checks_path) and return
      end

      # Mark as good or wrong
      db_suspension_check.status = params[:suspension_status]
      db_suspension_check.checked_by = current_user.id

      # Update DB entry
      db_suspension_check.save

      flash[:success] = "Suspension check status successfully saved"
      redirect_to(suspension_checks_path) and return
    end

    # Get request things
    if params[:status]
      if params[:status] == "all"
        # Show all cases if status is set to "all"
        if params[:from_user_id]
          @suspension_checks = SuspensionCheck.where(suspension_from_user_id: params[:from_user_id]).order(id: :asc).paginate(page: params[:page])
        else
          @suspension_checks = SuspensionCheck.order(id: :asc).paginate(page: params[:page])
        end
      else
        if params[:from_user_id]
          @suspension_checks = SuspensionCheck.where(status: params[:status], suspension_from_user_id: params[:from_user_id]).order(id: :asc).paginate(page: params[:page])
        else
          @suspension_checks = SuspensionCheck.where(status: params[:status]).order(id: :asc).paginate(page: params[:page])
        end
      end
    else
      # Show new checks as default
      if params[:from_user_id]
        @suspension_checks = SuspensionCheck.where(status: "new", suspension_from_user_id: params[:from_user_id]).order(id: :asc).paginate(page: params[:page])
      else
        @suspension_checks = SuspensionCheck.where(status: "new").order(id: :asc).paginate(page: params[:page])
      end
    end

    # Get suspensions for the checks
    suspension_ids = []
    if @suspension_checks.any?
      @suspension_checks.each do |suspension_check|
        suspension_ids << suspension_check.suspension_id
      end
    end

    @suspensions = Suspension.where(id: suspension_ids)
  end


  private

    def suspension_create_params
      params.require(:suspension).permit(
        :from_user_id,

        :target_user_username,
        :target_user_id,

        :content,
        :reason,
        :detailed_reason,
        :description,

        :duration,
        :ip_ban,
        :cleared_channel_images,

        :origin
      )
    end

    def suspension_search_params
      params.permit(
        suspensions: [
          :id,

          :from_user_id,
          :from_user_type,

          :target_user_username,
          :target_user_display_name,
          :target_user_id,
          :target_user_type,
          :target_user_partner,
          :target_user_game,

          :content,
          :reason,
          :detailed_reason,
          :description,

          :duration,
          :ip_ban,
          :cleared_channel_images,

          :created_at
        ]
      ).to_h
    end

    def send_suspension_to_spade(suspension)
      # Generate the right array structure
      event = {}
      event[:event] = 'leviathan_suspension'
      event[:properties] = {}
      event[:properties][:suspension_id] = suspension.id

      event[:properties][:from_user_id] = suspension.from_user_id
      event[:properties][:from_user_type] = suspension.from_user_type

      event[:properties][:target_user_login] = suspension.target_user_username
      event[:properties][:target_user_display_name] = suspension.target_user_display_name
      event[:properties][:target_user_id] = suspension.target_user_id
      event[:properties][:target_user_type] = suspension.target_user_type
      event[:properties][:target_user_partner] = suspension.target_user_partner
      event[:properties][:target_user_game] = suspension.target_user_game

      event[:properties][:content] = suspension.content
      event[:properties][:reason] = suspension.reason
      event[:properties][:detailed_reason] = suspension.detailed_reason

      event[:properties][:account_ban_duration] = suspension.duration
      event[:properties][:ip_ban_duration] = suspension.ip_ban
      event[:properties][:cleared_channel_images] = suspension.cleared_channel_images

      event[:properties][:created_at] = suspension.created_at.to_i

      # Remove nil values (Spade fills not submitted ones itself (with nil))
      event[:properties] = event[:properties].delete_if {|k,v| v.blank? }

      # Add event to the ActiveJob queue (this way we don't block on this stats call)
      SpadeRequestJob.perform_later(event.to_json)
    end

    def retry_suspension(suspension_params)
      if !suspension_params[:try_counter] || suspension_params[:try_counter] < 10
        if suspension_params[:try_counter]
          suspension_params[:try_counter] = suspension_params[:try_counter] + 1
        else
          suspension_params[:try_counter] = 2
        end

        SuspensionRetryJob.set(wait: 5.seconds).perform_later(suspension_params.to_json)
      else
        logger.warn "Suspension failed after 10 tries: #{suspension_params.to_json}"
      end
    end

    def require_view_reports_permission
      unless current_user && current_user.has_permission_to?(:view_reports)
        render 'shared/_no_permission' and return
      end
    end

    def require_counter_check_suspensions_permission
      unless current_user && current_user.has_permission_to?(:counter_check_suspensions)
        render 'shared/_no_permission' and return
      end
    end
end
