require 'net/http'

class TwitchUsersService

  def self.get_user_by_id(id, no_cache = false)
    if id == nil
      return nil
    end

    # Make sure ID is an int
    id = id.to_i

    # Negative IDs to positive
    if id < 0
      id = id.abs
    end

    # Check if we have it already cached
    unless no_cache
      cache = $redis.get("TwitchUsersService:id:#{id}")
      if cache
        return JSON.parse(cache)
      end
    end

    # Do the api request
    url = URI.parse("http://users-service.prod.us-west2.twitch.tv/users/#{id}")
    http = Net::HTTP.new(url.host, url.port)
    http.use_ssl = true if url.scheme == "https"
    http.open_timeout = 1
    http.read_timeout = 3
    req = Net::HTTP::Get.new(url.request_uri)
    res = http.request(req)

    unless res.is_a?(Net::HTTPSuccess)
      raise "Twitch API error"
    end

    res_hash = JSON.parse(res.body)
    if res_hash["id"] == id
      # Gen hash with only the info we need
      user_hash = {}
      user_hash["id"] = res_hash["id"]
      user_hash["username"] = res_hash["login"]
      user_hash["display_name"] = res_hash["displayname"]
      if res_hash["admin"] == true
        user_hash["type"] = "staff"
      elsif res_hash["subadmin"] == true
        user_hash["type"] = "admin"
      elsif res_hash["global_mod"] == true
        user_hash["type"] = "global_mod"
      else
        user_hash["type"] = "user"
      end

      unless no_cache
        cache_id_username_user_json(user_hash["id"], user_hash)
      end

      return user_hash
    end

    raise "ID not found"
  end

  def self.get_user_by_username(username, no_cache = false)
    if username.blank?
      return nil
    end

    # Case sensitivity doesn't matter for Twitch so we downcase it
    username = username.downcase.strip

    # Check if we have it already cached
    unless no_cache
      cache = $redis.get("TwitchUsersService:username:#{username}")
      if cache
        return JSON.parse(cache)
      end
    end

    # Do the api request
    url = URI.parse("http://users-service.prod.us-west2.twitch.tv/users?login=#{URI::escape(username)}")
    http = Net::HTTP.new(url.host, url.port)
    http.use_ssl = true if url.scheme == "https"
    http.open_timeout = 1
    http.read_timeout = 3
    req = Net::HTTP::Get.new(url.request_uri)
    res = http.request(req)

    unless res.is_a?(Net::HTTPSuccess)
      raise "Twitch API error"
    end

    res_hash = JSON.parse(res.body)
    if !res_hash["results"].empty? && res_hash["results"][0]["login"] == username
      # Gen hash with only the info we need
      user_hash = {}
      user_hash["id"] = res_hash["results"][0]["id"]
      user_hash["username"] = res_hash["results"][0]["login"]
      user_hash["display_name"] = res_hash["results"][0]["displayname"]
      if res_hash["results"][0]["admin"] == true
        user_hash["type"] = "staff"
      elsif res_hash["results"][0]["subadmin"] == true
        user_hash["type"] = "admin"
      elsif res_hash["results"][0]["global_mod"] == true
        user_hash["type"] = "global_mod"
      else
        user_hash["type"] = "user"
      end

      unless no_cache
        cache_id_username_user_json(user_hash["id"], user_hash)
      end

      return user_hash
    else
      raise "Username not found"
    end
  end

  def self.cache_id_username_user_json(id, user_hash)
    if id.blank? || user_hash["username"].blank?
      return
    end

    # Set ID => Username
    $redis.setex("TwitchUsersService:id:#{id}", 5.minutes, user_hash.to_json)

    # Set Username => ID
    $redis.setex("TwitchUsersService:username:#{user_hash['username']}", 5.minutes, user_hash.to_json)
  end
end
