require 'active_record/diff'

class User < ActiveRecord::Base
  include ActiveRecord::Diff
  diff :exclude => [:created_at, :updated_at]

  before_save { self.email = email.downcase }
  has_one :permission, dependent: :destroy, autosave: true
  validates :email, presence: true, format: { with: /\A.*@(twitch\.tv|justin\.tv|mods\.twitch\.tv)\z/i }, uniqueness: { case_sensitive: false }, on: :create
  validates :twitch_id, numericality: { only_integer: true }, allow_nil: true
  validates :report_mode, presence: true, inclusion: { in: %w( normal fade ) }
  validates :style, presence: true, inclusion: { in: %w( white dark ) }


  ## OAuth login
  def self.from_omniauth(auth)
    logger.debug "Auth: #{auth.to_json}" if Rails.env.development?
    if auth.extra.id_info.email_verified == true
      user = User.find_by(email: auth.info.email.downcase)
      unless user
        # Create new User / account
        user = User.new(email: auth.info.email.downcase)
        if user.save
          # Create user default permissions
          user.build_permission
          if user.save
            # Audit it
            UserAudit.new(
              user_id: user.id,
              action: "create",
              action_by: user.id
            ).save
          else
            user = nil
          end
        else
          user = nil
        end
      end
    else
      user = nil
    end
    return user
  end

  # Check if the user has permissions to do this action
  def has_permission_to?(action)
    permissions = self.permission
    if permissions[action] == true
      return true
    end
    return false
  end
end