package usersservice

import (
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"

	"golang.org/x/net/context"

	"code.justin.tv/common/twitchhttp"
)

const (
	defaultStatSampleRate = 1.0

	// batchGetLimit sets the max amount of users that can be queried.
	batchGetLimit = 50
)

// NewClient creates a client for the users service.
func NewClient(conf twitchhttp.ClientConf) (Client, error) {
	if conf.TimingXactName == "" {
		conf.TimingXactName = "users_service"
	}
	twitchClient, err := twitchhttp.NewClient(conf)
	if err != nil {
		return nil, err
	}
	return &clientImpl{twitchClient}, nil
}

type clientImpl struct {
	twitchhttp.Client
}

// BatchGetPropertiesFromID fetches properties for multiple users given a list of IDs.
func (c *clientImpl) BatchGetPropertiesFromID(ctx context.Context, ids []string, properties []Property, reqOpts *twitchhttp.ReqOpts) (resp *UsersResponse, err error) {
	values, err := buildValuesFromLoginsAndIDs(nil, ids)
	if err != nil {
		return nil, err
	}
	for _, prop := range properties {
		values.Add("prop", string(prop))
	}

	path := (&url.URL{
		Path:     "/users",
		RawQuery: values.Encode(),
	}).String()
	req, err := c.NewRequest("GET", path, nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "service.users_service.batch_get_properties_from_id",
		StatSampleRate: defaultStatSampleRate,
	})
	httpResp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}
	defer func() {
		if cerr := httpResp.Body.Close(); cerr != nil && err == nil {
			err = cerr
		}
	}()

	if httpResp.StatusCode != http.StatusOK {
		return nil, fmt.Errorf("Unexpected status code %d", httpResp.StatusCode)
	}

	var decoded UsersResponse
	if err := json.NewDecoder(httpResp.Body).Decode(&decoded); err != nil {
		return nil, err
	}
	return &decoded, nil
}

func buildValuesFromLoginsAndIDs(logins, ids []string) (url.Values, error) {
	total := len(logins) + len(ids)
	if total == 0 || total > batchGetLimit {
		return nil, ErrBatchGetLimitExceeded
	}
	return url.Values{
		"login": logins,
		"id":    ids,
	}, nil
}
