package usermutation

import (
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
)

// CheckpointStore is used to persist and read a shard's checkpoint
type CheckpointStore interface {
	// GetCheckpoint returns a shard's currently persisted checkpoint
	GetCheckpoint(shardID string) (*string, error)
	// SaveCheckpoint updates a shard's checkpoint
	SaveCheckpoint(shardID string, checkpoint string) error
}

type dynamoDBCheckpointStoreImpl struct {
	dynamoClient    *dynamodb.DynamoDB
	dynamoTableName string
	environment     string
}

// NewDynamoDBCheckpointStore allocates and returns a DynamoDB based CheckpointStore
func NewDynamoDBCheckpointStore(dynamoDBClient *dynamodb.DynamoDB, dynamoDBTableName string, environment string) CheckpointStore {
	return &dynamoDBCheckpointStoreImpl{
		dynamoClient:    dynamoDBClient,
		dynamoTableName: dynamoDBTableName,
		environment:     environment,
	}
}

func (d *dynamoDBCheckpointStoreImpl) GetCheckpoint(shardID string) (*string, error) {
	params := &dynamodb.GetItemInput{
		TableName: aws.String(d.dynamoTableName),
		Key: map[string]*dynamodb.AttributeValue{
			"ShardId": {
				S: aws.String(shardID),
			},
			"Environment": {
				S: aws.String(d.environment),
			},
		},
	}

	resp, err := d.dynamoClient.GetItem(params)
	if err != nil {
		return nil, err
	}

	if resp.Item == nil {
		return nil, nil // checkpoint doesn't exist yet, not an error
	}
	checkpoint := resp.Item["Checkpoint"].S
	return checkpoint, nil
}

func (d *dynamoDBCheckpointStoreImpl) SaveCheckpoint(shardID string, checkpoint string) error {
	params := &dynamodb.PutItemInput{
		TableName: aws.String(d.dynamoTableName),
		Item: map[string]*dynamodb.AttributeValue{
			"ShardId": {
				S: aws.String(shardID),
			},
			"Environment": {
				S: aws.String(d.environment),
			},
			"Checkpoint": {
				S: aws.String(checkpoint),
			},
		},
	}

	_, err := d.dynamoClient.PutItem(params)

	return err
}
