package usermutation

import (
	"encoding/json"
	"fmt"
	"log"
	"math/rand"
	"os"
	"sync"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/kinesis"
)

const (
	maxTickerTime   = 10 * time.Second
	minTickerTime   = 1 * time.Second
	maxMillisBehind = 300000
	maxGetRecords   = 5000
)

type worker struct {
	checkpointStore CheckpointStore
	kinesisClient   KinesisClient
	shardID         string
	callback        ConsumerFunc
	logger          *log.Logger
	stopped         bool
	stoppedSignal   *sync.WaitGroup
}

func NewWorker(kinesisClient KinesisClient, checkpointStore CheckpointStore, shardID string, callback ConsumerFunc) *worker {
	return &worker{
		kinesisClient:   kinesisClient,
		checkpointStore: checkpointStore,
		shardID:         shardID,
		callback:        callback,
		logger:          log.New(os.Stderr, fmt.Sprintf("worker-%s: ", shardID), log.LstdFlags),
	}
}

func (w *worker) Run() error {
	shardIteratorResp, err := w.getShardIterator(aws.String(w.shardID))
	if err != nil {
		return err
	}

	// Add some random jitter
	time.Sleep(time.Duration(float64(maxTickerTime) * rand.Float64()))

	w.logger.Printf("Worker starting for shard: %s", w.shardID)

	shardIterator := shardIteratorResp.ShardIterator

	var lastShardTick *time.Time
	currentTickerTime := maxTickerTime

	for {
		if !w.stopped {
			if lastShardTick != nil {
				lastLoopTime := time.Since(*lastShardTick)
				sleepTime := currentTickerTime - lastLoopTime
				if sleepTime > 0 {
					time.Sleep(sleepTime)
				}
			}
			now := time.Now()
			lastShardTick = &now
		}
		if w.stopped {
			w.logger.Printf("Worker shutdown requested, terminating loop")
			w.stoppedSignal.Done()
			return nil
		}

		params := &kinesis.GetRecordsInput{
			ShardIterator: aws.String(*shardIterator),
			Limit:         aws.Int64(maxGetRecords),
		}
		recordsOutput, err := w.kinesisClient.GetRecords(params)
		if err != nil {
			panic(err)
		}

		records := recordsOutput.Records

		w.logger.Printf("Processing %d records", len(records))
		if len(records) > 0 {
			if recordsOutput.MillisBehindLatest != nil {
				w.logger.Printf("Currently %d milliseconds behind latest record", *recordsOutput.MillisBehindLatest)

				if *recordsOutput.MillisBehindLatest > maxMillisBehind {
					if currentTickerTime > minTickerTime {
						currentTickerTime = currentTickerTime / 2
					}
				} else if currentTickerTime < maxTickerTime {
					currentTickerTime = currentTickerTime * 2
				}
			}

			var checkpoint *string
			for _, record := range recordsOutput.Records {
				var msg Message
				err := json.Unmarshal(record.Data, &msg)
				if err != nil {
					if checkpoint != nil {
						saveErr := w.checkpointStore.SaveCheckpoint(w.shardID, *checkpoint)
						if saveErr != nil {
							w.logger.Printf("Error saving checkpoint: %s", saveErr.Error())
						}
					}
					return err
				}
				err = w.callback(msg)
				if err != nil {
					if checkpoint != nil {
						saveErr := w.checkpointStore.SaveCheckpoint(w.shardID, *checkpoint)
						if saveErr != nil {
							w.logger.Printf("Error saving checkpoint: %s", saveErr.Error())
						}
					}
					return err
				}
				checkpoint = record.SequenceNumber
			}

			err = w.checkpointStore.SaveCheckpoint(w.shardID, *checkpoint)
			if err != nil {
				w.logger.Printf("Error saving checkpoint: %s", err.Error())
				return err
			}
		} else if recordsOutput.NextShardIterator == nil {
			w.logger.Printf("Shard has been closed, terminating loop")
			return nil
		} else if *shardIterator == *recordsOutput.NextShardIterator {
			w.logger.Printf("Last record processed, terminating loop")
			return nil
		}

		shardIterator = recordsOutput.NextShardIterator
	}
}

func (w *worker) getShardIterator(shardID *string) (*kinesis.GetShardIteratorOutput, error) {
	startingSequenceNumber, err := w.checkpointStore.GetCheckpoint(w.shardID)
	if err != nil {
		return nil, err
	}

	params := &kinesis.GetShardIteratorInput{
		ShardId:    shardID,
		StreamName: aws.String(kinesisStream),
	}

	if startingSequenceNumber != nil {
		params.ShardIteratorType = aws.String(kinesis.ShardIteratorTypeAfterSequenceNumber)
		params.StartingSequenceNumber = startingSequenceNumber
		w.logger.Printf("Getting ShardIterator for %s from saved checkpoint: %s", *shardID, *startingSequenceNumber)
	} else {
		params.ShardIteratorType = aws.String(kinesis.ShardIteratorTypeTrimHorizon)
		w.logger.Printf("Getting ShardIterator for %s from last untrimmed record", *shardID)
	}

	output, err := w.kinesisClient.GetShardIterator(params)
	if err != nil {
		return nil, err
	}
	return output, nil
}

func (w *worker) Stop() {
	w.logger.Printf("Worker shutdown request registered")
	w.stoppedSignal = &sync.WaitGroup{}
	w.stoppedSignal.Add(1)
	w.stopped = true
	w.stoppedSignal.Wait()
}
